/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2007 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import net.fckeditor.tool.Utils;

/**
 * Static object which manages the allowed and denied extensions for each file type. The different extensions are preset
 * by the defaults defined in default.properties.
 * 
 * @version $Id: ExtensionsHandler.java 1347 2008-01-17 11:20:47Z th-schwarz $
 */
public class ExtensionsHandler {

    private static Map<String, Set<String>> extensionsAllowed = new HashMap<String, Set<String>>();
    private static Map<String, Set<String>> extensionsDenied = new HashMap<String, Set<String>>();

    static {
	// load defaults 
	extensionsAllowed.put(Constants.FILE_TYPE_FILE, 
		Utils.getSet(ConfigurationHandler.getDefaultProperty("fckeditor.file.extensions.allowed")));
	extensionsDenied.put(Constants.FILE_TYPE_FILE, 
		Utils.getSet(ConfigurationHandler.getDefaultProperty("fckeditor.file.extensions.denied")));
	extensionsAllowed.put(Constants.FILE_TYPE_MEDIA, 
		Utils.getSet(ConfigurationHandler.getDefaultProperty("fckeditor.media.extensions.allowed")));
	extensionsDenied.put(Constants.FILE_TYPE_MEDIA, 
		Utils.getSet(ConfigurationHandler.getDefaultProperty("fckeditor.media.extensions.denied")));
	extensionsAllowed.put(Constants.FILE_TYPE_IMAGE, 
		Utils.getSet(ConfigurationHandler.getDefaultProperty("fckeditor.image.extensions.allowed")));
	extensionsDenied.put(Constants.FILE_TYPE_IMAGE, 
		Utils.getSet(ConfigurationHandler.getDefaultProperty("fckeditor.image.extensions.denied")));
	extensionsAllowed.put(Constants.FILE_TYPE_FLASH, 
		Utils.getSet(ConfigurationHandler.getDefaultProperty("fckeditor.flash.extensions.allowed")));
	extensionsDenied.put(Constants.FILE_TYPE_FLASH, 
		Utils.getSet(ConfigurationHandler.getDefaultProperty("fckeditor.flash.extensions.denied")));
    }

    /**
     * Getter for the allowed extensions of a file type.
     * 
     * @param type
     *                The file type.
     * @return Set of allowed extensions or an empty set.
     */
    public static Set<String> getExtensionsAllowed(final String type) {
	return ExtensionsHandler.extensionsAllowed.get(type);
    }

    /**
     * Setter for the allowed extensions of a file type.<br>
     * If 'extensionsList' is null, allowed extensions kept untouched.
     * 
     * @param type
     *                The file type.
     * @param extensionsList
     *                Required format: <code>ext1&#124;ext2&#124;ext3</code>
     */
    public static void setExtensionsAllowed(final String type, final String extensionsList) {
	if (extensionsList != null)
	    ExtensionsHandler.extensionsAllowed.put(type, Utils.getSet(extensionsList));
    }

    /**
     * Getter for the denied extensions of a file type.
     * 
     * @param type
     *                The file type.
     * @return Set of denied extensions or an empty set.
     */
    public static Set<String> getExtensionsDenied(final String type) {
	return ExtensionsHandler.extensionsDenied.get(type);
    }

    /**
     * Setter for the denied extensions of a file type.<br>
     * If 'extensionsList' is null, denied extensions kept untouched.
     * 
     * @param type
     *                The file type.
     * @param extensionsList
     *                Required format: <code>ext1&#124;ext2&#124;ext3</code>
     */
    public static void setExtensionsDenied(final String type, final String extensionsList) {
	if (extensionsList != null)
	    ExtensionsHandler.extensionsDenied.put(type, Utils.getSet(extensionsList));
    }

    /**
     * Checks, if an extension is allowed for a file type.
     * 
     * @param type
     * @param extension
     * @return True, false.
     */
    public static boolean isAllowed(final String type, final String extension) {
	String ext = extension.toLowerCase();
	Set<String> allowed = extensionsAllowed.get(type);
	Set<String> denied = extensionsDenied.get(type);
	if (allowed.isEmpty())
	    return (denied.contains(ext)) ? false : true;
	if (denied.isEmpty())
	    return (allowed.contains(ext)) ? true : false;
	return false;
    }
}
