<?php
//Script for automatic line-ending corrections
//Requires PHP5 to run
//http://www.gophp5.org/ ;)

error_reporting(E_ALL | E_STRICT);

/**
 * Strip whitespace from the end of file
 *
 */
define('EOF_STRIP_WHITESPACE', true);
/**
 * Force new line character at the end of file
 *
 */
define('EOF_NEW_LINE', true);

/**
 * Carriage return
 *
 */
define('CR', "\r");
/**
 * Line feed
 *
 */
define('LF', "\n");
/**
 * Carriage return + Line feed
 *
 */
define('CRLF', "\r\n");

/**
 * Array, where:
 *  file extension is the key
 *  value is the new line character
 */
$list = array();
$list["readme"] = CRLF;
$list["afp"] = CRLF;
$list["afpa"] = CRLF;
$list["ascx"] = CRLF;
$list["asp"] = CRLF;
$list["aspx"] = CRLF;
$list["bat"] = CRLF;
$list["cfc"] = CRLF;
$list["cfm"] = CRLF;
$list["cgi"] = LF;
$list["code"] = CRLF;
$list["command"] = CRLF;
$list["conf"] = CRLF;
$list["css"] = CRLF;
$list["dtd"] = CRLF;
$list["htaccess"] = CRLF;
$list["htc"] = CRLF;
$list["htm"] = CRLF;
$list["html"] = CRLF;
$list["js"] = CRLF;
$list["jsp"] = CRLF;
$list["lasso"] = CRLF;
$list["php"] = CRLF;
$list["pl"] = LF;
$list["py"] = CRLF;
$list["sample"] = CRLF;
$list["sh"] = CRLF;
$list["txt"] = CRLF;
$list["xml"] = CRLF;

/**
 * Filter file list using regular expression (negative result)
 *
 */
class NegRegexFilter extends FilterIterator
{
    protected $regex;
    public function __construct(Iterator $it, $regex)
    {
        parent::__construct($it);
        $this->regex=$regex;
    }
    public function accept()
    {
        return !preg_match($this->regex, $this->current());
    }

}

/**
 * Filter file list using regular expression
 *
 */
class RegexFilter extends FilterIterator
{
    protected $regex;
    public function __construct(Iterator $it, $regex)
    {
        parent::__construct($it);
        $this->regex=$regex;
    }
    public function accept()
    {
        return preg_match($this->regex, $this->current());
    }

}

/**
 * Fix new line characters in given file
 * Returns true if file was changed
 *
 * @param string $path relative or absolute path name to file
 * @param string $nl name of a constant that holds new line character (CRLF|CR|LF)
 * @return bool 
 */
function fixFile($path, $nl) {

    $contents = file($path);
    if ($contents === false) {
        echo "\rERROR: couldn't read the " . $path . " file". "\n";
        return false;
    }

    $modified = false;

    if ($contents === false) {
        echo "\rERROR: couldn't open the " . $path . " file" . "\n";
        return false;
    }

    $new_content = "";
    $contents_len = sizeof($contents);

    if (EOF_STRIP_WHITESPACE) {
        $lines_processed=0;
        //iterate through lines, from the end of file
        for ($i=$contents_len-1; $i>=0; $i--) {
            $old_line = $contents[$i];
            $contents[$i] = rtrim($contents[$i]);
            if ($old_line !== $contents[$i]) {
                if (!EOF_NEW_LINE || $old_line !== $contents[$i] . constant($nl) || $lines_processed>0) {
                    $modified = true;
                }
            }

            if (empty($contents[$i])) {
                //we have an empty line at the end of file, just skip it
                unset($contents[$contents_len--]);
            }
            else {
                if (EOF_NEW_LINE) {
                    $contents[$i] .= constant($nl);
                    if ($old_line !== $contents[$i]) {
                        $modified = true;
                    }
                }
                //we have found non-empty line, there is no need to go further
                break;
            }
            $lines_processed++;
        }
    }

    for ($i=0; $i<$contents_len; $i++) {
        $is_last_line = ($i == $contents_len-1);
        $line = $contents[$i];

        switch ($nl)
        {
            case 'CRLF':
                if (substr($line, -2) !== CRLF) {
                    if (substr($line, -1) === LF || substr($line, -1) === CR) {
                        $line = substr($line, 0, -1) . CRLF;
                        $modified = true;
                    }
                    elseif(strlen($line)) {
                        if (!$is_last_line) {
                            echo "\rERROR: wrong line ending: " . $path . "@line " . ($i+1) . "\n";
                            return false;
                        }
                        elseif(!EOF_STRIP_WHITESPACE) {
                            $line = $line . CRLF;
                            $modified = true;
                        }
                    }
                }
                break;

            case 'CR':
                if (substr($line, -1) !== CR) {
                    if (substr($line, -1) === LF) {
                        $line = substr($line, 0, -1) . CR;
                        $modified = true;
                    }
                    elseif(strlen($line)) {
                        if (!$is_last_line) {
                            echo "\rERROR: wrong line ending: " . $path . "@line " . ($i+1) . "\n";
                            return false;
                        }
                        elseif(!EOF_STRIP_WHITESPACE) {
                            $line = $line . CR;
                            $modified = true;
                        }
                    }
                }
                break;

            case 'LF':
                if(substr($line, -2) === CRLF) {
                    $line = substr($line, 0, -2) . LF;
                    $modified = true;
                }
                elseif (substr($line, -1) !== LF) {
                    if (substr($line, -1) === CR) {
                        $line = substr($line, 0, -1) . LF;
                        $modified = true;
                    }
                    elseif(strlen($line)) {
                        if (!$is_last_line) {
                            echo "\rERROR: wrong line ending: " . $path . "@line " . ($i+1) . "\n";
                            return false;
                        }
                        elseif(!EOF_STRIP_WHITESPACE) {
                            $line = $line . LF;
                            $modified = true;
                        }
                    }
                }
                break;
        }
        $new_content .= $line;
    }

    if ($modified) {
        $fp = fopen($path, "wb");
        if (!$fp) {
            echo "\rERROR: couldn't open the " . $path . " file". "\n";
            return false;
        }
        else {
            if (flock($fp, LOCK_EX)) {
                fwrite($fp, $new_content);
                flock($fp, LOCK_UN);
                echo "\rMODIFIED to " . $nl . ": " . $path . "\n";
            } else {
                echo "\rERROR: couldn't lock the " . $path . " file". "\n";
                return false;
            }
            fclose($fp);
        }
    }

    return $modified;
}

/**
 * Fix ending lines in all files at given path
 *
 * @param string $path
 */
function fixPath($path)
{
    if (is_file($path)) {
        $ext = strtolower(substr($path, strrpos($path, ".")));
        foreach (array('CRLF', 'LF', 'CR') as $nl) {
            //find out what's the correct line ending and fix file
            //no need to process further
            if (in_array($ext, $GLOBALS['extList'][$nl])) {
                echo "Fixing single file:\n";
                fixFile($path, $nl);
                break;
            }
        }

    }
    else {
        $dir = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($path), true);
        $dir = new NegRegexFilter($dir, "/\/(\.svn|CVS)/");
        foreach (array('CRLF', 'LF', 'CR') as $nl) {

            $filtered_dir = new RegexFilter($dir, "/\.(".implode("|", $GLOBALS['extList'][$nl]).")$/i");

            $extensions = array();
            $j = 0;
            $progressbar = "|/-\\";
            foreach ($filtered_dir as $file) {
                if (!is_dir($file)) {
                    fixFile($file, $nl);
                    print "\r ".$progressbar[$j++ % 4]. " ". str_pad(basename($file), 35, " ", STR_PAD_RIGHT);
                }
            }
        }
    }
}

/**
 * Get input from console
 *
 * @param string $prompt
 * @param array $options
 * @param string $default
 * @return string
 */
function getInput($prompt, $options = null, $default = null) {
    $stdin = fopen('php://stdin', 'r');
    if (!is_array($options)) {
        $print_options = '';
    } else {
        $print_options = '(' . implode('/', $options) . ')';
    }

    if ($default == null) {
        echo $prompt . " $print_options \n" . '> ';
    } else {
        echo $prompt . " $print_options \n" . "[$default] > ";
    }
    $result = fgets($stdin);

    if ($result === false){
        exit;
    }
    $result = trim($result);

    if ($default != null && empty($result)) {
        return $default;
    } else {
        return $result;
    }
}

/**
 * Validate path entered by user
 *
 * @param string $result
 * @return string
 */
function validatePath($result) {
    if (empty($result)) {
        die();
    }
    if (!is_dir($result) && !is_file($_SERVER['argv'][1])) {
        //let's process single file is valid path is given
        echo "Invalid directory: " . $result ."\n";
        validatePath(getInput("Enter valid path:"));
    }
    return $result;
}

//$extList holds the associative array of extensions
//key is the name of a constant that holds the line character
$extList = array();
$extList['CRLF'] = $extList['CR'] = $extList['LF'] = array();

foreach ($list as $ext => $nl) {
    $extRegex = preg_quote($ext);
    switch ($nl) {
        case CRLF:
            $extList['CRLF'][] = $extRegex;
            break;
        case LF:
            $extList['LF'][] = $extRegex;
            break;
        case CR:
            $extList['CR'][] = $extRegex;
            break;
        default:
            die("Unknown line ending");
            break;
    }
}

$path = validatePath(getInput("Enter path to the directory containing files that should be fixed. \nPath to single file is also allowed."));
fixPath($path);