/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.connector;

import static net.fckeditor.tool.Utils.forceSingleExtension;
import static net.fckeditor.tool.Utils.isEmpty;
import static net.fckeditor.tool.Utils.isValidPath;
import static net.fckeditor.tool.Utils.replaceAll;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.List;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import net.fckeditor.ResourceType;
import net.fckeditor.handlers.ConfigurationHandler;
import net.fckeditor.handlers.ExtensionsHandler;
import net.fckeditor.tool.UploadResponse;
import net.fckeditor.tool.Utils;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.commons.io.filefilter.FileFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * Servlet to upload and browse files.<br>
 * 
 * This servlet accepts 4 commands used to retrieve and create files and folders
 * from a server directory. The allowed commands are:
 * <ul>
 * <li>GetFolders: Retrive the list of directory under the current folder
 * <li>GetFoldersAndFiles: Retrive the list of files and directory under the
 * current folder
 * <li>CreateFolder: Create a new directory under the current folder
 * <li>FileUpload: Send a new file to the server (must be sent with a POST)
 * </ul>
 * 
 * @version $Id: ConnectorServlet.java 1537 2008-02-17 13:09:35Z mosipov $
 */
public class ConnectorServlet extends HttpServlet {

	private static final long serialVersionUID = -5742008970929377161L;
	private static final Logger logger = LoggerFactory
			.getLogger(ConnectorServlet.class);

	/**
	 * Initialize the servlet.<br>
	 * The init parameters will be read and used to adjust the defaults of the
	 * {@link ConfigurationHandler} and the {@link ExtensionsHandler}.
	 */
	public void init() throws ServletException, IllegalArgumentException {
		// read the required parameters
		ConfigurationHandler.constructSessionDataObject(getInitParameter("SessionDataImpl"));

		// read the optional parameters
		ConfigurationHandler.setForceSingleExtension(getInitParameter("ForceSingleExtension"));
		if (getInitParameter("UserFilesPath") != null) {
			if (!Utils.isValidPath(getInitParameter("UserFilesPath"))) {
				logger
						.warn("UserFilesPath is an illegal reference, using default path");
				ConfigurationHandler.setUserFilesPath(ConfigurationHandler
						.getDefaultUserFilesPath());
			}
		} else
			ConfigurationHandler
					.setUserFilesPath(getInitParameter("UserFilesPath"));

		setExtension(ResourceType.FILE, getInitParameter("AllowedExtensionsFile"),
		        getInitParameter("DeniedExtensionsFile"));
		setExtension(ResourceType.IMAGE, getInitParameter("AllowedExtensionsImage"),
		        getInitParameter("DeniedExtensionsImage"));
		setExtension(ResourceType.FLASH, getInitParameter("AllowedExtensionsFlash"),
		        getInitParameter("DeniedExtensionsFlash"));
		setExtension(ResourceType.MEDIA, getInitParameter("AllowedExtensionsMedia"),
		        getInitParameter("DeniedExtensionsMedia"));

		// check, if 'baseDir' exists
		String realBaseDir = getServletContext().getRealPath(
		        ConfigurationHandler.getDefaultUserFilesPath());
		File baseFile = new File(realBaseDir);
		if (!baseFile.exists()) {
			baseFile.mkdirs();
		}

		logger.info("Init parameters successfully set!");
	}

	/**
	 * Manage the Get requests (GetFolders, GetFoldersAndFiles, CreateFolder).<br>
	 * 
	 * The servlet accepts commands sent in the following format:<br>
	 * connector?Command=CommandName&Type=ResourceType&CurrentFolder=FolderPath<br>
	 * <br>
	 * It executes the command and then return the results to the client in XML
	 * format.
	 * 
	 */
	public void doGet(HttpServletRequest request, HttpServletResponse response)
			throws ServletException, IOException {
		logger.debug("Entering ConnectorServlet#doGet");

		response.setCharacterEncoding("UTF-8");
		response.setContentType("application/xml; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();

		String commandStr = request.getParameter("Command");
		String typeStr = request.getParameter("Type");
		String currentFolderStr = request.getParameter("CurrentFolder");
		ResourceType fileType = ResourceType.getDefaultResourceType(typeStr);

		logger.debug("Parameter Command: {}", commandStr);
		logger.debug("Parameter CurrentFolder: {}", currentFolderStr);
		logger.debug("Parameter Type: {}", typeStr);

		String currentPath = constructTypeBasedFolderString(fileType,
				currentFolderStr, request);
		String currentDirPath = getServletContext().getRealPath(currentPath);

		File currentDir = new File(currentDirPath);
		if (!currentDir.exists()
				&& ConfigurationHandler.isEnabledForFileBrowsing(request)) {
			currentDir.mkdirs();
			logger.debug("Dir successfully created: {}", currentDirPath);
		}

		Document document = null;
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			document = builder.newDocument();
		} catch (ParserConfigurationException pce) {
			logger.error("Error while building new XML document", pce);
			// FIXME servlet cannot go on working if building fails
		}

		Node root = createCommonXml(document, commandStr, typeStr,
				currentFolderStr, request.getContextPath().concat(currentPath));

		if (!ConfigurationHandler.isEnabledForFileBrowsing(request)) {
			createResponse("1",
					"The current user isn't authorized for file browsing!",
					root, document);
		} else if (commandStr.equals("GetFolders")) {
			getFolders(currentDir, root, document);
		} else if (commandStr.equals("GetFoldersAndFiles")) {
			getFolders(currentDir, root, document);
			getFiles(currentDir, root, document);
		} else if (commandStr.equals("CreateFolder")) {
			String newFolderStr = request.getParameter("NewFolderName");
			File newFolder = new File(currentDir, newFolderStr);
			String retValue = "110";

			if (newFolder.exists()) {
				retValue = "101";
			} else {
				try {
					boolean dirCreated = newFolder.mkdir();
					if (dirCreated)
						retValue = "0";
					else
						retValue = "102";
				} catch (SecurityException sex) {
					retValue = "103";
				}

			}
			createFolderResponse(retValue, root, document);
		}

		document.getDocumentElement().normalize();
		try {
			TransformerFactory tFactory = TransformerFactory.newInstance();
			Transformer transformer = tFactory.newTransformer();

			DOMSource source = new DOMSource(document);
			StreamResult result = new StreamResult(out);
			transformer.transform(source, result);

		} catch (Exception e) {
			// FIXME servlet cannot go on working if transforming fails
			logger.error("Error while transforming DOM to HttpServletResponse",
					e);
		}

		out.flush();
		out.close();
		logger.debug("Exiting ConnectorServlet#doGet");
	}

	/**
	 * Manage the Post requests (FileUpload).<br>
	 * 
	 * The servlet accepts commands sent in the following format:<br>
	 * connector?Command=FileUpload&Type=ResourceType&CurrentFolder=FolderPath<br>
	 * <br>
	 * It store the file (renaming it in case a file with the same name exists)
	 * and then return an HTML file with a javascript command in it.
	 * 
	 */
	public void doPost(HttpServletRequest request, HttpServletResponse response)
			throws ServletException, IOException {
		logger.debug("Entering Connector#doPost");

		response.setCharacterEncoding("UTF-8");
		response.setContentType("text/html; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();

		String commandStr = request.getParameter("Command");
		String typeStr = request.getParameter("Type");
		String currentFolderStr = request.getParameter("CurrentFolder");

		logger.debug("Parameter Command: {}", commandStr);
		logger.debug("Parameter CurrentFolder: {}", currentFolderStr);
		logger.debug("Parameter Type: {}", typeStr);

		if (!ResourceType.isValid(typeStr))
			logger.warn("Unknown Type requested: {}", typeStr);

		if (isEmpty(commandStr) && isEmpty(currentFolderStr)) {
			commandStr = "QuickUpload";
			currentFolderStr = "/";
		}

		ResourceType fileType = ResourceType.getDefaultResourceType(typeStr);

		UploadResponse ur = null;

		if (!ConfigurationHandler.isEnabledForFileUpload(request)) {
			ur = new UploadResponse(UploadResponse.EN_SECURITY_ERROR, null,
					null,
					"The current user isn't authorized for uploading files!");
		} else if (isEmpty(commandStr) || isEmpty(currentFolderStr)
				|| isEmpty(typeStr))
			ur = UploadResponse.UR_BAD_REQUEST;
		else if (!commandStr.matches("(File|Quick)Upload"))
			ur = UploadResponse.UR_SECURITY_ERROR;
		else {

			String currentPath = constructTypeBasedFolderString(fileType,
					currentFolderStr, request);
			String currentDirPath = getServletContext()
					.getRealPath(currentPath);

			if (!isValidPath(currentFolderStr)
					|| !(new File(currentDirPath).exists()))
				ur = UploadResponse.UR_BAD_REQUEST;
			else {

				String newFilename = null;
				FileItemFactory factory = new DiskFileItemFactory();
				ServletFileUpload upload = new ServletFileUpload(factory);
				try {
					List<FileItem> items = upload.parseRequest(request);

					// We upload only one file at the same time
					FileItem uplFile = items.get(0);
					String filename = FilenameUtils.getName(uplFile.getName());
					String baseName = FilenameUtils.removeExtension(filename);
					String extension = FilenameUtils.getExtension(filename);

					boolean validExtension = ExtensionsHandler.isAllowed(
							fileType, extension);

					if (!validExtension)
						ur = UploadResponse.UR_INVALID_EXTENSION;
					else {

						if (ConfigurationHandler.isForceSingleExtension()) {
							filename = forceSingleExtension(filename);
							baseName = FilenameUtils.removeExtension(filename);
						}

						File pathToSave = new File(currentDirPath, filename);
						int counter = 1;
						while (pathToSave.exists()) {
							newFilename = baseName + "(" + counter + ")" + "."
									+ extension;
							pathToSave = new File(currentDirPath, newFilename);
							counter++;
						}

						uplFile.write(pathToSave);
						if (isEmpty(newFilename)) {
							ur = new UploadResponse(UploadResponse.EN_OK,
									request.getContextPath() + currentPath
											+ filename);
						} else {
							ur = new UploadResponse(UploadResponse.EN_RENAMED,
									request.getContextPath() + currentPath
											+ newFilename, newFilename);
						}
					}
				} catch (FileUploadException ex) {
					ur = UploadResponse.UR_BAD_REQUEST;
				} catch (Exception e) {
					ur = UploadResponse.UR_SECURITY_ERROR;
				}
			}
		}
		out.print(ur);
		out.flush();
		out.close();

		logger.debug("Exiting Connector#doPost");
	}

	private void createResponse(String retValue, String msg, Node root,
			Document doc) {
		Element myEl = doc.createElement("Error");
		myEl.setAttribute("number", retValue);
		if (Utils.isNotEmpty(msg))
			myEl.setAttribute("text", msg);
		root.appendChild(myEl);
	}

	private void createFolderResponse(String retValue, Node root, Document doc) {
		createResponse(retValue, null, root, doc);
	}

	private void getFolders(File dir, Node root, Document doc) {
		Element folders = doc.createElement("Folders");
		root.appendChild(folders);
		File[] fileList = dir
				.listFiles((FileFilter) DirectoryFileFilter.DIRECTORY);
		for (File file : fileList) {
			Element folderElement = doc.createElement("Folder");
			folderElement.setAttribute("name", file.getName());
			folders.appendChild(folderElement);
		}
	}

	private void getFiles(File dir, Node root, Document doc) {
		Element files = doc.createElement("Files");
		root.appendChild(files);
		File[] fileList = dir.listFiles((FileFilter) FileFileFilter.FILE);

		for (File file : fileList) {
			Element fileElement = doc.createElement("File");
			fileElement.setAttribute("name", file.getName());
			fileElement.setAttribute("size", String
					.valueOf(file.length() / 1024));
			files.appendChild(fileElement);
		}
	}

	private Node createCommonXml(Document doc, String commandStr,
			String typeStr, String currentPath, String currentUrl) {
		Element root = doc.createElement("Connector");
		doc.appendChild(root);
		root.setAttribute("command", commandStr);
		root.setAttribute("resourceType", typeStr);

		Element myEl = doc.createElement("CurrentFolder");
		myEl.setAttribute("path", currentPath);
		myEl.setAttribute("url", currentUrl);
		root.appendChild(myEl);

		return root;
	}

	private String constructTypeBasedFolderString(final ResourceType fileType,
			final String currentFolderString, final HttpServletRequest request) {
		StringWriter retval = new StringWriter();
		retval.write(ConfigurationHandler.getUserFilesPath(request));
		retval.write(fileType.getPath());
		retval.write(currentFolderString);
		return replaceAll(retval.toString(), "//", "/");
	}

	/**
	 * Initializes the {@link ExtensionsHandler} with the allowed or denied
	 * extensions of a file type.
	 * 
	 * @param type
	 * @param allowedList
	 * @param deniedList
	 * @throws IllegalArgumentException
	 *             if allowed and denied extensions are set.
	 */
	private void setExtension(final ResourceType type,
			final String allowedList, final String deniedList) {
		// if both lists are set, we have to throw an error, because only one
		// list should be set
		if (Utils.isNotEmpty(allowedList) && Utils.isNotEmpty(deniedList)) {
			String errorMsg = "Allowed and denied extensions are set for ["
					+ type + "]. Just one of them should be set!";
			logger.error(errorMsg);
			throw new IllegalArgumentException(errorMsg);
		}

		if (Utils.isNotEmpty(allowedList))
			ExtensionsHandler.setExtensionsAllowed(type, allowedList);
		else
			ExtensionsHandler.setExtensionsDenied(type, deniedList);
	}
}
