/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.connector;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.fckeditor.handlers.CommandHandler;
import net.fckeditor.handlers.ConnectorHandler;
import net.fckeditor.handlers.ResourceType;
import net.fckeditor.response.GetResponse;
import net.fckeditor.response.UploadResponse;
import net.fckeditor.tool.Utils;
import net.fckeditor.tool.UtilsFile;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Encapsulates the verification of the parameters and {@link Connector}-calls. So the complete 
 * XML-handling could be changed without affort to the {@link Connector}s.
 * 
 * @version $Id$
 */
public class Dispatcher {
	private static final Logger logger = LoggerFactory.getLogger(Dispatcher.class);
	private Connector connector = null;
	
	protected Dispatcher(final Connector connector) {
		this.connector = connector;
		this.connector.init(null);
		logger.info("Initialized!");
	}
	
	
	/**
	 * Manage the <code>GET</code> requests (<code>GetFolders</code>,
	 * <code>GetFoldersAndFiles</code>, <code>CreateFolder</code>).<br/>
	 * 
	 * The method accepts commands sent in the following format:<br/>
	 * <code>connector?Command=&lt;CommandName&gt;&Type=&lt;ResourceType&gt;&CurrentFolder=&lt;FolderPath&gt;</code>
	 * <p>
	 * It executes the relevant commands and then returns the result to the client in XML
	 * format.
	 * </p>
	 */
	public void doGet(final HttpServletRequest request, final HttpServletResponse response) throws IOException {
		logger.debug("Entering Dispatcher#doGet");

		response.setCharacterEncoding("UTF-8");
		response.setContentType("application/xml; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();

		String commandStr = request.getParameter("Command");
		String typeStr = request.getParameter("Type");
		String currentFolderStr = request.getParameter("CurrentFolder");

		logger.debug("Parameter Command: {}", commandStr);
		logger.debug("Parameter Type: {}", typeStr);
		logger.debug("Parameter CurrentFolder: {}", currentFolderStr);
		
		GetResponse getResponse;
		// check conditions/parameters
		// FIXME check permissions for user actions !!!
		if (!CommandHandler.isValidForGet(commandStr))
			getResponse = GetResponse.getErrorInvalidCommand();
		else if (!ResourceType.isValid(typeStr))
			getResponse = GetResponse.getErrorInvalidType();
		else if (!UtilsFile.isValidPath(currentFolderStr))
			getResponse = GetResponse.getErrorInvalidCurrentFolder();
		else {
			
			// call the right Connector method depending on the command
			ResourceType type = ResourceType.getDefaultResourceType(typeStr);
			CommandHandler command = CommandHandler.getCommand(commandStr);
			if (command.equals(CommandHandler.GET_FOLDERS))
				getResponse = connector.getFolders(type, currentFolderStr);
			else if (command.equals(CommandHandler.GET_FOLDERS_AND_FILES))
				getResponse = connector.getFilesAndFolders(type, currentFolderStr);
			else if (command.equals(CommandHandler.CREATE_FOLDER)) {
				String newFolderStr = UtilsFile.sanitizeFolderName(request.getParameter("NewFolderName"));
				logger.debug("Parameter NewFolderName: {}", newFolderStr);
				getResponse = connector.createFolder(type, currentFolderStr, newFolderStr);
			} else 
				getResponse = GetResponse.getErrorUnknown();
		}
		
		out.print(getResponse);
		out.flush();
		out.close();
		logger.debug("Exiting Dispatcher#doGet");
	}

	/**
	 * Manage the <code>POST</code> requests (<code>FileUpload</code>).<br />
	 * 
	 * The method accepts commands sent in the following format:<br />
	 * <code>connector?Command=&lt;FileUpload&gt;&Type=&lt;ResourceType&gt;&CurrentFolder=&lt;FolderPath&gt;</code>
	 * with the file in the <code>POST</code> body.<br />
	 * <br>
	 * The Connector stores an uploaded file (renames a file if another exists with the
	 * same name) and then returns the JavaScript callback.
	 * @throws IOException 
	 */
	public void doPost(final HttpServletRequest request, final HttpServletResponse response) throws IOException {
		logger.debug("Entering Dispatcher#doPost");
		response.setCharacterEncoding("UTF-8");
		response.setContentType("text/html; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();

		String commandStr = request.getParameter("Command");
		String typeStr = request.getParameter("Type");
		String currentFolderStr = request.getParameter("CurrentFolder");

		logger.debug("Parameter Command: {}", commandStr);
		logger.debug("Parameter Type: {}", typeStr);
		logger.debug("Parameter CurrentFolder: {}", currentFolderStr);
		logger.debug("Exiting Dispatcher#doPost");

		// if this is a QuickUpload request, 'commandStr' and 'currentFolderStr'
		// are empty
		if (Utils.isEmpty(commandStr) && Utils.isEmpty(currentFolderStr)) {
			commandStr = "QuickUpload";
			currentFolderStr = "/";
		}
		
		UploadResponse uploadResponse;
		// FIXME check permissions for user actions !!!
		if (!CommandHandler.isValidForPost(commandStr))
			uploadResponse = UploadResponse.getErrorInvalidCommand();
		else if (!ResourceType.isValid(typeStr))
			uploadResponse = UploadResponse.getErrorInvalidType();
		else if (!UtilsFile.isValidPath(currentFolderStr))
			uploadResponse = UploadResponse.getErrorInvalidCurrentFolder();
		else {

			// call the Connector#fileUpload
			ResourceType type = ResourceType.getDefaultResourceType(typeStr);
			FileItemFactory factory = new DiskFileItemFactory();
			ServletFileUpload upload = new ServletFileUpload(factory);
			try {
				@SuppressWarnings("unchecked") List<FileItem> items = upload.parseRequest(request);
				// We upload just one file at the same time
				FileItem uplFile = items.get(0);
				// Secure image check
				if (type.equals(ResourceType.IMAGE) && ConnectorHandler.isSecureImageUploads()
						&& !UtilsFile.isImage(uplFile.getInputStream())) {
					uploadResponse = UploadResponse.getErrorInvalidExtension();
				} else {
					String fileName = FilenameUtils.getName(UtilsFile.sanitizeFileName(uplFile.getName()));
					/* TODO check, if we could get out of memory, if we have a lot of uploads. Maybe it's better to
					to commit the FileItem itself and the Connector has to do the deletion??*/
					uploadResponse = connector.fileUpload(type, currentFolderStr, fileName, uplFile.get());
				}
				uplFile.delete();
			} catch (Exception e) {
				uploadResponse = UploadResponse.getErrorSecurity();
			}
		}
		out.print(uploadResponse);
		out.flush();
		out.close();
		logger.debug("Exiting Dispatcher#doPost");
	}
	
	
}
