/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.connector.impl;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletContext;

import net.fckeditor.connector.Connector;
import net.fckeditor.handlers.CommandHandler;
import net.fckeditor.handlers.ConnectorHandler;
import net.fckeditor.handlers.ExtensionsHandler;
import net.fckeditor.handlers.PropertiesLoader;
import net.fckeditor.handlers.ResourceType;
import net.fckeditor.response.GetResponse;
import net.fckeditor.response.UploadResponse;
import net.fckeditor.tool.UtilsFile;
import net.fckeditor.tool.UtilsResponse;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.commons.io.filefilter.FileFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Implementation of the {@link Connector} for the file system. All path are interpreted as sub-directories 
 * of the {@link ConnectorHandler#getUserFilesPath()}.<br>
 * 
 * TODO should we move the static methods into a util-class in package net.fckeditor.connector.impl ???
 * @version $Id: SimpleFileSystemConnector.java 2436 2008-09-05 10:50:04Z th-schwarz $
 */
public class SimpleFileSystemConnector implements Connector {
	private static final Logger logger = LoggerFactory.getLogger(SimpleFileSystemConnector.class);
	private static Map<ResourceType, String> paths = new HashMap<ResourceType, String>(4);
	private static ServletContext servletContext;
	
	static {
		// initialize the sub folders for each resource type
		paths.put(ResourceType.FILE, PropertiesLoader.getProperty("connector.resourceType.file.path"));
		paths.put(ResourceType.IMAGE, PropertiesLoader.getProperty("connector.resourceType.image.path"));
		paths.put(ResourceType.FLASH, PropertiesLoader.getProperty("connector.resourceType.flash.path"));
		paths.put(ResourceType.MEDIA, PropertiesLoader.getProperty("connector.resourceType.media.path"));
	}

	/* (non-Javadoc)
	 * @see net.fckeditor.connector.Connector#init()
	 */
	public void init(final ServletContext servletContext) {
		// create user's default dir
		SimpleFileSystemConnector.servletContext = servletContext;
		String realDefaultUserFilesPath = SimpleFileSystemConnector.servletContext.getRealPath(ConnectorHandler.getDefaultUserFilesPath());
		File defaultUserFilesDir = new File(realDefaultUserFilesPath);
		UtilsFile.checkDirAndCreate(defaultUserFilesDir);
		logger.info("Initialized!");
	}
	

	/* (non-Javadoc)
	 * @see net.fckeditor.connector.Connector#fileUpload(net.fckeditor.handlers.ResourceType, java.lang.String, java.lang.String, java.io.InputStream)
	 */
	public UploadResponse fileUpload(final ResourceType type, final String currentFolder, final String fileName, final InputStream inputStream) {
		if (!ExtensionsHandler.isAllowed(type, FilenameUtils.getExtension(fileName)))
			return UploadResponse.getErrorInvalidExtension();
		String baseDir = ConnectorHandler.getDefaultUserFilesPath();
		File typeDir = getAndCreateResourceTypeDir(baseDir, type);
		File currentDir = new File(typeDir, currentFolder);
		if (!currentDir.exists())
			return UploadResponse.getErrorInvalidCurrentFolder();
		File newFile = new File(currentDir, fileName);
		File fileToSave = getUniqueFile(newFile.getAbsoluteFile());
		try {
			BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(fileToSave));
			IOUtils.copyLarge(inputStream, out);
		} catch (Exception e) {
			return UploadResponse.getErrorSecurity();
		}
		String responseUrl = UtilsResponse.constructResponseUrl(type, paths.get(type), ConnectorHandler.getUserFilesPath(), currentFolder)
				.concat(fileToSave.getName());
		UploadResponse uploadResponse;
		if (!fileToSave.getAbsoluteFile().equals(newFile.getAbsoluteFile()))
			uploadResponse = new UploadResponse(UploadResponse.SC_RENAMED, responseUrl, fileToSave.getName());
		else
			uploadResponse = new UploadResponse(UploadResponse.SC_OK, responseUrl);
		return uploadResponse;
	}
	
	/* (non-Javadoc)
	 * @see net.fckeditor.connector.Connector#createFolder(net.fckeditor.handlers.ResourceType, java.lang.String, java.lang.String)
	 */
	public GetResponse createFolder(final ResourceType type, final String currentFolder, final String newFolder) {
		String baseDir = ConnectorHandler.getDefaultUserFilesPath();
		File typeDir = getAndCreateResourceTypeDir(baseDir, type);
		File currentDir = new File(typeDir, currentFolder);
		if (!currentDir.exists())
			return GetResponse.getErrorInvalidCurrentFolder();
		File newDir = new File(currentDir, newFolder);
		if (newDir.exists())
			return GetResponse.getErrorFolderAlreadyExists();
		try {
			return (newDir.mkdir()) ? GetResponse.getOK() : GetResponse.getErrorInvalidFolderName();
		} catch (SecurityException e) {
			return GetResponse.getErrorSecurity();
		}
	}

	/* (non-Javadoc)
	 * @see net.fckeditor.connector.Connector#getFolders(net.fckeditor.handlers.ResourceType, java.lang.String)
	 */
	public GetResponse getFolders(final ResourceType type, final String currentFolder) {
		String baseDir = ConnectorHandler.getDefaultUserFilesPath();
		File typeDir = getAndCreateResourceTypeDir(baseDir, type);
		File currentDir = new File(typeDir, currentFolder);
		if (!currentDir.exists())
			return GetResponse.getErrorInvalidCurrentFolder();
		
		GetResponse getResponse = new GetResponse(CommandHandler.GET_FOLDERS_AND_FILES, type, 
				currentFolder, UtilsResponse.constructResponseUrl(type, paths.get(type), ConnectorHandler.getUserFilesPath(), currentFolder));
		getResponse.setFolders(getFolders(currentDir));
		return getResponse;
	}

	/* (non-Javadoc)
	 * @see net.fckeditor.connector.Connector#getFiles(net.fckeditor.handlers.ResourceType, java.lang.String)
	 */
	public GetResponse getFilesAndFolders(final ResourceType type, final String currentFolder) {
		String baseDir = ConnectorHandler.getDefaultUserFilesPath();
		File typeDir = getAndCreateResourceTypeDir(baseDir, type);
		File currentDir = new File(typeDir, currentFolder);
		if (!currentDir.exists())
			return GetResponse.getErrorInvalidCurrentFolder();
		
		GetResponse getResponse = new GetResponse(CommandHandler.GET_FOLDERS_AND_FILES, type, 
				currentFolder, UtilsResponse.constructResponseUrl(type, paths.get(type), ConnectorHandler.getUserFilesPath(), currentFolder));
		getResponse.setFolders(getFolders(currentDir));
		// collect files
		File[] fileList = currentDir.listFiles((FileFilter) FileFileFilter.FILE);
		Map<String, Long> files = new HashMap<String, Long>(fileList.length);
		for (File file : fileList) {
			files.put(file.getName(), new Long(file.length()));
		}
		getResponse.setFiles(files);
		return getResponse;
	}
	
	private static List<String> getFolders(final File currentDir) {
		if (!currentDir.isDirectory())
			throw new IllegalArgumentException();
		String[] fileList = currentDir.list(DirectoryFileFilter.DIRECTORY);
		return Arrays.asList(fileList);
	}
	
	private static File getAndCreateResourceTypeDir(final String baseDir, final ResourceType type) {
		File dir = new File(servletContext.getRealPath(baseDir), paths.get(type));
		if (!dir.exists())
			dir.mkdirs();
		return dir;
	}
	
	private static File getUniqueFile(final File file) {
		if (!file.exists())
			return file;
		
		File tmpFile = new File(file.getAbsolutePath());
		File dir = tmpFile.getParentFile();
		int count = 1;
		String extension = ".".concat(FilenameUtils.getExtension(tmpFile.getName()));
		String baseName = FilenameUtils.getBaseName(tmpFile.getName());
		do {
			tmpFile = new File(dir, baseName + "(".concat(String.valueOf(count)).concat(")").concat(extension));
			count++;
		} while (tmpFile.exists());
		return tmpFile;
	}
}
