/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.connector.impl;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletContext;

import net.fckeditor.connector.Connector;
import net.fckeditor.connector.exception.FolderAlreadyExistsException;
import net.fckeditor.connector.exception.InvalidCurrentFolderException;
import net.fckeditor.connector.exception.InvalidFolderNameException;
import net.fckeditor.connector.exception.SecurityIssueException;
import net.fckeditor.connector.exception.UnknownException;
import net.fckeditor.handlers.ConnectorHandler;
import net.fckeditor.handlers.PropertiesLoader;
import net.fckeditor.handlers.ResourceType;
import net.fckeditor.tool.UtilsFile;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.commons.io.filefilter.FileFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Implementation of the {@link Connector} for the file system. All path are interpreted as sub-directories 
 * of the {@link ConnectorHandler#getUserFilesPath()}.<br>
 * 
 * TODO should we move the static methods into a util-class in package net.fckeditor.connector.impl ???
 * @version $Id: SimpleFileSystemConnector.java 2460 2008-09-14 13:39:54Z mosipov $
 */
// FIXME the name is inappropriate, it should be ContextFileSystemConnector
// we should provide another implementation which leads outside of the context
public class SimpleFileSystemConnector implements Connector {
	private static final Logger logger = LoggerFactory.getLogger(SimpleFileSystemConnector.class);
	private static Map<ResourceType, String> paths = new HashMap<ResourceType, String>(4);
	private static ServletContext servletContext;
	
	// TODO I don't like this because every one has to initialize this on his own
	static {
		// initialize the sub folders for each resource type
		paths.put(ResourceType.FILE, PropertiesLoader.getProperty("connector.resourceType.file.path"));
		paths.put(ResourceType.IMAGE, PropertiesLoader.getProperty("connector.resourceType.image.path"));
		paths.put(ResourceType.FLASH, PropertiesLoader.getProperty("connector.resourceType.flash.path"));
		paths.put(ResourceType.MEDIA, PropertiesLoader.getProperty("connector.resourceType.media.path"));
	}

	/* (non-Javadoc)
	 * @see net.fckeditor.connector.Connector#init()
	 */
	public void init(final ServletContext servletContext) {
		// create user's default dir
		SimpleFileSystemConnector.servletContext = servletContext;
		String realDefaultUserFilesPath = SimpleFileSystemConnector.servletContext.getRealPath(ConnectorHandler.getDefaultUserFilesPath());
		File defaultUserFilesDir = new File(realDefaultUserFilesPath);
		UtilsFile.checkDirAndCreate(defaultUserFilesDir);
		logger.info("Initialized!");
	}
	

	/* (non-Javadoc)
	 * @see net.fckeditor.connector.Connector#fileUpload(net.fckeditor.handlers.ResourceType, java.lang.String, java.lang.String, java.io.InputStream)
	 */
	public String fileUpload(final ResourceType type, final String currentFolder, final String fileName, final InputStream inputStream) 
			throws InvalidCurrentFolderException, SecurityIssueException {
		String baseDir = ConnectorHandler.getDefaultUserFilesPath();
		File typeDir = getAndCreateResourceTypeDir(baseDir, type);
		File currentDir = new File(typeDir, currentFolder);
		if (!currentDir.exists())
			throw new InvalidCurrentFolderException();
		File newFile = new File(currentDir, fileName);
		File fileToSave = getUniqueFile(newFile.getAbsoluteFile());
		try {
			BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(fileToSave));
			IOUtils.copyLarge(inputStream, out);
		} catch (Exception e) {
			throw new SecurityIssueException();
		}
		return fileToSave.getName();
	}
	
	/* (non-Javadoc)
	 * @see net.fckeditor.connector.Connector#createFolder(net.fckeditor.handlers.ResourceType, java.lang.String, java.lang.String)
	 */
	public void createFolder(final ResourceType type, final String currentFolder, final String newFolder) 
			throws InvalidCurrentFolderException, SecurityIssueException, InvalidFolderNameException, FolderAlreadyExistsException {
		String baseDir = ConnectorHandler.getDefaultUserFilesPath();
		File typeDir = getAndCreateResourceTypeDir(baseDir, type);
		File currentDir = new File(typeDir, currentFolder);
		if (!currentDir.exists())
			throw new InvalidCurrentFolderException();
		File newDir = new File(currentDir, newFolder);
		if (newDir.exists())
			throw new FolderAlreadyExistsException();
		try {
			if (!newDir.mkdir()) 
				throw new InvalidFolderNameException();
		} catch (SecurityException e) {
			throw new SecurityIssueException();
		}
	}

	/* (non-Javadoc)
	 * @see net.fckeditor.connector.Connector#getFiles(net.fckeditor.handlers.ResourceType, java.lang.String)
	 */
	public Map<String, Long> getFiles(ResourceType type, String currentFolder)
			throws InvalidCurrentFolderException, SecurityIssueException, UnknownException {
		String baseDir = ConnectorHandler.getDefaultUserFilesPath();
		File typeDir = getAndCreateResourceTypeDir(baseDir, type);
		File currentDir = new File(typeDir, currentFolder);
		if (!currentDir.exists())
			throw new InvalidCurrentFolderException();

		// collect files
		Map<String, Long> files;
		try {
			File[] fileList = currentDir.listFiles((FileFilter) FileFileFilter.FILE);
			files = new HashMap<String, Long>(fileList.length);
			for (File file : fileList)
				files.put(file.getName(), new Long(file.length()));
			return files;
		} catch (SecurityException e) {
			throw new SecurityIssueException();
		}
	}


	/* (non-Javadoc)
	 * @see net.fckeditor.connector.Connector#getFolders(net.fckeditor.handlers.ResourceType, java.lang.String)
	 */
	public  List<String> getFolders(final ResourceType type, final String currentFolder)
			throws InvalidCurrentFolderException, SecurityIssueException, UnknownException {
		String baseDir = ConnectorHandler.getDefaultUserFilesPath();
		File typeDir = getAndCreateResourceTypeDir(baseDir, type);
		File currentDir = new File(typeDir, currentFolder);
		if (!currentDir.exists())
			throw new InvalidCurrentFolderException();
		if (!currentDir.isDirectory())
			throw new UnknownException();
		try {
			String[] fileList = currentDir.list(DirectoryFileFilter.DIRECTORY);
			return Arrays.asList(fileList);
		} catch (Exception e) {
			throw new SecurityIssueException();
		}
	}

	
	private static File getAndCreateResourceTypeDir(final String baseDir, final ResourceType type) {
		File dir = new File(servletContext.getRealPath(baseDir), paths.get(type));
		if (!dir.exists())
			dir.mkdirs();
		return dir;
	}
	
	// TODO maybe this can be achieved in fewer lines
	private static File getUniqueFile(final File file) {
		if (!file.exists())
			return file;
		
		File tmpFile = new File(file.getAbsolutePath());
		File dir = tmpFile.getParentFile();
		int count = 1;
		String extension = ".".concat(FilenameUtils.getExtension(tmpFile.getName()));
		String baseName = FilenameUtils.getBaseName(tmpFile.getName());
		do {
			tmpFile = new File(dir, baseName + "(".concat(String.valueOf(count)).concat(")").concat(extension));
			count++;
		} while (tmpFile.exists());
		return tmpFile;
	}
}
