/*
 * CKEditor - The text editor for Internet - http://ckeditor.com
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 *
 * == BEGIN LICENSE ==
 *
 * Licensed under the terms of any of the following licenses at your
 * choice:
 *
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 *
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 *
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 *
 * == END LICENSE ==
 */

/**
 * @fileOverview The floating dialog plugin.
 */

CKEDITOR.plugins.add( 'dialog',
	{
		requires : [ 'dialogui' ]
	});

/**
 * No resize for this dialog.
 * @constant
 */
CKEDITOR.DIALOG_RESIZE_NONE = 0;

/**
 * Only allow horizontal resizing for this dialog, disable vertical resizing.
 * @constant
 */
CKEDITOR.DIALOG_RESIZE_WIDTH = 1;

/**
 * Only allow vertical resizing for this dialog, disable horizontal resizing.
 * @constant
 */
CKEDITOR.DIALOG_RESIZE_HEIGHT = 2;

/*
 * Allow the dialog to be resized in both directions.
 * @constant
 */
CKEDITOR.DIALOG_RESIZE_BOTH = 3;


/**
 * This is the base class for runtime dialog objects.
 * @param {Object} editor The editor which created the dialog.
 * @param {String} dialogName The dialog's registered name.
 * @constructor
 * @example
 * var dialogObj = new CKEDITOR.dialog( editor, 'smiley' );
 */
CKEDITOR.dialog = function( editor, dialogName )
{
	if ( arguments.length < 2 )
		return;

	// Load the dialog definition.
	var definition = CKEDITOR.dialog._.dialogDefinitions[dialogName];
	if ( typeof( definition ) != 'function' )
	{
		console.log( 'Error: The dialog "' + dialogName + '" is not defined.' );
		return;
	}
	definition = CKEDITOR.tools.extend( {}, CKEDITOR.dialog._.defaultDialogDefinition( editor ), definition( editor ), true );
	if ( !( definition.title && definition.contents ) )
	{
		console.log( 'Error: The dialog "' + dialogName + '" is missing its title or contents.' );
		return;
	}
	if ( definition.contents.length < 1 )
	{
		console.log( 'Error: The dialog "' + dialogName + '" has no contents.' );
		return;
	}
	definition = new CKEDITOR.dialog._.definitionObject( this, definition );
	definition = CKEDITOR.fire( 'dialogDefinition', { name : dialogName, definition : definition }, editor ).definition;

	// Initialize some basic parameters.
	CKEDITOR.tools.extend( ( this._ || ( this._ = {} ) ), {
			editor : editor,
			element : editor.theme.buildDialog( editor ),
			name : dialogName,
			definition : definition,
			size : { width : 0, height : 0 },
			contents : {},
			buttons : {},
			accessKeyMap : {}
		}, true );

	// Initialize the parts map.
	var element = this._.element.getFirst(),
		i, me = this;

	/**
	 * An associative map of elements in the dialog. It has the following members:
	 * <ul>
	 * 	<li>tl - top left corner.</li>
	 * 	<li>tl_resize - resize handle at the top left corner.</li>
	 * 	<li>t - top side.</li>
	 * 	<li>t_resize - resize handle at the top.</li>
	 * 	<li>tr - top right corner.</li>
	 * 	<li>tr_resize - resize handle at the top right.</li>
	 * 	<li>l - left side.</li>
	 * 	<li>l_resize - resize handle at the left side.</li>
	 * 	<li>c - center area.</li>
	 * 	<li>r - right side.</li>
	 * 	<li>r_resize - resize handle at the right side.</li>
	 * 	<li>bl - bottom left corner.</li>
	 * 	<li>bl_resize - resize handle at the bottom left.</li>
	 * 	<li>b - bottom side.</li>
	 * 	<li>b_resize - resize handle at the bottom.</li>
	 * 	<li>br - bottom right corner.</li>
	 * 	<li>br_resize - resize handle at the bottom right.</li>
	 * 	<li>title - title area.</li>
	 * 	<li>close - close button.</li>
	 * 	<li>tabs - tabs area.</li>
	 * 	<li>bridge - the bridge between the active tab and the content page.</li>
	 * 	<li>contents - the content page area.</li>
	 * 	<li>footer - the footer area.</li>
	 * </ul>
	 * @type Object
	 * @field
	 */
	this.parts = {
		'tl' : [0,0],
		'tl_resize' : [0,0,0],
		't' : [0,1],
		't_resize' : [0,1,0],
		'tr' : [0,2],
		'tr_resize' : [0,2,0],
		'l' : [1,0],
		'l_resize' : [1,0,0],
		'c' : [1,1],
		'r' : [1,2],
		'r_resize' : [1,2,0],
		'bl' : [2,0],
		'bl_resize' : [2,0,0],
		'b' : [2,1],
		'b_resize' : [2,1,0],
		'br' : [2,2],
		'br_resize' : [2,2,0],
		'title' : [1,1,0],
		'close' : [1,1,0,0],
		'tabs' : [1,1,1],
		'bridge' : [1,1,2],
		'contents' : [1,1,3],
		'footer' : [1,1,4]
	};
	for ( i in this.parts )
		this.parts[i] = element.getChild( this.parts[i] );

	// Initialize the tab and page map.
	CKEDITOR.tools.extend (this._,
		{
			tabs : {},
			pageCount : 0,
			lastTab : null
		}, true);

	// Make the dialog an event hub.
	CKEDITOR.event.implementOn( this );

	// Initialize load, ok and cancel events.
	if ( definition.onLoad )
		this.on( 'load', definition.onLoad, this, null, CKEDITOR.tools.getNextNumber() );
	if ( definition.onOk )
		this.on( 'ok', function( evt )
				{
					if ( definition.onOk.call( this, evt ) === false )
						evt.data.hide = false;
				}, this, null, 0 );
	if ( definition.onCancel )
		this.on( 'cancel', function( evt )
				{
					if ( definition.onCancel.call( this, evt ) === false )
						evt.data.hide = false;
				}, this, null, 0);

	/** @ignore */
	var iterContents = function( func )
	{
		var breakFlag = false;
		for ( var i in me._.contents )
		{
			for ( var j in me._.contents[i] )
			{
				breakFlag = func.call( this, me._.contents[i][j] );
				if ( breakFlag )
					break;
			}
			if ( breakFlag )
				break;
		}
	}

	this.on( 'ok', function( evt )
			{
				iterContents( function( item )
					{
						if ( item.validate && !item.validate( this ) )
						{
							evt.data.hide = false;
							return true;
						}
					});
			}, this, null, 0 );
	this.on( 'cancel', function( evt )
			{
				iterContents( function( item )
					{
						if ( item.isChanged() )
						{
							if ( !confirm( 'Some of the options have been changed. Are you sure to close the dialog?' ) )
								evt.data.hide = false;
							return true;
						}
					});
			}, this, null, 0 );

	this.parts.close.on( 'click', function( evt )
			{
				if ( this.fire( 'cancel', { hide : true } ).hide !== false )
					this.hide();
			}, this );

	CKEDITOR.dialog._.initDragAndDrop( this );
	CKEDITOR.dialog._.initResizeHandles( this );

	// Insert the title.
	( new CKEDITOR.dom.text( definition.title, CKEDITOR.document ) ).appendTo( this.parts.title );

	// Insert the tabs and contents.
	for ( i = 0 ; i < definition.contents.length ; i++ )
		this.addPage( definition.contents[i] );

	var classRegex = /cke_dialog_tab(\s|$|_)/;
	this.parts['tabs'].on( 'click', function( evt )
			{
				var target = evt.data.getTarget(), firstNode = target, id, page;

				// If we aren't inside a tab, bail out.
				if ( !classRegex.test( target.$.className ) )
					return;

				// Find the outer <span> container of the tab.
				while ( target.getName() == 'div' && classRegex.test( target.$.className ) )
					target = target.getParent();
				id = target.$.id.substr( 0, target.$.id.lastIndexOf( '_' ) );
				this.selectPage( id );
			}, this );

	// Insert buttons.
	var buttonsHtml = [],
		buttons = CKEDITOR.dialog._.uiElementBuilders.hbox.build( this,
			{
				type : 'hbox',
				className : 'cke_dialog_footer_buttons',
				widths : [],
				children : definition.buttons
			}, buttonsHtml ).getChild();
	this.parts.footer.setHtml( buttonsHtml.join( '' ) );

	for ( var i = 0 ; i < buttons.length ; i++ )
		this._.buttons[ buttons[i].id ] = buttons[i];

	// Insert dummy text box for grabbing focus away from the editing area.
	this._.dummyText = CKEDITOR.dom.element.createFromHtml( '<input type="text" style="position: absolute; left: -100000px; top: -100000px" />' );
	this._.dummyText.appendTo( element );

	CKEDITOR.skins.load( editor.config.skin, 'dialog' );
};

CKEDITOR.dialog.prototype = 
{
	/**
	 * Resizes the dialog.
	 * @param {Number} width The width of the dialog in pixels.
	 * @param {Number} height The height of the dialog in pixels.
	 * @function
	 * @example
	 * dialogObj.resize( 800, 640 );
	 */
	resize : (function()
	{
		return function( width, height )
		{
			if ( this._.size && this._.size.width == width && this._.size.height == height )
				return;

			CKEDITOR.dialog.fire( 'resize',
				{
					dialog : this,
					skin : this._.editor.config.skin,
					width : width,
					height : height
				}, this._.editor );

			this._.size = { width : width, height : height };
		};
	})(),

	/**
	 * Gets the current size of the dialog in pixels.
	 * @returns {Object} An object with "width" and "height" properties.
	 * @example
	 * var width = dialogObj.getSize().width;
	 */
	getSize : function()
	{
		return CKEDITOR.tools.extend( {}, this._.size );
	},

	/**
	 * Moves the dialog to an (x, y) coordinate relative to the window.
	 * @function
	 * @param {Number} x The target x-coordinate.
	 * @param {Number} y The target y-coordinate.
	 * @example
	 * dialogObj.move( 10, 40 );
	 */
	move : (function()
	{
		var isFixed;
		return function( x, y )
		{
			// The dialog may be fixed positioned or absolute positioned. Ask the
			// browser what is the current situation first.
			if ( isFixed === undefined )
				isFixed = this._.element.getFirst().getComputedStyle( 'position' ) == 'fixed';

			if ( isFixed && this._.position && this._.position.x == x && this._.position.y == y )
				return;

			// Save the current position.
			this._.position = { x : x, y : y };

			// If not fixed positioned, add scroll position to the coordinates.
			if ( !isFixed )
			{
				var scrollPosition = CKEDITOR.document.getWindow().getScrollPosition();
				x += scrollPosition.x;
				y += scrollPosition.y;
			}

			this._.element.getFirst().setStyles(
					{
						'left' : x + 'px',
						'top' : y + 'px'
					});
		}
	})(),

	/**
	 * Gets the dialog's position in the window.
	 * @returns {Object} An object with "x" and "y" properties.
	 * @example
	 * var dialogX = dialogObj.getPosition().x;
	 */
	getPosition : function(){ return CKEDITOR.tools.extend( {}, this._.position ); },

	/**
	 * Shows the dialog box.
	 * @example
	 * dialogObj.show();
	 */
	show : function()
	{
		// Insert the dialog's element to the root document.
		var element = this._.element;
		var definition = this._.definition;
		if ( !( element.getParent() && element.getParent().equals( CKEDITOR.document.getBody() ) ) )
			element.appendTo( CKEDITOR.document.getBody() );
		else
			return;

		// First, set the dialog to an appropriate size.
		this.resize( definition.minWidth, definition.minHeight );

		// Rearrange the dialog to the middle of the window.
		var viewSize = CKEDITOR.document.getWindow().getViewPaneSize();
		this.move( ( viewSize.width - this._.size.width ) / 2, ( viewSize.height - this._.size.height ) / 2 );
		
		// Select the first tab by default.
		this.selectPage( this._.definition.contents[0].id );

		// Reset all inputs back to their default value.
		for ( var i in this._.contents )
			for ( var j in this._.contents[i] )
			{
				var c = this._.contents[i][j];
				if ( c.reset )
					c.reset();
			}

		// Set z-index.
		if ( CKEDITOR.dialog._.currentZIndex === null )
			CKEDITOR.dialog._.currentZIndex = this._.editor.config.basePopupZIndex;
		this._.element.getFirst().setStyle( 'z-index', CKEDITOR.dialog._.currentZIndex += 10 );

		// Maintain the dialog ordering and dialog cover.
		// Also register key handlers if first dialog.
		if ( CKEDITOR.dialog._.currentTop == null )
		{
			CKEDITOR.dialog._.currentTop = this;
			this._.parentDialog = null;
			CKEDITOR.dialog._.addCover( this._.editor );

			CKEDITOR.document.on( 'keydown', CKEDITOR.dialog._.accessKeyDownHandler );
			CKEDITOR.document.on( 'keyup', CKEDITOR.dialog._.accessKeyUpHandler );
		}
		else
		{
			this._.parentDialog = CKEDITOR.dialog._.currentTop;
			var parentElement = this._.parentDialog.getElement().getFirst();
			parentElement.$.style.zIndex  -= Math.floor( this._.editor.config.basePopupZIndex / 2 );
			CKEDITOR.dialog._.currentTop = this;
		}

		// Save editor selection and grab the focus.
		if ( !this._.parentDialog )
			this.saveSelection();
		this._.dummyText.focus();
		this._.dummyText.$.select();
		
		// Execute onLoad for the first show.
		this.fireOnce( 'load', {} );
	},

	/**
	 * Hides the dialog box.
	 * @example
	 * dialogObj.hide();
	 */
	hide : function()
	{
		// Remove the dialog's element from the root document.
		var element = this._.element;
		if ( !element.getParent() )
			return;
		element.remove();

		// Unregister all access keys associated with this dialog.
		CKEDITOR.dialog._.unregisterAccessKey( this );

		// Maintain dialog ordering and remove cover if needed.
		if ( this._.parentDialog == null )
			CKEDITOR.dialog._.removeCover();
		else
		{
			var parentElement = this._.parentDialog.getElement().getFirst();
			parentElement.setStyle( 'z-index', parseInt( parentElement.$.style.zIndex ) + Math.floor( this._.editor.config.basePopupZIndex / 2 ) );
		}
		CKEDITOR.dialog._.currentTop = this._.parentDialog;

		// Deduct or clear the z-index.
		if ( this._.parentDialog == null )
		{
			CKEDITOR.dialog._.currentZIndex = null;

			// Remove access key handlers.
			CKEDITOR.document.removeListener( 'keydown', CKEDITOR.dialog._.accessKeyDownHandler );
			CKEDITOR.document.removeListener( 'keyup', CKEDITOR.dialog._.accessKeyUpHandler );

			// Restore focus and (if not already restored) selection in the editing area.
			this.restoreSelection();
			this._.editor.focus();
		}
		else
			CKEDITOR.dialog._.currentZIndex -= 10;

	},

	/**
	 * Adds a tabbed page into the dialog.
	 * @param {Object} contents Content definition.
	 * @example
	 */
	addPage : function( contents )
	{
		var pageHtml = [],
			page,
			tab,
			titleHtml = contents.title ? 'title="' + CKEDITOR.tools.htmlEncode( contents.title ) + '" ' : '',
			elements = contents.elements,
			vbox = CKEDITOR.dialog._.uiElementBuilders.vbox.build( this,
					{
						type : 'vbox',
						className : 'cke_dialog_page_contents',
						children : contents.elements,
						expand : !!contents.expand 
					}, pageHtml );

		// Create the HTML for the tab and the content block.
		page = CKEDITOR.dom.element.createFromHtml( pageHtml.join( '' ) );
		tab = CKEDITOR.dom.element.createFromHtml( [
				'<span><div class="cke_dialog_tab" ', titleHtml, '>',
					'<div class="cke_dialog_tab_left"></div>',
					'<div class="cke_dialog_tab_center">',
						CKEDITOR.tools.htmlEncode( contents.label ),
					'</div>',
					'<div class="cke_dialog_tab_right"></div>',
				'</div></span>'
			].join( '' ) );

		// First and last tab styles classes.
		if ( this._.lastTab )
			this._.lastTab.removeClass( 'last' );
		tab.addClass( this._.pageCount > 0 ? 'last' : 'first' );

		// If only a single page exist, a different style is used in the central pane.
		if ( this._.pageCount == 0 )
			this.parts.c.addClass( 'single_page' );
		else
			this.parts.c.removeClass( 'single_page' );

		// Take records for the tabs and elements created.
		this._.tabs[ contents.id ] = [ tab, page ];
		this._.pageCount++;
		this._.lastTab = tab;
		var contentMap = this._.contents[ contents.id ] = {},
			cursor,
			children = vbox.getChild();
		while ( ( cursor = children.shift() ) )
		{
			contentMap[ cursor.id ] = cursor;
			if ( typeof( cursor.getChild ) == 'function' )
				children.push.apply( children, cursor.getChild() );
		}

		// Attach the DOM nodes.
		tab.getFirst().unselectable();
		page.appendTo( this.parts.contents );
		tab.appendTo( this.parts.tabs );
		tab.setAttribute( 'id', contents.id + '_' + CKEDITOR.tools.getNextNumber() );
		page.setAttribute( 'name', contents.id );

		// Add access key handlers if access key is defined.
		if ( contents.accessKey )
		{
			CKEDITOR.dialog._.registerAccessKey( this, this, contents.accessKey,
				CKEDITOR.dialog._.tabAccessKeyDown, CKEDITOR.dialog._.tabAccessKeyUp );
			this._.accessKeyMap[ contents.accessKey ] = contents.id;
		}
	},

	/**
	 * Activates a tab page in the dialog by its id.
	 * @param {String} id The id of the dialog tab to be activated.
	 * @example
	 * dialogObj.selectPage( 'tab_1' );
	 */
	selectPage : function( id )
	{
		var bridge = this.parts.bridge;

		// Hide the non-selected tabs and pages.
		for ( var i in this._.tabs )
		{
			var tab = this._.tabs[i][0],
				page = this._.tabs[i][1];
			if ( i != id )
			{
				tab.removeClass( 'cke_dialog_tab_selected' );
				page.hide();
			}
		}

		var selected = this._.tabs[id];
		selected[0].addClass( 'cke_dialog_tab_selected' );
		selected[1].show();
		var me = this;

		// Place the bridge just under the selected tab. Count 1px left and right
		// of the tab as border space.
		var placeBridge = function()
		{
			var selectedElement = selected[0].getFirst().$;

			// Gecko and Opera have a strange bug that cause the bridge to be
			// added to the wrong layout element initially.
			if ( !selectedElement.offsetParent || selectedElement.offsetParent.nodeName.toLowerCase() == 'body' )
			{
				setTimeout( placeBridge, 50 );
				return;
			}
			bridge.setStyles(
					{
						left : me._.editor.lang.dir == 'ltr' ? ( selectedElement.offsetLeft + 17 || 0 ) + 'px' : 'auto',
						right : me._.editor.lang.dir == 'rtl' ?
							( selectedElement.offsetParent.offsetWidth - selectedElement.offsetLeft - selectedElement.offsetWidth + 17 || 0 ) + 'px' :
							'auto',
						width : ( selected[0].getFirst().$.offsetWidth - 2 || 0 ) + 'px'
					});
		};
		placeBridge();
	},

	/**
	 * Gets the root DOM element of the dialog.
	 * @returns {CKEDITOR.dom.element} The &lt;span&gt; element containing this dialog.
	 * @example
	 * var dialogElement = dialogObj.getElement().getFirst();
	 * dialogElement.setStyle( 'padding', '5px' );
	 */
	getElement : function()
	{
		return this._.element;
	},

	/**
	 * Gets a dialog UI element object from a dialog page.
	 * @param {String} pageId id of dialog page.
	 * @param {String} elementId id of UI element.
	 * @example
	 * @returns {CKEDITOR.ui.dialog.uiElement} The dialog UI element.
	 */
	getContentElement : function( pageId, elementId )
	{
		return this._.contents[pageId][elementId];
	},

	/**
	 * Gets the value of a dialog UI element.
	 * @param {String} pageId id of dialog page.
	 * @param {String} elementId id of UI element.
	 * @example
	 * @returns {Object} The value of the UI element.
	 */
	getValueOf : function( pageId, elementId )
	{
		return this.getContentElement( pageId, elementId ).getValue();
	},

	/**
	 * Sets the value of a dialog UI element.
	 * @param {String} pageId id of the dialog page.
	 * @param {String} elementId id of the UI element.
	 * @param {Object} value The new value of the UI element.
	 * @example
	 */
	setValueOf : function( pageId, elementId, value )
	{
		return this.getContentElement( pageId, elementId ).setValue( value );
	},

	/**
	 * Gets the UI element of a button in the dialog's button row.
	 * @param {String} id The id of the button.
	 * @example
	 * @returns {CKEDITOR.ui.dialog.button} The button object.
	 */
	getButton : function( id )
	{
		return this._.buttons[ id ];
	},

	/**
	 * Simulates a click to a dialog button in the dialog's button row.
	 * @param {String} id The id of the button.
	 * @example
	 * @returns The return value of the dialog's "click" event.
	 */
	click : function( id )
	{
		return this._.buttons[ id ].click();
	},

	/**
	 * Disables a dialog button.
	 * @param {String} id The id of the button.
	 * @example
	 */
	disableButton : function( id )
	{
		return this._.buttons[ id ].disable();
	},

	/**
	 * Enables a dialog button.
	 * @param {String} id The id of the button.
	 * @example
	 */
	enableButton : function( id )
	{
		return this._.buttons[ id ].enable();
	},

	/**
	 * Gets the number of pages in the dialog.
	 * @returns {Number} Page count.
	 */
	getPageCount : function()
	{
		return this._.pageCount;
	},

	saveSelection : function()
	{
		if ( this._.editor.mode )
		{
			var selection = new CKEDITOR.dom.selection( this._.editor.document );
			this._.selectedRanges = selection.getRanges();
		}
	},

	restoreSelection : function()
	{
		if ( this._.editor.mode && this._.selectedRanges )
		{
			( new CKEDITOR.dom.selection( this._.editor.document ) ).selectRanges( this._.selectedRanges );
			this._.selectedRanges = null;
		}
	}
};

CKEDITOR.tools.extend( CKEDITOR.dialog,
	/**
	 * @lends CKEDITOR.dialog
	 */
	{
		/**
		 * Registers a dialog.
		 * @param {String} name The dialog's name.
		 * @param {Function|String} dialogDefinition
		 * A function returning the dialog's definition, or the URL to the .js file holding the function.
		 * The function should accept an argument "editor" which is the current editor instance, and
		 * return an object conforming to {@link CKEDITOR.dialog.dialogDefinition}.
		 * @example
		 * @see CKEDITOR.dialog.dialogDefinition
		 */
		add : function( name, dialogDefinition )
		{
			this._.dialogDefinitions[name] = dialogDefinition;
		},

		/**
		 * The default OK button for dialogs. Fires the "ok" event and closes the dialog if the event succeeds.
		 * @static
		 * @field
		 * @example
		 * @type CKEDITOR.ui.dialog.button
		 */
		okButton : 
		{
			id : 'ok',
			type : 'button',
			label : 'OK',
			style : 'width: 60px',
			onClick : function( evt )
			{
				var dialog = evt.data.dialog;
				if ( dialog.fire( 'ok', { hide : true } ).hide !== false )
					dialog.hide();
			}
		},

		/**
		 * The default cancel button for dialogs. Fires the "cancel" event and closes the dialog if no UI element value changed.
		 * @static
		 * @field
		 * @example
		 * @type CKEDITOR.ui.dialog.button
		 */
		cancelButton :
		{
			id : 'cancel',
			type : 'button',
			label : 'Cancel',
			style : 'width: 60px',
			onClick : function( evt )
			{
				var dialog = evt.data.dialog;
				if ( dialog.fire( 'cancel', { hide : true } ).hide !== false )
					dialog.hide();
			}
		},

		/**
		 * Registers a dialog UI element.
		 * @param {String} typeName The name of the UI element.
		 * @param {Function} builder The function to build the UI element.
		 * @example
		 */
		addUIElement : function( typeName, builder )
		{
			this._.uiElementBuilders[typeName] = builder;
		},

		/**
		 * Sets the width of margins of dialogs, which is used for the dialog moving and resizing logic.
		 * The margin here means the area between the dialog's container <div> and the visual boundary of the dialog.
		 * Typically this area is used for dialog shadows.
		 * @param {Number} top The top margin in pixels.
		 * @param {Number} right The right margin in pixels.
		 * @param {Number} bottom The bottom margin in pixels.
		 * @param {Number} left The left margin in pixels.
		 * @example
		 */
		setMargins : function( top, right, bottom, left )
		{
			this._.margins = [ top, right, bottom, left ];
		}
	});

CKEDITOR.dialog._ = 
{
	defaultDialogDefinition : function( editor )
	{
		return {
			resizable : CKEDITOR.DIALOG_RESIZE_NONE,
			minWidth : 600,
			minHeight : 400,
			buttons : [ CKEDITOR.dialog.okButton, CKEDITOR.dialog.cancelButton ]
		}
	},

	uiElementBuilders : {},

	dialogDefinitions : 
	{
		testOnly : '_source/dialogs/testOnly/dialog.js'
	},

	addCover : function( editor )
	{
		var html = [
				'<div style="position: ', ( CKEDITOR.env.ie && CKEDITOR.env.version < 7 ? 'absolute' : 'fixed' ),
				'; z-index: ', editor.config.basePopupZIndex,
				'; top: 0px; left: 0px; ',
				'background-color: ', editor.config.backgroundCoverColor,
				'" id="cke_dialog_background_cover">'
			],
			win = CKEDITOR.document.getWindow();
		if ( CKEDITOR.env.ie && CKEDITOR.env.version < 7 )
		{
			html.push( '<iframe hidefocus="true" frameborder="0" name="cke_dialog_background_iframe" src="javascript: \'\'" ',
					'style="position: absolute; left: 0px; top: 0px; width: 100%; height: 100%; ',
					'progid:DXImageTransform.Microsoft.Alpha(opacity=0)" ></iframe>' );
		}
		html.push( '</div>' );
		var element = CKEDITOR.dom.element.createFromHtml( html.join( '' ) ),
			resizeFunc = function()
			{
				var size = win.getViewPaneSize();
				element.setStyles( 
						{
							width : size.width + 'px',
							height : size.height + 'px'
						});
			},
			scrollFunc = function()
			{
				var pos = win.getScrollPosition(),
					cursor = CKEDITOR.dialog._.currentTop;
				element.setStyles(
						{
							left : pos.x + 'px',
							top : pos.y + 'px'
						});

				do
				{
					var dialogPos = cursor.getPosition();
					cursor.move( dialogPos.x, dialogPos.y );
				} while( cursor = cursor._.parentDialog );
			};
		CKEDITOR.dialog._.resizeCover = resizeFunc;
		win.on( 'resize', resizeFunc );
		resizeFunc();
		if ( CKEDITOR.env.ie && CKEDITOR.env.version < 7 )
		{
			// IE BUG: win.$.onscroll assignment doesn't work.. it must be window.onscroll.
			// So we need to invent a really funny way to make it work.
			var myScrollHandler = function()
				{
					scrollFunc();
					arguments.callee.prevScrollHandler.apply( this, arguments );
				};
			win.$.setTimeout( function()
				{
					myScrollHandler.prevScrollHandler = window.onscroll || function(){};
					window.onscroll = myScrollHandler;
				}, 0 );
			scrollFunc();
		}
		element.setOpacity( editor.config.backgroundCoverOpacity );
		element.appendTo( CKEDITOR.document.getBody() );
	},

	removeCover : function()
	{
		var element = CKEDITOR.document.getById( 'cke_dialog_background_cover' ),
			win = CKEDITOR.document.getWindow();
		if ( element )
		{
			element.remove();
			win.removeListener( 'resize', CKEDITOR.dialog._.resizeCover );

			if ( CKEDITOR.env.ie && CKEDITOR.env.version < 7 )
			{
				win.$.setTimeout( function()
					{
						var prevScrollHandler = window.onscroll && window.onscroll.prevScrollHandler;
						window.onscroll = prevScrollHandler || null;
					}, 0 );
			}
			CKEDITOR.dialog._.resizeCover = null;
		}
	},

	initDragAndDrop : function( dialog )
	{
		var lastCoords = null,
			abstractDialogCoords = null,
			element = dialog.getElement().getFirst(),
			magnetDistance = dialog._.editor.config.magnetDistance,
			mouseMoveHandler = function( evt )
			{
				var dialogSize = dialog.getSize(),
					viewPaneSize = CKEDITOR.document.getWindow().getViewPaneSize(),
					x = evt.data.$.screenX,
					y = evt.data.$.screenY,
					dx = x - lastCoords.x,
					dy = y - lastCoords.y,
					realX, realY;

				lastCoords = { x : x, y : y };
				abstractDialogCoords.x += dx;
				abstractDialogCoords.y += dy;

				if ( abstractDialogCoords.x + CKEDITOR.dialog._.margins[3] < magnetDistance )
					realX = - CKEDITOR.dialog._.margins[3];
				else if ( abstractDialogCoords.x - CKEDITOR.dialog._.margins[1] > viewPaneSize.width - dialogSize.width - magnetDistance )
					realX = viewPaneSize.width - dialogSize.width + CKEDITOR.dialog._.margins[1];
				else
					realX = abstractDialogCoords.x;

				if ( abstractDialogCoords.y + CKEDITOR.dialog._.margins[0] < magnetDistance )
					realY = - CKEDITOR.dialog._.margins[0];
				else if ( abstractDialogCoords.y - CKEDITOR.dialog._.margins[2] > viewPaneSize.height - dialogSize.height - magnetDistance )
					realY = viewPaneSize.height - dialogSize.height + CKEDITOR.dialog._.margins[2];
				else
					realY = abstractDialogCoords.y;

				dialog.move( realX, realY );

				evt.data.preventDefault();
			},
			mouseUpHandler = function( evt )
			{
				CKEDITOR.document.removeListener( 'mousemove', mouseMoveHandler );
				CKEDITOR.document.removeListener( 'mouseup', mouseUpHandler );

				if ( CKEDITOR.env.ie && CKEDITOR.env.version < 7 )
				{
					var coverDoc = new CKEDITOR.dom.document( frames( 'cke_dialog_background_iframe' ).document );
					coverDoc.removeListener( 'mousemove', mouseMoveHandler );
					coverDoc.removeListener( 'mouseup', mouseUpHandler );
				}
			};

		dialog.parts.title.on( 'mousedown', function( evt )
				{
					lastCoords = { x : evt.data.$.screenX, y : evt.data.$.screenY };

					CKEDITOR.document.on( 'mousemove', mouseMoveHandler );
					CKEDITOR.document.on( 'mouseup', mouseUpHandler );
					abstractDialogCoords = dialog.getPosition();

					if ( CKEDITOR.env.ie && CKEDITOR.env.version < 7 )
					{
						var coverDoc = new CKEDITOR.dom.document( frames( 'cke_dialog_background_iframe' ).document );
						coverDoc.on( 'mousemove', mouseMoveHandler );
						coverDoc.on( 'mouseup', mouseUpHandler );
					}

					evt.data.preventDefault();
				}, dialog );
	},

	initResizeHandles : function( dialog )
	{
		var definition = dialog._.definition,
			minWidth = definition.minWidth || 0,
			minHeight = definition.minHeight || 0,
			resizable = definition.resizable,
			topSizer = function( coords, dy )
			{
				coords.y += dy;
			},
			rightSizer = function( coords, dx )
			{
				coords.x2 += dx;
			},
			bottomSizer = function( coords, dy )
			{
				coords.y2 += dy;
			},
			leftSizer = function( coords, dx )
			{
				coords.x += dx;
			},
			lastCoords = null,
			abstractDialogCoords = null,
			magnetDistance = dialog._.editor.config.magnetDistance,
			parts = [ 'tl', 't', 'tr', 'l', 'r', 'bl', 'b', 'br' ],
			mouseDownHandler = function( evt )
			{
				var partName = evt.listenerData.part, size = dialog.getSize();
				abstractDialogCoords = dialog.getPosition();
				CKEDITOR.tools.extend( abstractDialogCoords, 
					{
						x2 : abstractDialogCoords.x + size.width,
						y2 : abstractDialogCoords.y + size.height
					} );
				lastCoords = { x : evt.data.$.screenX, y : evt.data.$.screenY };

				CKEDITOR.document.on( 'mousemove', mouseMoveHandler, dialog, { part : partName } );
				CKEDITOR.document.on( 'mouseup', mouseUpHandler, dialog, { part : partName } );

				if ( CKEDITOR.env.ie && CKEDITOR.env.version < 7 )
				{
					var coverDoc = new CKEDITOR.dom.document( frames( 'cke_dialog_background_iframe' ).document );
					coverDoc.on( 'mousemove', mouseMoveHandler, dialog, { part : partName } );
					coverDoc.on( 'mouseup', mouseUpHandler, dialog, { part : partName } );
				}

				evt.data.preventDefault();
			},
			mouseMoveHandler = function( evt )
			{
				var x = evt.data.$.screenX,
					y = evt.data.$.screenY,
					dx = x - lastCoords.x,
					dy = y - lastCoords.y,
					viewPaneSize = CKEDITOR.document.getWindow().getViewPaneSize(),
					partName = evt.listenerData.part;

				if ( partName.search( 't' ) != -1 )
					topSizer( abstractDialogCoords, dy );
				if ( partName.search( 'l' ) != -1 )
					leftSizer( abstractDialogCoords, dx );
				if ( partName.search( 'b' ) != -1 )
					bottomSizer( abstractDialogCoords, dy );
				if ( partName.search( 'r' ) != -1 )
					rightSizer( abstractDialogCoords, dx );

				lastCoords = { x : x, y : y };

				var realX, realY, realX2, realY2;

				if ( abstractDialogCoords.x + CKEDITOR.dialog._.margins[3] < magnetDistance )
					realX = - CKEDITOR.dialog._.margins[3];
				else if ( partName.search( 'l' ) != -1 && abstractDialogCoords.x2 - abstractDialogCoords.x < minWidth + magnetDistance )
					realX = abstractDialogCoords.x2 - minWidth;
				else
					realX = abstractDialogCoords.x;

				if ( abstractDialogCoords.y + CKEDITOR.dialog._.margins[0] < magnetDistance )
					realY = - CKEDITOR.dialog._.margins[0];
				else if ( partName.search( 't' ) != -1 && abstractDialogCoords.y2 - abstractDialogCoords.y < minHeight + magnetDistance )
					realY = abstractDialogCoords.y2 - minHeight;
				else
					realY = abstractDialogCoords.y;

				if ( abstractDialogCoords.x2 - CKEDITOR.dialog._.margins[1] > viewPaneSize.width - magnetDistance )
					realX2 = viewPaneSize.width + CKEDITOR.dialog._.margins[1] ;
				else if ( partName.search( 'r' ) != -1 && abstractDialogCoords.x2 - abstractDialogCoords.x < minWidth + magnetDistance )
					realX2 = abstractDialogCoords.x + minWidth;
				else
					realX2 = abstractDialogCoords.x2;

				if ( abstractDialogCoords.y2 - CKEDITOR.dialog._.margins[2] > viewPaneSize.height - magnetDistance )
					realY2= viewPaneSize.height + CKEDITOR.dialog._.margins[2] ;
				else if ( partName.search( 'b' ) != -1 && abstractDialogCoords.y2 - abstractDialogCoords.y < minHeight + magnetDistance )
					realY2 = abstractDialogCoords.y + minHeight;
				else
					realY2 = abstractDialogCoords.y2 ;

				dialog.move( realX, realY );
				dialog.resize( realX2 - realX, realY2 - realY );

				evt.data.preventDefault();
			},
			mouseUpHandler = function( evt )
			{
				CKEDITOR.document.removeListener( 'mouseup', mouseUpHandler );
				CKEDITOR.document.removeListener( 'mousemove', mouseMoveHandler );

				if ( CKEDITOR.env.ie && CKEDITOR.env.version < 7 )
				{
					var coverDoc = new CKEDITOR.dom.document( frames( 'cke_dialog_background_iframe' ).document );
					coverDoc.removeListener( 'mouseup', mouseUpHandler );
					coverDoc.removeListener( 'mousemove', mouseMoveHandler );
				}
			};

		var widthTest = /[lr]/,
			heightTest = /[tb]/;
		for ( var i = 0 ; i < parts.length ; i++ )
		{
			var element = dialog.parts[ parts[i] + '_resize' ];
			if ( resizable == CKEDITOR.DIALOG_RESIZE_NONE || 
					resizable == CKEDITOR.DIALOG_RESIZE_HEIGHT && widthTest.test( parts[i] ) ||
			  		resizable == CKEDITOR.DIALOG_RESIZE_WIDTH && heightTest.test( parts[i] ) )
			{
				element.hide();
				continue;
			}
			element.on( 'mousedown', mouseDownHandler, dialog, { part : parts[i] } );
		}
	},

	currentTop : null,

	currentZIndex : null,
	
	storedDialogs : {},

	margins : [0, 0, 0, 0],

	/**
	 * This class is not really part of the API. It is the "definition" property value
	 * passed to "dialogDefinition" event handlers.
	 * @constructor
	 * @name CKEDITOR.dialog.dialogDefinitionObject
	 * @extends CKEDITOR.dialog.dialogDefinition
	 * @example
	 * CKEDITOR.on( 'dialogDefinition', function( evt )
	 * 	{
	 * 		var definition = evt.data.definition;
	 * 		var content = definition.getContents( 'page1' );
	 * 		...
	 * 	} );
	 */
	definitionObject : function( dialog, dialogDefinition )
	{
		if ( arguments.length < 2 )
			return;
		( this._ || ( this._ = {} ) ).dialog = dialog;
		CKEDITOR.tools.extend( this, dialogDefinition );

		var cursor, i;
		for ( i = 0 ; i < dialogDefinition.contents.length && ( cursor = dialogDefinition.contents[i] ) ; i++ )
			dialogDefinition.contents[i] = new CKEDITOR.dialog._.contentObject( dialog, cursor.id, cursor );
	},

	/**
	 * This class is not really part of the API. It is the template of the objects
	 * representing content pages inside the CKEDITOR.dialog.dialogDefinitionObject.
	 * @constructor
	 * @name CKEDITOR.dialog.contentDefinitionObject
	 * @example
	 * CKEDITOR.on( 'dialogDefinition', function( evt )
	 * 	{
	 * 		var definition = evt.data.definition;
	 * 		var content = definition.getContents( 'page1' );
	 *		content.remove( 'textInput1' );
	 * 		...
	 * 	} );
	 */
	contentObject : function( dialog, id, contentDefinition )
	{
		if ( arguments.length < 3 )
			return;
		CKEDITOR.tools.extend( ( this._ || ( this._ = {} ) ),
				{
					id : id,
					dialog : dialog
				});
		CKEDITOR.tools.extend( this, contentDefinition );
	},

	accessKeyDownHandler : function( evt )
	{
		var ctrl = evt.data.$.ctrlKey || evt.data.$.metaKey,
			key = String.fromCharCode( evt.data.$.keyCode ),
			keyProcessor = CKEDITOR.dialog._.accessKeyProcessors[key];

		if ( !ctrl || !keyProcessor || !keyProcessor.length )
			return;

		keyProcessor = keyProcessor[keyProcessor.length - 1];
		keyProcessor.keydown.call( keyProcessor.uiElement, keyProcessor.dialog, keyProcessor.key );
		evt.data.preventDefault();
	},

	accessKeyUpHandler : function( evt )
	{
		var ctrl = evt.data.$.ctrlKey || evt.data.$.metaKey,
			key = String.fromCharCode( evt.data.$.keyCode ),
			keyProcessor = CKEDITOR.dialog._.accessKeyProcessors[key];

		if ( !ctrl || !keyProcessor || !keyProcessor.length )
			return;

		keyProcessor = keyProcessor[keyProcessor.length - 1];
		keyProcessor.keyup.call( keyProcessor.uiElement, keyProcessor.dialog, keyProcessor.key );
		evt.data.preventDefault();
	},

	registerAccessKey : function( uiElement, dialog, key, downFunc, upFunc )
	{
		var procList = this.accessKeyProcessors[key] || ( this.accessKeyProcessors[key] = [] );
		procList.push( {
				uiElement : uiElement,
				dialog : dialog,
				key : key,
				keyup : upFunc || uiElement.accessKeyUp,
				keydown : downFunc || uiElement.accessKeyDown
			} );
	},

	unregisterAccessKey : function( obj )
	{
		for ( var i in this.accessKeyProcessors )
		{
			var list = this.accessKeyProcessors[i];
			for ( var j = list.length - 1 ; j >= 0 ; j-- )
			{
				if ( list[j].dialog == obj || list[j].uiElement == obj )
					list.splice( j, 1 );
			}
			if ( list.length == 0 )
				delete this.accessKeyProcessors[i];
		}
	}, 

	accessKeyProcessors : {},

	tabAccessKeyUp : function( dialog, key )
	{
		if ( dialog._.accessKeyMap[key] )
			dialog.selectPage( dialog._.accessKeyMap[key] );
	},

	tabAccessKeyDown : function( dialog, key )
	{
	}
};

CKEDITOR.event.implementOn( CKEDITOR.dialog );

CKEDITOR.dialog._.definitionObject.prototype = 
/** @lends CKEDITOR.dialog.dialogDefinitionObject.prototype */
{
	/**
	 * Gets a content definition in the dialog definition.
	 * @param {String} id The id of the content definition.
	 * @returns {CKEDITOR.dialog.contentDefinition} The content definition with id.
	 * @example
	 */
	getContents : function( id )
	{
		for ( var i = 0, cursor ; i < this.contents.length && ( cursor = this.contents[i] ) ; i++ )
		{
			if ( cursor.id == id )
				return cursor;
		}
		return null;
	},

	/**
	 * Gets a button definition in the dialog definition.
	 * @param {String} id The id of the button definition.
	 * @returns {CKEDITOR.dialog.buttonDefinition} The button definition with id.
	 * @example
	 */
	getButton : function( id )
	{
		for ( var i = 0, cursor ; i < this.buttons.length && ( cursor = this.buttons[i] ) ; i++ )
		{
			if ( cursor.id == id )
				return cursor;
		}
		return null;
	},

	/**
	 * Adds a content definition object under the dialog definition.
	 * @param {CKEDITOR.dialog.contentDefinition} The content definition.
	 * @param {String} nextSiblingId The id of an existing content definition which the new content definition
	 * will be inserted before. Omit if the new content definition is to be inserted as the last item.
	 * @returns {CKEDITOR.dialog.contentDefinition} The inserted content definition.
	 * @example
	 */
	addContents : function( contentDefinition, nextSiblingId )
	{
		for ( var i = 0, cursor ; i < this.contents.length && ( cursor = this.contents[i] ) ; i++ )
		{
			if ( cursor.id == nextSiblingId )
			{
				this.contents.splice( i, 0, contentDefinition );
				return contentDefinition;
			}
		}
		this.contents.push( contentDefinition );
		return contentDefinition;
	},

	/**
	 * Adds a button definition object under the dialog definition.
	 * @param {CKEDITOR.dialog.buttonDefinition} The button definition.
	 * @param {String} nextSiblingId The id of an existing button definition which the new button definition
	 * will be inserted before. Omit if the new button definition is to be inserted as the last item.
	 * @returns {CKEDITOR.dialog.buttonDefinition} The inserted button definition.
	 * @example
	 */
	addButton : function( buttonDefinition, nextSiblingId )
	{
		for ( var i = 0, cursor ; i < this.buttons.length && ( cursor = this.buttons[i] ) ; i++ )
		{
			if ( cursor.id == nextSiblingId )
			{
				this.buttons.splice( i, 0, buttonDefinition );
				return buttonDefinition;
			}
		}
		this.buttons.push( buttonDefinition );
		return buttonDefinition;
	},

	/**
	 * Removes a content definition from the dialog definition.
	 * @param {String} id The id of the content definition to be removed.
	 * @returns {CKEDITOR.dialog.contentDefinition} The removed content definition.
	 * @example
	 */
	removeContents : function( id )
	{
		for ( var i = 0; i < this.contents.length ; i++ )
		{
			if ( this.contents[i].id == id )
				return this.contents.splice( i, 1 );
		}
		return null;
	},

	/**
	 * Removes a button definition from the dialog definition.
	 * @param {String} id The id of the button definition to be removed.
	 * @returns {CKEDITOR.dialog.buttonDefinition} The removed button definition.
	 * @example
	 */
	removeButton : function( id )
	{
		for ( var i = 0 ; i < this.buttons.length ; i++ )
		{
			if ( this.buttons[i].id == id )
				return this.buttons.splice( i, 1 );
		}
		return null;
	}
};

CKEDITOR.dialog._.contentObject.prototype = 
/** @lends CKEDITOR.dialog.contentDefinitionObject.prototype */
{
	/**
	 * Gets a UI element definition under the content definition.
	 * @param {String} id The id of the UI element definition.
	 * @returns {CKEDITOR.dialog.uiElementDefinition} 
	 * @example
	 */
	get : function( id )
	{
		for ( var i = 0, cursor ; i < this.elements.length && ( cursor = this.elements[i] ) ; i++ )
		{
			if ( cursor.id == id )
				return cursor;
		}
		return null;
	},

	/**
	 * Adds a UI element definition to the content definition.
	 * @param {CKEDITOR.dialog.uiElementDefinition} elementDefinition The UI elemnet definition
	 * to be added.
	 * @param {String} nextSiblingId The id of an existing UI element definition which the new 
	 * UI element definition will be inserted before. Omit if the new button definition is to 
	 * be inserted as the last item.
	 * @returns {CKEDITOR.dialog.uiElementDefinition} The element definition inserted.
	 * @example
	 */
	add : function( elementDefinition, nextSiblingId )
	{
		for ( var i = 0, cursor ; i < this.elements.length && ( cursor = this.elements[i] ) ; i++ )
		{
			if ( cursor.id == nextSiblingId )
			{
				this.elements.splice( i, 0, elementDefinition );
				return elementDefinition;
			}
		}
		this.elements.push( elementDefinition );
		return elementDefinition;
	},

	/**
	 * Removes a UI element definition from the content definition.
	 * @param {String} id The id of the UI element definition to be removed.
	 * @returns {CKEDITOR.dialog.uiElementDefinition} The element definition removed.
	 * @example
	 */
	remove : function( id )
	{
		for ( var i = 0 ; i < this.elements.length ; i++ )
		{
			if ( this.elements[i].id == id )
				return this.elements.splice( i, 1 );
		}
		return null;
	}
};

(function()
{
	var decimalRegex = /^\d+(?:\.\d+)?$/,
		fixLength = function( length )
		{
			return length + ( decimalRegex.test( length ) ? 'px' : '' );
		}

	CKEDITOR.ui.dialog = 
	{
		/**
		 * The base class of all dialog UI elements.
		 * @constructor
		 * @example
		 */
		uiElement : function( dialog, elementDefinition, htmlList, nodeNameArg, stylesArg, attributesArg, contentsArg )
		{
			if (arguments.length < 4 )
				return;

			var nodeName = ( nodeNameArg.call ? nodeNameArg( elementDefinition ) : nodeNameArg ) || 'div',
				html = [ '<', nodeName, ' ' ],
				styles = ( stylesArg && stylesArg.call ? stylesArg( elementDefinition ) : stylesArg ) || {},
				attributes = ( attributesArg && attributesArg.call ? attributesArg( elementDefinition ) : attributesArg ) || {},
				innerHTML = ( contentsArg && contentsArg.call ? contentsArg( dialog, elementDefinition ) : contentsArg ) || '',
				domId = this.domId = attributes.id || CKEDITOR.tools.getNextNumber() + '_uiElement',
				id = this.id = elementDefinition.id,
				i;

			// Set the id, a unique id is required for getElement() to work.
			attributes.id = domId;

			// Set the type and definition CSS class names.
			var classes = {};
			if ( elementDefinition.type )
				classes[ 'cke_dialog_ui_' + elementDefinition.type ] = 1;
			if ( elementDefinition.className )
				classes[ elementDefinition.className ] = 1;
			var attributeClasses = ( attributes['class'] && attributes['class'].split ) ? attributes['class'].split( ' ' ) : [];
			for ( i = 0 ; i < attributeClasses.length ; i++ )
			{
				if ( attributeClasses[i] )
					classes[ attributeClasses[i] ] = 1;
			}
			var finalClasses = [];
			for ( i in classes )
				finalClasses.push( i );
			attributes['class'] = finalClasses.join( ' ' );

			// Set the popup tooltop.
			if ( elementDefinition.title )
				attributes.title = elementDefinition.title;

			// Write the inline CSS styles.
			var styleStr = ( elementDefinition.style || '' ).split( ';' );
			for ( i in styles )
				styleStr.push( i + ':' + styles[i] );
			for ( i = styleStr.length - 1 ; i >= 0 ; i-- )
			{
				if ( styleStr[i] === '' )
					styleStr.splice( i, 1 );
			}
			if ( styleStr.length > 0 )
				attributes.style = ( attributes.style || '' ) + styleStr.join( '; ' );

			// Write the attributes.
			for ( i in attributes )
				html.push( i + '="' + CKEDITOR.tools.htmlEncode( attributes[i] ) + '" ');

			// Write the content HTML.
			html.push( '>', innerHTML, '</', nodeName, '>' );

			// Add contents to the parent HTML array.
			htmlList.push( html.join( '' ) );

			( this._ || ( this._ = {} ) ).dialog = dialog;
			
			// Add events.
			this.registerEvents( elementDefinition );
			if ( this.accessKeyUp && this.accessKeyDown && elementDefinition.accessKey )
				CKEDITOR.dialog._.registerAccessKey( this, dialog, elementDefinition.accessKey );
		},

		/**
		 * Horizontal layout box for dialog UI elements, auto-expends to available width of container.
		 * @constructor
		 * @extends CKEDITOR.ui.dialog.uiElement
		 * @example
		 */
		hbox : function( dialog, childObjList, childHtmlList, htmlList, elementDefinition )
		{
			if ( arguments.length < 4 )
				return;

			this._ || ( this._ = {} );

			var children = this._.children = childObjList,
				widths = elementDefinition && elementDefinition.widths || null,
				height = elementDefinition && elementDefinition.height || null,
				i;
			/** @ignore */
			var innerHTML = function()
			{
				var html = [ '<tbody><tr class="cke_dialog_ui_hbox">' ];
				for ( i = 0 ; i < childHtmlList.length ; i++ )
				{
					var className = 'cke_dialog_ui_hbox_child',
						styles = [];
					if ( i === 0 )
						className = 'cke_dialog_ui_hbox_first';
					if ( i == childHtmlList.length - 1 )
						className = 'cke_dialog_ui_hbox_last';
					html.push( '<td class="', className, '" ' );
					if ( widths )
					{
						if ( widths[i] )
							styles.push( 'width:' + fixLength( widths[i] ) );
					}
					else
						styles.push( 'width:' + Math.floor( 100 / childHtmlList.length ) + '%' );
					if ( height )
						styles.push( 'height:' + fixLength( height ) );
					if ( styles.length > 0 )
						html.push( 'style="' + styles.join('; ') + '" ' );
					html.push( '>', childHtmlList[i], '</td>' );
				}
				html.push( '</tr></tbody>' );
				return html.join( '' );
			};
			CKEDITOR.ui.dialog.uiElement.call( this, dialog, elementDefinition || { type : 'hbox' }, htmlList, 'table', null, null, innerHTML );
		},

		/**
		 * Vertical layout box for dialog UI elements.
		 * @constructor
		 * @extends CKEDITOR.ui.dialog.hbox
		 * @example
		 */
		vbox : function( dialog, childObjList, childHtmlList, htmlList, elementDefinition )
		{
			if (arguments.length < 3 )
				return;

			this._ || ( this._ = {} );

			var children = this._.children = childObjList,
				width = elementDefinition && elementDefinition.width || null,
				heights = elementDefinition && elementDefinition.heights || null;
			/** @ignore */
			var innerHTML = function()
			{
				var html = [ '<table cellspacing="0" border="0" ' ];
				html.push( 'style="' );
				if ( elementDefinition.expand )
					html.push( 'height:100%;' );
				html.push( 'width:' + fixLength( elementDefinition.width || '100%' ), ';' );
				html.push( '"' );

				html.push( '><tbody>' );
				for ( var i = 0 ; i < childHtmlList.length ; i++ )
				{
					var styles = [];
					html.push( '<tr><td ' );
					if ( width )
						styles.push( 'width:' + fixLength( width || '100%' ) );
					if ( heights )
						styles.push( 'height:' + fixLength( heights[i] ) );
					else if ( elementDefinition.expand )
						styles.push( 'height:' + Math.floor( 100 / childHtmlList.length ) + '%' );
					if ( styles.length > 0 )
						html.push( 'style="', styles.join( '; ' ), '" ' );
					html.push( ' class="cke_dialog_ui_vbox_child">', childHtmlList[i], '</td></tr>' );
				}
				html.push( '</tbody></table>' );
				return html.join( '' );
			};
			CKEDITOR.ui.dialog.uiElement.call( this, dialog, elementDefinition || { type : 'vbox' }, htmlList, 'div', null, null, innerHTML );
		}
	};
})();

CKEDITOR.ui.dialog.uiElement.prototype = 
{
	/**
	 * Gets the root DOM element of this dialog UI object.
	 * @returns {CKEDITOR.dom.element} Root DOM element of UI object.
	 * @example
	 * uiElement.getElement().hide();
	 */
	getElement : function()
	{
		return CKEDITOR.document.getById( this.domId );
	},

	/**
	 * Gets the parent dialog object containing this UI element.
	 * @returns {CKEDITOR.dialog} Parent dialog object.
	 * @example
	 * var dialog = uiElement.getDialog();
	 */
	getDialog : function()
	{
		return this._.dialog;
	},

	/**
	 * Sets the value of this dialog UI object.
	 * @param {Object} value The new value.
	 * @example
	 * uiElement.setValue( 'Dingo' );
	 */
	setValue : function( value )
	{
		this.getElement().$.value = value;
	},

	/**
	 * Gets the current value of this dialog UI object.
	 * @returns {Object} The current value.
	 * @example
	 * var myValue = uiElement.getValue();
	 */
	getValue : function()
	{
		return this.getElement().$.value;
	},

	/**
	 * Tells whether the UI object's value has changed.
	 * @returns {Boolean} true if changed, false if not changed.
	 * @example
	 * if ( uiElement.isChanged() )
	 * &nbsp;&nbsp;confirm( 'Value changed! Continue?' );
	 */
	isChanged : function()
	{
		// Override in input classes.
		return false;
	},

	/**
	 * Puts the focus to the UI object. Switches tabs if the UI object isn't in the active tab page.
	 * @example
	 * uiElement.focus();
	 */
	focus : function()
	{
		var element = this.getElement(),
			cursor = element,
			tabId;
		while ( ( cursor = cursor.getParent() ) && cursor.$.className.search( 'cke_dialog_page_contents' ) == -1 );
		tabId = cursor.getAttribute( 'name' );

		this._.dialog.selectPage( tabId );
		element.focus();
	},

	registerEvents : function( definition )
	{
		var regex = /^on([A-Z]\w+)/,
			match,
			registerDomEvent = function( uiElement, dialog, eventName, func )
			{
				dialog.on( 'load', function()
				{
					uiElement.getElement().on( eventName, func, uiElement );
				});
			};

		for ( var i in definition )
		{
			if ( !( match = i.match( regex ) ) )
				continue;
			if ( this.eventProcessors[i] )
				this.eventProcessors[i].call( this, this._.dialog, definition[i] );
			else
				registerDomEvent( this, this._.dialog, match[1].toLowerCase(), definition[i] );
		}
	},

	eventProcessors : 
	{
		onLoad : function( dialog, func )
		{
			dialog.on( 'load', func, this, null, CKEDITOR.tools.getNextNumber() );
		}
	},

	accessKeyDown : function( dialog, key )
	{
		this.focus();
	},

	accessKeyUp : function( dialog, key )
	{
	}
};

CKEDITOR.ui.dialog.hbox.prototype = CKEDITOR.tools.extend( new CKEDITOR.ui.dialog.uiElement,
	/**
	 * @lends CKEDITOR.ui.dialog.hbox.prototype
	 */
	{
		/**
		 * Gets a child UI element inside this container.
		 * @param {Array|Number} indices An array or a single number to indicate the child's 
		 * position in the container's descendant tree. Omit to get all the children in an array.
		 * @returns {Array|CKEDITOR.ui.dialog.uiElement} Array of all UI elements in the container
		 * if no argument given, or the specified UI element if indices is given.
		 * @example
		 * var checkbox = hbox.getChild( [0,1] );
		 * checkbox.setValue( true );
		 */
		getChild : function( indices )
		{
			// If no arguments, return a clone of the children array.
			if ( arguments.length < 1 )
				return this._.children.concat();

			// If indices isn't array, make it one.
			if ( !indices.splice )
				indices = [ indices ];

			// Retrieve the child element according to tree position.
			if ( indices.length < 2 )
				return this._.children[ indices[0] ];
			else
				return ( this._.children[ indices[0] ] && this._.children[ indices[0] ].getChild ) ?
					this._.children[ indices[0] ].getChild( indices.slice( 1, indices.length ) ) :
					null;
		}
	}, true );

CKEDITOR.ui.dialog.vbox.prototype = new CKEDITOR.ui.dialog.hbox();

CKEDITOR.tools.extend( CKEDITOR.editor.prototype,
	/** @lends CKEDITOR.editor.prototype */
	{
		/**
		 * Loads and opens a registered dialog.
		 * @param {String} dialogName The registered name of the dialog.
		 * @see CKEDITOR.dialog.add
		 * @example
		 * CKEDITOR.instances.editor1.openDialog( 'smiley' ); 
		 * @returns {CKEDITOR.dialog} The dialog object corresponding to the dialog displayed. null if the dialog name is not registered.
		 */
		openDialog : function( dialogName )
		{
			var me = this;

			// If the dialogDefinition is already loaded, open it immediately.
			if ( typeof( CKEDITOR.dialog._.dialogDefinitions[dialogName] ) == 'function' )
			{
				var dialog = CKEDITOR.dialog._.storedDialogs[dialogName] || new CKEDITOR.dialog( this, dialogName );
				CKEDITOR.dialog._.storedDialogs[dialogName] = dialog;
				dialog.show();
				return dialog;
			}

			// Not loaded? Load the .js file first.
			var body = CKEDITOR.document.getBody(),
				cursor = body.$.style.cursor;
			body.setStyle( 'cursor', 'wait' );
			CKEDITOR.scriptLoader.load( CKEDITOR.getUrl( CKEDITOR.dialog._.dialogDefinitions[dialogName] ), function()
				{
					me.openDialog( dialogName );
					body.setStyle( 'cursor', cursor );
				} );
		}
	});

(function()
{
	var commonBuilder = {
		build : function( dialog, elementDefinition, output )
		{
			var children = elementDefinition.children,
				child,
				childHtmlList = [],
				childObjList = [];
			for ( var i = 0 ; ( i < children.length && ( child = children[i] ) ) ; i++ )
			{
				var childHtml = [];
				childHtmlList.push( childHtml );
				childObjList.push( CKEDITOR.dialog._.uiElementBuilders[ child.type ].build( dialog, child, childHtml ) );
			}
			return new CKEDITOR.ui.dialog[elementDefinition.type]( dialog, childObjList, childHtmlList, output, elementDefinition );
		}
	};

	CKEDITOR.dialog.addUIElement( 'hbox', commonBuilder );
	CKEDITOR.dialog.addUIElement( 'vbox', commonBuilder );
})();
