/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2004-2009 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.handlers;

import java.text.MessageFormat;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.ResourceBundle;

import javax.servlet.http.HttpServletRequest;

import net.fckeditor.localization.LocaleResolver;
import net.fckeditor.tool.Utils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class wraps to a {@link LocaleResolver} implementation and retrieves
 * localized strings for a given request. This class safely returns default
 * strings if anything fails. Loaded strings will be cached on load.
 * 
 * @version $Id: LocalizedMessages.java 2868 2009-01-09 21:54:36Z mosipov $
 * 
 */
public class LocalizedMessages {

	private static Map<Locale, LocalizedMessages> prototypes = new HashMap<Locale, LocalizedMessages>();
	private Properties properties;
	private static LocaleResolver localeResolver;
	private static final Locale NEUTRAL = new Locale("");
	private static final Logger logger = LoggerFactory
			.getLogger(LocalizedMessages.class);

	/**
	 * This method return an instance of this class for the given locale.
	 * 
	 * @param request
	 *            The current request instance
	 * @return The LP instance with localized strings.
	 */
	public static LocalizedMessages getMessages(HttpServletRequest request) {

		Locale locale;

		if (request == null)
			locale = NEUTRAL;
		else
			locale = getLocaleResolverInstance().resolveLocale(request);

		if (locale == null)
			locale = NEUTRAL;

		LocalizedMessages lp = prototypes.get(locale);
		if (lp == null) {
			lp = new LocalizedMessages(locale);
			prototypes.put(locale, lp);
		}

		// for now we don't need any cloning since the values are accessed
		// read-only
		return lp;

	}

	private static LocaleResolver getLocaleResolverInstance() {

		if (localeResolver == null) {
			String className = PropertiesLoader.getLocaleResolverImpl();

			if (Utils.isEmpty(className))
				logger
						.error("Empty LocaleResolver implementation class name provided");
			else
				try {
					Class<?> clazz = Class.forName(className);
					localeResolver = (LocaleResolver) clazz.newInstance();
					logger.info("LocaleResolver initialized to {}", className);
				} catch (Exception e) {
					logger.error("LocaleResolver implementation {} could not be instantiated", className);
					// TODO should be throw a RuntimeException here
				}
		}

		return localeResolver;
	}

	/**
	 * Loads the default strings and the string for the given locale and only
	 * for this locale.
	 * 
	 * @param locale
	 *            Given locale
	 */
	private LocalizedMessages(Locale locale) {

		ResourceBundle rb = null;
		try {
			rb = ResourceBundle.getBundle("fckeditor", locale, Thread
					.currentThread().getContextClassLoader());
		} catch (Exception e) {
			logger.debug("No resource bundle found for locale {}", locale);
		}

		this.properties = new Properties(PropertiesLoader.getProperties());

		if (rb != null
				&& rb.getLocale().getLanguage().equals(locale.getLanguage())) {
			Enumeration<String> keys = rb.getKeys();

			while (keys.hasMoreElements()) {
				String key = keys.nextElement();
				properties.setProperty(key, rb.getString(key));
			}
		}

	}

	private String getString(String key) {
		return properties.getProperty(key);
	}

	/**
	 * Getter for localized <code>message.editor.compatibleBrowser.yes</code> property
	 * 
	 * @return CompatibleBrowserYes
	 */
	public String getCompatibleBrowserYes() {
		return getString("message.editor.compatibleBrowser.yes");
	}

	/**
	 * Getter for localized <code>message.editor.compatibleBrowser.no</code> property
	 * 
	 * @return CompatibleBrowserNo
	 */
	public String getCompatibleBrowserNo() {
		return getString("message.editor.compatibleBrowser.no");
	}

	/**
	 * Getter for localized <code>message.connector.fileUpload.enabled</code>
	 * property
	 * 
	 * @return FileUploadEnabled
	 */
	public String getFileUploadEnabled() {
		return getString("message.connector.fileUpload.enabled");
	}

	/**
	 * Getter for localized <code>message.connector.fileUpload.disabled</code>
	 * property
	 * 
	 * @return FileUploadDisabled
	 */
	public String getFileUploadDisabled() {
		return getString("message.connector.fileUpload.disabled");
	}
	
	/**
	 * Getter for localized <code>message.connector.file_renamed_warning</code>
	 * property
	 * 
	 * @param newFileName the new filename
	 * @return FileRenamedWarning
	 */
	public String getFileRenamedWarning(String newFileName) {
		return MessageFormat.format(
				getString("message.connector.fileUpload.file_renamed_warning"),
				newFileName);
	}
	
	/**
	 * Getter for localized <code>message.connector.fileUpload.invalid_file_type_specified</code>
	 * property
	 * 
	 * @return InvalidFileTypeSpecified
	 */
	public String getInvalidFileTypeSpecified() {
		return getString("message.connector.fileUpload.invalid_file_type_specified");
	}
	
	/**
	 * Getter for localized
	 * <code>message.connector.fileUpload.write_error</code> property
	 * 
	 * @return FileUploadWiteError
	 */
	public String getFileUploadWriteError() {
		return getString("message.connector.fileUpload.write_error");
	}

	/**
	 * Getter for localized <code>message.connector.getResources.enabled</code>
	 * property
	 * 
	 * @return GetResourcesEnabled
	 */
	public String getGetResourcesEnabled() {
		return getString("message.connector.getResources.enabled");
	}

	/**
	 * Getter for localized <code>message.connector.getResources.disabled</code>
	 * property
	 * 
	 * @return GetResourcesDisabled
	 */
	public String getGetResourcesDisabled() {
		return getString("message.connector.getResources.disabled");
	}
	
	/**
	 * Getter for localized <code>message.connector.getResources.read_error</code>
	 * property
	 * 
	 * @return GetResourcesReadError
	 */
	public String getGetResourcesReadError() {
		return getString("message.connector.getResources.read_error");
	}

	/**
	 * Getter for localized
	 * <code>message.connector.createFolder.enabled</code> property
	 * 
	 * @return FolderCreationEnabled
	 */
	public String getCreateFolderEnabled() {
		return getString("message.connector.createFolder.enabled");
	}

	/**
	 * Getter for localized
	 * <code>message.connector.createFolder.disabled</code> property
	 * 
	 * @return FolderCreationDisabled
	 */
	public String getCreateFolderDisabled() {
		return getString("message.connector.createFolder.disabled");
	}

	/**
	 * Getter for localized
	 * <code>message.connector.invalid_command_specified</code> property
	 * 
	 * @return InvalidCommandSpecified
	 */
	public String getInvalidCommandSpecified() {
		return getString("message.connector.invalid_command_specified");
	}
	
	/**
	 * Getter for localized
	 * <code>message.connector.createFolder.folder_already_exists_error</code> property
	 * 
	 * @return FolderAlreadyExistsError
	 */
	public String getFolderAlreadyExistsError() {
		return getString("message.connector.createFolder.folder_already_exists_error");
	}
	
	/**
	 * Getter for localized
	 * <code>message.connector.createFolder.invalid_new_folder_name_specified</code> property
	 * 
	 * @return InvalidNewFolderNameSpecified
	 */
	public String getInvalidNewFolderNameSpecified() {
		return getString("message.connector.createFolder.invalid_new_folder_name_specified");
	}
	
	/**
	 * Getter for localized
	 * <code>message.connector.createFolder.write_error</code> property
	 * 
	 * @return CreateFolderWriteError
	 */
	public String getCreateFolderWriteError() {
		return getString("message.connector.createFolder.write_error");
	}

	/**
	 * Getter for localized
	 * <code>message.connector.invalid_resource_type_specified</code> property
	 * 
	 * @return InvalidTypeSpecified
	 */
	public String getInvalidResouceTypeSpecified() {
		return getString("message.connector.invalid_resource_type_specified");
	}

	/**
	 * Getter for localized
	 * <code>message.connector.invalid_current_folder_specified</code> property
	 * 
	 * @return InvalidCurrentFolderSpecified
	 */
	public String getInvalidCurrentFolderSpecified() {
		return getString("message.connector.invalid_current_folder_specified");
	}

}
