/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2004-2009 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.response;

import java.io.StringWriter;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import net.fckeditor.connector.Connector;
import net.fckeditor.handlers.Command;
import net.fckeditor.handlers.LocalizedMessages;
import net.fckeditor.handlers.ResourceType;
import net.fckeditor.requestcycle.ThreadLocalData;
import net.fckeditor.tool.Utils;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Creates an XML response for every <code>GET</code> request of the Connector
 * servlet. This class maps directly to the XML layout described <a
 * href="http://docs.fckeditor.net/FCKeditor_2.x/Developers_Guide/Server_Side_Integration#The_Commands">here</a>.
 *
 * @version $Id: GetResponse.java 2868 2009-01-09 21:54:36Z mosipov $
 */
public class GetResponse {

	private Document document;
	private Element errorElement;
	private Element foldersElement;
	private Element filesElement;
	
	/** Error number OK */
	public static final int EN_OK = 0;
	
	/** Error number CUSTOM ERROR */
	public static final int EN_CUSTOM_ERROR = 1;

	/** Error number FOLDER ALREADY EXISTS */
	public static final int EN_FOLDER_ALREADY_EXISTS_ERROR = 101;

	/** Error number INVALID NEW FOLDER NAME */
	public static final int EN_INVALID_NEW_FOLDER_NAME_ERROR = 102;

	/** Error number SECURITY ERROR */
	public static final int EN_CREATE_FOLDER_SECURITY_ERROR = 103;

	/** Error number UNKNOWN ERROR */
	public static final int EN_UKNOWN_CREATE_FOLDER_ERROR = 110;
	
	/**
	 * 
	 * Use this constructor if want to respond a negative/error message with
	 * custom text.
	 * 
	 * @param number
	 * @param text
	 */
	public GetResponse(int number, String text) {
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			document = builder.newDocument();
		} catch (ParserConfigurationException e) {
			throw new RuntimeException(e);
		}

		Element root = document.createElement("Connector");
		document.appendChild(root);
		setError(number, text);
	}
	
	/**
	 * Use this constructor if want to respond a positive message.
	 * 
	 * @param command
	 * @param resourceType
	 * @param currentFolder
	 * @param constructedUrl
	 */
	public GetResponse(Command command, ResourceType resourceType, 
			String currentFolder, String constructedUrl) {
	
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			document = builder.newDocument();
		} catch (ParserConfigurationException e) {
			throw new RuntimeException(e);
		}
	
		Element root = document.createElement("Connector");
		document.appendChild(root);
		root.setAttribute("command", command.toString());
		root.setAttribute("resourceType", resourceType.getName());
	
		Element currentFolderElement = document.createElement("CurrentFolder");
		currentFolderElement.setAttribute("path", currentFolder);
	
		currentFolderElement.setAttribute("url", constructedUrl);
		root.appendChild(currentFolderElement);
	
	}

	/**
	 * Use this constructor if want to respond a negative/error message only.
	 * 
	 * @param number
	 */
	public GetResponse(int number) {
		this(number, null);
	}
	
	/**
	 * Sets an error number with a custom message.
	 * 
	 * @param number
	 * @param text
	 */
	public void setError(int number, String text) {

		if (errorElement == null) {
			errorElement = document.createElement("Error");
			document.getDocumentElement().appendChild(errorElement);
		}

		errorElement.setAttribute("number", String.valueOf(number));
		if (Utils.isNotEmpty(text))
			errorElement.setAttribute("text", text);

	}

	/**
	 * Sets an error number.
	 * 
	 * @param number
	 */
	public void setError(int number) {
		setError(number, null);
	}

	/**
	 * Lists all folders as XML tags.
	 * @param dir
	 */
	public void setFolders(final List<String> dirs) {
		if (foldersElement != null) {
			Element parent = (Element) foldersElement.getParentNode();
			parent.removeChild(foldersElement);
		}

		foldersElement = document.createElement("Folders");
		document.getDocumentElement().appendChild(foldersElement);

		for (String file : dirs) {
			Element folderElement = document.createElement("Folder");
			folderElement.setAttribute("name", file);
			foldersElement.appendChild(folderElement);
		}
	}
	
	/**
	 * Lists all files XML tags.
	 * 
	 * @param Map, key is the file name and value is the size of the file in bytes
	 */
	public void setFiles(final List<Map<String, Object>> files) {
		if (filesElement != null) {
			Element parent = (Element) filesElement.getParentNode();
			parent.removeChild(filesElement);
		}

		filesElement = document.createElement("Files");
		document.getDocumentElement().appendChild(filesElement);
		
		long length = 1L;
		long tempLength;
		
		for (Map<String, Object> file : files) {
			Element fileElement = document.createElement("File");
			fileElement.setAttribute("name", (String) file.get(Connector.KEY_NAME));
			tempLength = (Long) file.get(Connector.KEY_SIZE);
			if (tempLength > 1024)
				length = tempLength/1024;
			fileElement.setAttribute("size", String.valueOf(length));
			filesElement.appendChild(fileElement);
		}
	}
	
	@Override
	public String toString() {
		document.getDocumentElement().normalize();
		TransformerFactory factory = TransformerFactory.newInstance();

		StringWriter sw = new StringWriter();

		try {
			Transformer transformer = factory.newTransformer();

			DOMSource source = new DOMSource(document);
			StreamResult result = new StreamResult(sw);

			transformer.transform(source, result);
		} catch (TransformerException e) {
			throw new RuntimeException(e);
		}

		return sw.toString();
	}
	
	public static GetResponse getOK() {
		return new GetResponse(EN_OK);
	}
	
	public static GetResponse getInvalidCommandError() {
		LocalizedMessages lm = LocalizedMessages.getMessages(ThreadLocalData.getRequest());
		return new GetResponse(EN_CUSTOM_ERROR, lm.getInvalidCommandSpecified());
	}
	
	public static GetResponse getInvalidResourceTypeError() {
		LocalizedMessages lm = LocalizedMessages.getMessages(ThreadLocalData.getRequest());
		return new GetResponse(EN_CUSTOM_ERROR, lm.getInvalidResouceTypeSpecified());
	}
	
	public static GetResponse getInvalidCurrentFolderError() {
		LocalizedMessages lm = LocalizedMessages.getMessages(ThreadLocalData.getRequest());
		return new GetResponse(EN_CUSTOM_ERROR, lm.getInvalidCurrentFolderSpecified());
	}
	
	// TODO which EN to respond?
	public static GetResponse getGetResourcesDisabledError() {
		LocalizedMessages lm = LocalizedMessages.getMessages(ThreadLocalData.getRequest());
		return new GetResponse(EN_CUSTOM_ERROR, lm.getGetResourcesDisabled());
	}
	
	// TODO which EN to respond?
	public static GetResponse getGetResourcesReadError() {
		LocalizedMessages lm = LocalizedMessages.getMessages(ThreadLocalData.getRequest());
		return new GetResponse(EN_CUSTOM_ERROR, lm.getGetResourcesReadError());
	}
	
	// TODO which EN to respond?
	public static GetResponse getCreateFolderDisabledError() {
		LocalizedMessages lm = LocalizedMessages.getMessages(ThreadLocalData.getRequest());
		return new GetResponse(EN_CUSTOM_ERROR, lm.getCreateFolderDisabled());
	}
	
	public static GetResponse getInvalidNewFolderNameError() {
		LocalizedMessages lm = LocalizedMessages.getMessages(ThreadLocalData.getRequest());
		return new GetResponse(EN_INVALID_NEW_FOLDER_NAME_ERROR, lm
				.getInvalidNewFolderNameSpecified());
	}
	
	public static GetResponse getFolderAlreadyExistsError() {
		LocalizedMessages lm = LocalizedMessages.getMessages(ThreadLocalData.getRequest());
		return new GetResponse(EN_FOLDER_ALREADY_EXISTS_ERROR, lm
				.getFolderAlreadyExistsError());
	}
	
	// TODO which EN to respond?
	public static GetResponse getCreateFolderWriteError() {
		LocalizedMessages lm = LocalizedMessages.getMessages(ThreadLocalData.getRequest());
		return new GetResponse(EN_UKNOWN_CREATE_FOLDER_ERROR, lm
				.getCreateFolderWriteError());
	}
}