/*
** This code has been placed into the Public Domain.
** This code was written by Timothy Gerard Endres in 1999.
** 
*/

package com.ice.tartool;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import java.util.zip.*;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import javax.activation.*;

import com.ice.tar.*;
import com.ice.util.AWTUtilities;
import com.ice.util.MenuProperties;


public
class		ArchiveTreePanel
extends		JPanel
implements	TreeSelectionListener, ActionListener
	{
	protected String			fileDlgDir = null;
	protected File				archiveFile = null;
	protected JTree				tree = null;	
	protected ArchiveTreeModel	model = null;
	protected JScrollPane		treeScroller = null;
	protected Vector			actListeners = new Vector();
	protected ActionEvent		actEvent = null;
	protected JPopupMenu		treePopup = null;


	public
	ArchiveTreePanel()
		{
		this.setLayout( new BorderLayout() );
		this.setBorder( new EmptyBorder( 5, 5, 5, 5 ) );

		this.model =
			new ArchiveTreeModel
				( new ArchiveTreeNode
					( "<No Archive>", new TarEntry( "./" ) ) );

		this.tree = new JTree( this.model );
		this.tree.addTreeSelectionListener( this );
		this.tree.setShowsRootHandles( true );
		this.tree.setRootVisible( true );

		this.treePopup =
			MenuProperties.loadPopupMenu( "treePopup", this );

		this.tree.addMouseListener
			( this.new TreeMouseAdapter( this.tree, this.tree, this.treePopup ) );

		this.treeScroller = new JScrollPane( this.tree );

		this.add( treeScroller );
		}

	private TarInputStream
	openTarInputStream( File archiveFile )
		throws IOException
		{
		String path = archiveFile.getPath();

		// First, read four magic bytes. This is enough to decide if we
		// have a GZIP or ZIP archive. If so, we will wrap the input stream...

		InputStream in = new FileInputStream( archiveFile );
		byte[] magic = new byte[4];
		int numRead = in.read( magic );
		in.close();

		if ( numRead < magic.length )
			throw new IOException
				( "unexpected end of file reading "
					+ magic.length + " 'magic' bytes" );

		// UNDONE
		// Can we recognize and handle other compression formats?
		//
		if ( magic[0] == (byte)0x1F && magic[1] == (byte)0x8B )
			{
			// This is a GZIP archive.
			//
			// Reopen the archive as a GZIPInputStream.
			//
			in = new GZIPInputStream
					( new FileInputStream( archiveFile ) );
			}
		else if ( magic[0] == (byte)'P' && magic[1] == (byte)'K'
					&& magic[2] == (byte)0x03 && magic[3] == (byte)0x04 )
			{
			// This is a ZIP archive.
			//
			// Reopen the archive as a ZIPInputStream, and
			// position to the first entry. We only handle
			// the case where the tar archive is the first
			// entry in the ZIP archive.
			//
			ZipInputStream zin =
				new ZipInputStream
					( new FileInputStream( archiveFile ) );

			// REVIEW
			// UNDONE
			// Could/should we use this ZipEntry to set the name
			// displayed for the root entry of the tree?
			//
			ZipEntry zipEnt = zin.getNextEntry();
			in = zin;
			}
		else
			{
			// We do not know what this is!
			//
			// So, we will assume it is a simple tar archive
			// and reopen the file as a FileInputStream.
			//
			in = new FileInputStream( archiveFile );
			}

		return new TarInputStream( in );
		}

	ArchiveTreeNode
	getRootNode()
		{
		return ( this.model == null
			? null : (ArchiveTreeNode)this.model.getRoot() );
		}

	public void
	closeArchive()
		throws IOException
		{
		this.model =
			new ArchiveTreeModel
				( new ArchiveTreeNode
					( "<No Archive>", new TarEntry( "./" ) ) );
		this.tree.setModel( this.model );
		this.repaint( 250 );
		}

	public void
	openArchive( File archiveFile )
		throws IOException
		{
		this.archiveFile = archiveFile;

		TarInputStream tarIn = this.openTarInputStream( archiveFile );

		ArchiveTreeNode root =
			new ArchiveTreeNode
				( archiveFile.getName(),
					new TarEntry( archiveFile.getName() + "/" ) );

		try {
			TarEntry entry = tarIn.getNextEntry();
			for ( ; entry != null ; )
				{
				root.addPath( entry );
				entry = tarIn.getNextEntry();

			//	if ( this.progressDisplay != null )
			//		this.progressDisplay.showTarProgressMessage
			//			( entry.getName() );
				}
			}
		catch ( InvalidHeaderException ex )
			{
			String title = "Invalid Archive";
			String msg =
				"The file does not contain a valid tar archive.\n"
				+ "The reported problem was:\n\n"
				+ ex.getMessage();

			JOptionPane.showMessageDialog
				( this, msg, title, JOptionPane.WARNING_MESSAGE );
			}

		tarIn.close();

		root.computeArchiveTotals();

		this.model = new ArchiveTreeModel( root );
		this.tree.setModel( this.model );
		this.tree.expandRow( 0 );

		this.repaint( 250 );
		}

	private Frame
	findFrame()
		{
		Frame result = null;
		Container par = this;
		for ( ; ; )
			{
			par = par.getParent();

			if ( par == null )
				break;

			if ( par instanceof Frame )
				{
				result = (Frame) par;
				break;
				}
			}
		return result;
		}

	private void
	extractFile(
			TarEntry entry, File destFile, boolean dirExtract,
			boolean ascii, boolean mimed )
		throws IOException
		{
		FileOutputStream out = null;
		TarInputStream tarIn = null;

		try {
			tarIn = this.openTarInputStream( this.archiveFile );

			for ( ; ; )
				{
				TarEntry e = tarIn.getNextEntry();

				if ( e == null )
					break;

				boolean doIt = false;

				if ( dirExtract )
					{
					if ( e.equals( entry ) )
						doIt = false;
					else
						doIt = e.getName().startsWith( entry.getName() );

					if ( doIt )
						{
						String subPath =
							e.getName().substring( entry.getName().length() );

						if ( subPath.length() > 0 )
							{
							if ( e.isDirectory() )
								{
								File subDir = new File( destFile, subPath );
								subDir.mkdirs();
								doIt = false;
								}
							else
								{
								File oFile = new File( destFile, subPath );
								out = new FileOutputStream( oFile );
								}
							}
						}
					}
				else
					{
					doIt = e.equals( entry );
					if ( doIt )
						{
						out = new FileOutputStream( destFile );
						}
					}

				if ( mimed && doIt )
					{
					ascii = false;

					String contentType =
						FileTypeMap.getDefaultFileTypeMap().getContentType
							( dirExtract ? e.getName() : destFile.getName() );

					try {
						MimeType mime = new MimeType( contentType );
						if ( mime.getPrimaryType().equalsIgnoreCase( "text" ) )
							ascii = true;
						}
					catch ( MimeTypeParseException ex )
						{ }
					}

				if ( doIt )
					{
					byte[] buf = new byte[ 32 * 1024 ];

					byte[] lineSep =
						System.getProperty( "line.separator", "\n" ).getBytes();

					for ( ; ; )
						{
						int numRead = tarIn.read( buf );

						if ( numRead == -1 )
							break;
						
						if ( ascii )
							{
							int beg = 0;
							int off = 0;
							for ( ; off < numRead ; )
								{
								if ( buf[off] == 10 )
									{
									if ( (off - 1) > beg )
										{
										out.write( buf, beg, ((off - 1) - beg ) );
										}

									out.write( lineSep );

									beg = ++off;
									}
								else
									{
									++off;
									}
								}

							if ( off > beg )
								{
								out.write( buf, beg, ( off - beg ) );
								}
							}
						else
							{
							out.write( buf, 0, numRead );
							}
						}

					if ( out != null )
						{
						out.close();
						out = null;
						}

					if ( ! dirExtract )
						break;
					}
				}
			}
		finally
			{
			if ( out != null )
				out.close();

			if ( tarIn != null )
				tarIn.close();
			}
		}

	public void
	actionPerformed( ActionEvent event )
		{
		String command = event.getActionCommand();

		if ( command.startsWith( "EXTRACT" ) )
			{
			boolean mimed = command.equals( "EXTRACT_MIME" );
			boolean ascii = command.equals( "EXTRACT_ASCII" );

			ArchiveTreeNode node = (ArchiveTreeNode)
				tree.getLastSelectedPathComponent();

			if ( node != null )
				{
				TarEntry entry = node.getEntry();

				FileDialog fdlg =
					new FileDialog
						( this.findFrame(), "Extract Into", FileDialog.SAVE );

				String fName = node.getName();

				if ( fName.endsWith( ".tar" ) )
					fName = fName.substring( 0, fName.length() - 4 );

				if ( fName.endsWith( ".tgz" ) )
					fName = fName.substring( 0, fName.length() - 4 );

				if ( fName.endsWith( ".tar.gz" ) )
					fName = fName.substring( 0, fName.length() - 7 );

				fdlg.setFile( fName );

				if ( this.fileDlgDir != null )
					fdlg.setDirectory( this.fileDlgDir );

				fdlg.show();

				String fStr = fdlg.getFile();
				this.fileDlgDir = fdlg.getDirectory();

				if ( fStr != null && this.fileDlgDir != null )
					{
					boolean proceed = true;

					if ( entry.isDirectory() )
						{
						File dirF = new File( this.fileDlgDir );
						File destDir = new File( dirF, fStr );

						if ( ! destDir.exists() )
							{
							destDir.mkdirs();
							}

						if ( proceed )
							{
							try {
								this.extractFile
									( entry, destDir, true, ascii, mimed );
								}
							catch ( IOException ex )
								{
								ex.printStackTrace();
								}
							}
						}
					else
						{
						File dirF = new File( this.fileDlgDir );
						File destFile = new File( dirF, fStr );

						if ( destFile.exists() )
							{
							if ( destFile.isDirectory() )
								{
								}
							else
								{
								}
							}

						if ( proceed )
							{
							try {
								this.extractFile
									( entry, destFile, false, ascii, mimed );
								}
							catch ( IOException ex )
								{
								ex.printStackTrace();
								}
							}
						}
					}
				}
			}
		}

	public void
	addActionListener( ActionListener l )
		{
		this.actListeners.addElement( l );
		}

	public void
	removeActionListener( ActionListener l )
		{
		this.actListeners.removeElement( l );
		}

	private void
	performAction( ActionEvent event )
		{
		Enumeration enum = this.actListeners.elements();

		for ( ; enum.hasMoreElements() ; )
			{
			((ActionListener) enum.nextElement()).actionPerformed( event );
			}
		}

	public void
	valueChanged( TreeSelectionEvent event )
		{
		Object source = tree.getLastSelectedPathComponent();

		if ( source != null )
			this.performAction
				( new ActionEvent
					( source, ActionEvent.ACTION_PERFORMED,
						"NODE_SELECTED" ) );
		}

	public String
	treePath( TreePath treePath )
		{
		ArchiveTreeNode node;
		Object[] list = treePath.getPath();
		StringBuffer path = new StringBuffer();

		for ( int i = 1 ; i < list.length ; i++ )
			{
			node = (ArchiveTreeNode) list[i];
			if ( i > 1 )
				path.append("/");
			path.append( node.getName() );
			}

		return path.toString();
		}

	private class
	TreeMouseAdapter extends MouseAdapter
		{
		private JTree		tree;
		private JPopupMenu	popupMenu;
		private JComponent	popupParent;
		private boolean		isPopupClick = false;

		public
		TreeMouseAdapter( JComponent parent, JTree tree, JPopupMenu menu )
			{
			super();
			this.tree = tree;
			this.popupMenu = menu;
			this.popupParent = parent;
			}

		public void
		mousePressed( MouseEvent event )
			{
			this.isPopupClick = false;

			if ( event.isPopupTrigger() )
				{
				int selRow =
					this.tree.getRowForLocation
						( event.getX(), event.getY() );

				if ( selRow != -1 )
					{
					this.tree.setSelectionRow( selRow );
					}

				this.isPopupClick = true;
				this.popupMenu.show
					( this.popupParent, event.getX(), event.getY() );
				}
			}

		public void
		mouseReleased( MouseEvent event )
			{
			if ( this.isPopupClick )
				return;

			if ( event.isPopupTrigger() )
				{
				int selRow =
					this.tree.getRowForLocation
						( event.getX(), event.getY() );

				if ( selRow != -1 )
					{
					this.tree.setSelectionRow( selRow );
					}

				this.isPopupClick = true;
				this.popupMenu.show
					( this.popupParent, event.getX(), event.getY() );
				}
			}

		public void
		mouseClicked( MouseEvent event )
			{
			if ( this.isPopupClick )
				{
				this.isPopupClick = false;
				return;
				}

			if ( event.getClickCount() == 2 )
				{
				this.processDoubleClick();
				}
			}

		private void
		processDoubleClick()
			{
			}
		}

	}


