/* 
 * FCKdtd2js - FCKeditor JavaScript DTD map generator - http://www.fckeditor.net 
 * Copyright (C) 2003-2007 Frederico Caldeira Knabben 
 * 
 * == BEGIN LICENSE == 
 * 
 * Licensed under the terms of any of the following licenses at your 
 * choice: 
 * 
 *  - GNU General Public License Version 2 or later (the "GPL") 
 *    http://www.gnu.org/licenses/gpl.html 
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL") 
 *    http://www.gnu.org/licenses/lgpl.html 
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL") 
 *    http://www.mozilla.org/MPL/MPL-1.1.html 
 * 
 * == END LICENSE == 
 */ 

package net.fckeditor.devutil.dtd;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.ParseException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeSet;

/**
 * Takes a DTD file as input, and produces a compressed JS map from it.
 * 
 * This feature is used to create the maps found in the FCK editors cleanup routine.
 */
public class DTDJsGenerator {
	
	private static final String APPLICATION_NAME = "FCKdtd2js";
	
	/**
	 * @param args
	 */
	public static void main(String[] args) {
		// Check number of arguments.
		if(args.length < 1) writeHelp(System.out);
		
		// Prepare holder for URI.
		URI dtdUri = null;
		
		// First test for file
		File dtdFile = new File(args[0].trim());
		if(dtdFile.exists()) {
			// Convert File to URI
			dtdUri = dtdFile.getAbsoluteFile().toURI();
		}
		else {
			// Try to parse as URI.
			try {
				dtdUri = new URI(args[0].trim());
				if(!dtdUri.isAbsolute()) {
					System.err.println("URI is not absolute : " + args[0].trim());
					return;
				}
			}
			catch(URISyntaxException e) {
				System.err.println("Unable to parse : " + args[0].trim());
				return;
			}
		}
		
		// Load all root tags.
		Set removeTags = new HashSet();
		if(args.length > 1) {
			StringTokenizer st = new StringTokenizer(args[1],",");
			while (st.hasMoreTokens()) {
		         String removeTag = st.nextToken().trim();
		         if(removeTag.length()>0) removeTags.add(removeTag);
		    }
		}
		
		// Create application instance, and execute the parsing.
		try {
			DTDJsGenerator generator = new DTDJsGenerator(dtdUri, removeTags);
			generator.run();
		}
		catch(Exception e) {
			System.err.println(e.getLocalizedMessage());
			e.printStackTrace(System.err);
		}
	}
	
	protected static void writeHelp(PrintStream out) {
		out.println(APPLICATION_NAME + ": Simple utility to create a JavaScript from a given DTD.\n");
		out.println("Usage: java -jar "+APPLICATION_NAME+".jar <dtdfile> <ignoretag,...>]");
		out.println("  dtdfile - The path/URL to the DTD file to parse.");
		out.println("  ignoretag - Comma separated list of tags to ignore in DTD");
		System.exit(0);
	}
	  	
	//#####################################################################
	// Instance methods.
	//#####################################################################
	
	protected URI 	_dtdUri;
	protected Set  	_removeTags;
	
	/**
	 * Creates a new DTDJsGenerator instance.
	 * 
	 * @param dtdFile The DTD file to parse.
	 * @param removeTags The tags to use as root tags.
	 */
	public DTDJsGenerator(URI dtdUri, Set removeTags) {
		if(dtdUri == null || removeTags == null) throw new IllegalArgumentException("File parameter cannot be null.");
		_dtdUri = dtdUri;
		if(removeTags == null) removeTags = Collections.EMPTY_SET;
		_removeTags = removeTags;
	}

	/**
	 * Runs the generator.
	 */
	public void run() throws IOException, ParseException {
		// Parse the DTD tree.
		XmlDefinitionParser xmlDefParser = new WutkaDTDParser();
		Map<String, ElementGroup> groupMap = xmlDefParser.parseXmlDefinition( _dtdUri );
		
		// Remove unwanted elements from the map.
		if(_removeTags!=null) {
			groupMap.keySet().removeAll(_removeTags);
		}
		
		// Compress the map.
		compressElementGroupMap(groupMap);

		// Reduce reference distance in map.
		flattenElementGroupMap(groupMap);
		
		// Display the group map internals.
		//printGroupMap(groupMap);
		
		// Build Javascript from the DTD map.
		String comment = "This file was automatically generated from : " + (new File(_dtdUri.getPath())).getName();
		
		ElementGroupMapJavascriptBuilder jsBuilder = new ElementGroupMapJavascriptBuilder();
		String javaScript = jsBuilder.buildJavaScript(comment, groupMap);
		System.out.println(javaScript);
	}

	protected void printGroupMap(Map<String, ElementGroup> groupMap) {
		// Iterate all groups.
		for(Iterator<String> it = groupMap.keySet().iterator(); it.hasNext(); ) {
			String key = it.next();
			ElementGroup group = groupMap.get(key);
			System.out.println("Element ["+key+"]:");
			System.out.println(group.toString()+"\n");		
		}
	}
	
	/**
	 * This method reduces distances in the resulting element group map by
	 * removing references to referece only subgroups. This is separated into
	 * a separate method since it is a self contained algorithm. Applying this
	 * process to the map will increase it's footprint, but the reduction
	 * in reference depth will result in faster traversion. However this is
	 * so marginal that I will consider if this feature will be included at all.
	 * 
	 * The principle of this method is that any subgroup which does not contain
	 * any elements (only subgroups), can be reduced by moving the subgroups up to
	 * the current group.
	 * 
	 * @param groupMap The groupmap to reduce.
	 */
	protected void flattenElementGroupMap(Map<String, ElementGroup> groupMap) {
		for(Iterator<String> it = groupMap.keySet().iterator(); it.hasNext(); ) {
			internalRecursiveFlattenElementGroupMap(groupMap.get(it.next()));
		}
	}
	// Helper method for reduceElementGroupMap
	private void internalRecursiveFlattenElementGroupMap(ElementGroup group) {
		List<ElementGroup> addList = new LinkedList<ElementGroup>();
		// Loop through subgroups.
		for(Iterator<ElementGroup> it = group.getSubGroups().iterator(); it.hasNext(); ) {
			ElementGroup currSubGroup = it.next();
			// Make sure subgroups are processed first.
			internalRecursiveFlattenElementGroupMap(currSubGroup);
			// Check if subgroup has no elements.
			if(currSubGroup.size()==0) {
				// Add all subgroups to this groups subgroup.
				addList.addAll(currSubGroup.getSubGroups());
				// Remove current subgroup.
				it.remove();
			}
		}
		// Add all discovered subgroups.
		group.getSubGroups().addAll(addList);
	}
	
	/**
	 * Method that collects equal element groups into common references
	 * and compresses the element group map.
	 * 
	 * @param groupMap The groupmap to compress.
	 */
	protected void compressElementGroupMap(Map<String, ElementGroup> groupMap) {
		
		/*
		 * The compression algorithm goes as follows:
		 * 
		 * 1. Insure all equal groups share the same group instance.
		 * 
		 * 2. Place all groups in a tree which sorts by size.
		 * 
		 * 3. Get the largest group in the tree.
		 * 
		 * 4. Iterate through the remaining groups in tree, and try to
		 *    create an intersection with the largest group.
		 *    
		 *    4.1 When an non-empty intersection occurrs, add the intersection
		 *        as a subgroup to the large element group. Add the subgroup
		 *        to the tree.
		 *        
		 *    4.2 If no intersection is found, remove the group from the tree.
		 *    
		 * 5. Continue until tree is empty (no more intersections are possible).
		 */
		
		// First insure all equal groups share the same group instance.
		for(Iterator<String> it1 = groupMap.keySet().iterator(); it1.hasNext(); ) {
			String key1 = it1.next();
			for(Iterator<String> it2 = groupMap.keySet().iterator(); it2.hasNext(); ) {
				String key2 = it2.next();
				// Retrieve group objects.
				ElementGroup group1 = groupMap.get(key1);
				ElementGroup group2 = groupMap.get(key2);
				
				// Check if groups are equal, but not the same object.
				if(group1 != group2 && group1.equals(group2)) {
					groupMap.put(key2, group1);
				}
			}
		}
		
		// Create tree that autosorts the map 
		TreeSet<ElementGroup> tree = new TreeSet<ElementGroup>(new ElementGroupSizeComparator());
		
		// Add all groups to the tree.
		tree.addAll(groupMap.values());
		
		// Loop until the tree does not contain any more elements.
		mainIntersectLoop:
		while(tree.size() > 0) {
			
			// Create a tree iterator.
			Iterator<ElementGroup> it = tree.iterator();
			
			// Get the top element from the tree.
			ElementGroup topGroup = it.next();
			
			// If the top group has 2 or less elements, we remove it from the tree and continue.
			if(topGroup.size() > 2) { //+ topGroup.getSubGroups().size() > 1) {
			
				// Loop through the remainig groups.
				while( it.hasNext() ) {
					// Get a candidate for intersection.
					ElementGroup candidate = it.next();
					ElementGroup intersection = topGroup.intersectGroup(candidate);
					
					// If the intersection contains less than 2 elements, we continue to the next candidate.
					if(intersection.size() < 2) continue;
					
					// An intersection has occurred.
					// Remove elements from tree.
					it.remove();
					tree.remove(tree.first());
					
					// Add the intersection as a subgroup of both involved groups.
					topGroup.addSubGroup(intersection);
					candidate.addSubGroup(intersection);
					
					// Re-add the groups to the tree.
					tree.add(topGroup);
					tree.add(candidate);
					
					// Add the intersection as a new group in the tree.
					tree.add(intersection);
					
					// Break out of loop, and continue at main loop.
					continue mainIntersectLoop;
				}
			}
			
			// If we get here, there was no possible intersections for the top element.
			// It should therefore be removed from the tree. 
			tree.remove(tree.first());
		}
	}


	
}
