package net.fckeditor.devutil.dtd;

import java.io.IOException;
import java.net.URI;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.wutka.dtd.DTD;
import com.wutka.dtd.DTDContainer;
import com.wutka.dtd.DTDElement;
import com.wutka.dtd.DTDEmpty;
import com.wutka.dtd.DTDItem;
import com.wutka.dtd.DTDName;
import com.wutka.dtd.DTDPCData;
import com.wutka.dtd.DTDParser;

/**
 * XML definition parser using the Wutka DTD parser library.
 */
public class WutkaDTDParser implements XmlDefinitionParser {

	/**
	 * {@inheritDoc}
	 */
	public Map<String, ElementGroup> parseXmlDefinition(URI uri) throws IOException {
		// Parse the DTD tree.
		DTDParser parser = new DTDParser(uri.toURL());
		DTD dtd = parser.parse();
		return createElementGroupMap(dtd);
	}

	/**
	 * Creates a map over all elementgroups in the DTD.
	 * @param dtd The DTD to create map from
	 * @return The Map containing all elementgroups.
	 */
	protected Map<String,ElementGroup> createElementGroupMap(DTD dtd) {
		// Create a common empty group...
		ElementGroup<String> emptyGroup = new ElementGroup<String>();
		
		// Create map containing the element groups.
		Map<String,ElementGroup> resultMap = new HashMap<String,ElementGroup>();
		// Get elements from DTD.
		Collection elements = dtd.elements.values();
		
		// Loop through elements and register them.		
		for(Iterator it = elements.iterator(); it.hasNext(); ) {
			DTDElement element = (DTDElement)it.next();
			
			// Check if element exists in nodemap.
			if(element.content instanceof DTDContainer) {
				
				//System.out.println("Container: " + element.name );
				
				// Create an element group to contain sub elements.
				ElementGroup<String> newGroup = new ElementGroup<String>();

				// Call recursive method to get all nodes.
				if(appendAllowedElementsRecursive((DTDContainer)element.content, newGroup)) {
					// Add the group to the result map.
					resultMap.put(element.name, newGroup);
				}
			}
			else
			if(element.content instanceof DTDEmpty){
				//System.out.println("Empty    : " +element.name);
				// Handle DTDEmpty
				resultMap.put(element.name, emptyGroup);
			}
		}
		
		return resultMap;
	}
	
	/**
	 * We need a recursive method in order to resolve all elements in a collection.
	 */
	protected boolean appendAllowedElementsRecursive(DTDContainer col, ElementGroup<String> targetGroup) {
		// Loop through allowed children.
		DTDItem[] allowedChilds = col.getItems();
		
		boolean hasRealContent = false;
		for(int i = 0; i < allowedChilds.length; i++) {
			// System.out.println(" - Allowed childs ["+col+"] : " + allowedChilds[i].getClass() );
			
			if(allowedChilds[i] instanceof DTDName) {
				targetGroup.add(((DTDName)allowedChilds[i]).value);
				hasRealContent = true;
			}
			else 
			if(allowedChilds[i] instanceof DTDContainer) {
				boolean recResult = appendAllowedElementsRecursive((DTDContainer)allowedChilds[i], targetGroup);
				hasRealContent = hasRealContent || recResult; 
			}
			else
			if(allowedChilds[i] instanceof DTDPCData) {
				targetGroup.add("#"); // Indicates that text is allowed.
				hasRealContent = true;
			}
			else
			if(allowedChilds[i] instanceof DTDEmpty) {
				hasRealContent = true;
			}
		}
		return hasRealContent;
	}
}
