/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2004-2009 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.handlers;

import java.util.HashMap;
import java.util.Map;

import net.fckeditor.tool.Utils;

/**
 * Reflects File Browser <code>GET</code> and <code>POST</code> commands.
 * <p>
 * The File Browser send a specific command for each and every request. This
 * class is intended to reflect those in a Enum-like manner.
 * </p>
 * <p>
 * The commands are for <code>GET</code>:
 * <ul>
 * <li>GetFolders</li>
 * <li>GetFoldersAndFiles</li>
 * <li>CreateFolder</li>
 * </ul>
 * and for <code>POST</code>:
 * <ul>
 * <li>FileUpload</li>
 * <li>QuickUpload</li>
 *</ul>
 * 
 * @version $Id: Command.java 3402 2009-04-20 12:12:29Z mosipov $
 */
public class Command {

	private String name;
	private static final Map<String, Command> getCommands = new HashMap<String, Command>(
			3);
	private static final Map<String, Command> postCommands = new HashMap<String, Command>(
			2);
	/** GET command <code>GetFolders</code> */
	public static final Command GET_FOLDERS = new Command("GetFolders");
	/** GET command <code>GetFoldersAndFiles</code> */
	public static final Command GET_FOLDERS_AND_FILES = new Command(
			"GetFoldersAndFiles");
	/** GET command <code>CreateFolder</code> */
	public static final Command CREATE_FOLDER = new Command("CreateFolder");
	/** POST command <code>FileUpload</code> */
	public static final Command FILE_UPLOAD = new Command("FileUpload");
	/** POST command <code>QuickUpload</code> */
	public static final Command QUICK_UPLOAD = new Command("QuickUpload");

	static {
		// initialize the get commands
		getCommands.put(GET_FOLDERS.getName(), GET_FOLDERS);
		getCommands.put(GET_FOLDERS_AND_FILES.getName(), GET_FOLDERS_AND_FILES);
		getCommands.put(CREATE_FOLDER.getName(), CREATE_FOLDER);

		// initialize the post commands
		postCommands.put(FILE_UPLOAD.getName(), FILE_UPLOAD);
		postCommands.put(QUICK_UPLOAD.getName(), QUICK_UPLOAD);
	}

	/**
	 * Constructs a command with the given name.
	 * 
	 * @param name
	 *            a reasonable command name
	 */
	private Command(final String name) {
		this.name = name;
	}

	/**
	 * Returns the command name.
	 * 
	 * @return the command name
	 */
	public String getName() {
		return name;
	}

	/**
	 * Returns a command object for a command name string.
	 * 
	 * @param name
	 *            command to retrieve
	 * @return {@link Command} object corresponding to the name
	 * @throws IllegalArgumentException
	 *             if specified name has no equivalent command
	 * @throws NullPointerException
	 *             if <code>name</code> if null or empty
	 */
	public static Command valueOf(final String name) {
		if (Utils.isEmpty(name))
			throw new NullPointerException("Parameter name is null or empty");

		Command command = getCommands.get(name);
		if (command == null)
			command = postCommands.get(name);
		if (command == null)
			throw new IllegalArgumentException("No equivalent command found");

		return command;
	}

	/**
	 * Checks if a specified command name represents a valid <code>GET</code>
	 * command.
	 * 
	 * @param name
	 *            command string to check
	 * @return <code>true</code> if the command exists else <code>false</code>.
	 */
	public static boolean isValidForGet(final String name) {
		return getCommands.containsKey(name);
	}

	/**
	 * Checks if a specified command name represents a valid <code>POST</code>
	 * command.
	 * 
	 * @param name
	 *            command string to check
	 * @return <code>true</code> if the command exists else <code>false</code>.
	 */
	public static boolean isValidForPost(final String name) {
		return postCommands.containsKey(name);
	}

	/**
	 * Returns a command object for a command name string. In contrast to
	 * {@link Command#valueOf(String)} it returns a <code>null</code> instead of
	 * throwing an exception if a command was not found.
	 * 
	 * @param name
	 *            command to retrieve
	 * @return {@link Command} object corresponding to the name (may be null)
	 */
	public static Command getCommand(final String name) {
		try {
			return Command.valueOf(name);
		} catch (Exception e) {
			return null;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;

		if (obj == null || this.getClass() != obj.getClass())
			return false;

		final Command command = (Command) obj;
		return name.equals(command.getName());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public int hashCode() {
		return name.hashCode();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return name;
	}
}
