/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2007 Frederico Caldeira Knabben
 *
 * == BEGIN LICENSE ==
 *
 * Licensed under the terms of any of the following licenses at your
 * choice:
 *
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 *
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 *
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 *
 * == END LICENSE ==
 *
 * This is the code behind of the connector.aspx page used by the 
 * File Browser.
 */

using System ;
using System.Globalization ;
using System.Xml ;
using System.Web ;

namespace FredCK.FCKeditorV2.FileBrowser
{
	public class Connector : FileWorkerBase
	{
		protected override void OnLoad(EventArgs e)
		{
			Config.LoadConfig();

			if ( !Config.Enabled )
			{
				XmlResponseHandler.SendError( Response, 1, "This connector is disabled. Please check the \"editor/filemanager/connectors/aspx/config.ascx\" file." );
				return;
			}

			// Get the main request information.
			string sCommand = Request.QueryString["Command"] ;
			string sResourceType = Request.QueryString[ "Type" ];
			string sCurrentFolder = Request.QueryString[ "CurrentFolder" ];

			if ( sCommand == null || sResourceType == null || sCurrentFolder == null )
			{
				XmlResponseHandler.SendError( Response, 1, "Invalid request." );
				return;
			}

			// Check if it is an allowed type.
			if ( !Config.CheckIsTypeAllowed( sResourceType ) )
			{
				XmlResponseHandler.SendError( Response, 1, "Invalid resource type specified." ) ;
				return ;
			}

			// Check the current folder syntax (must begin and start with a slash).
			if ( ! sCurrentFolder.EndsWith( "/" ) )
				sCurrentFolder += "/" ;
			if ( ! sCurrentFolder.StartsWith( "/" ) )
				sCurrentFolder = "/" + sCurrentFolder ;

			// Check for invalid folder paths (..).
			if ( sCurrentFolder.IndexOf( ".." ) >= 0 || sCurrentFolder.IndexOf( "\\" ) >= 0 )
			{
				XmlResponseHandler.SendError( Response, 102, "" );
				return;
			}

            // Start impersonation if configured
            System.Security.Principal.WindowsImpersonationContext impContext = null;
            System.Security.Principal.WindowsIdentity curIdentity;
            int iImpersonationState = -1; // -1: Inactive; 0: Failed; 1: Active
            if (Config.UseImpersonation)
            {
                iImpersonationState = 0;
                try
                {
                    curIdentity = (System.Security.Principal.WindowsIdentity)User.Identity;
                    impContext = curIdentity.Impersonate();
                    iImpersonationState = 1;
                }
                catch ( Exception )
                {
                    // Any Exception while initiating impersonation is returned as a generic Security Exception
                    XmlResponseHandler.SendError(Response, 103, "");
                    return;
                }
            }

            // Exception info buffers
            int iErrorCode = -1;
            string sErrorMsg = string.Empty;
            string sErrorType = string.Empty;

            // File Upload doesn't have to return XML, so it must be intercepted before anything.
			if ( sCommand == "FileUpload" )
			{
                try
                {
                    this.FileUpload(sResourceType, sCurrentFolder, false);
                }
                catch (System.IO.PathTooLongException ex)
                {
                    iErrorCode = 400;
                    sErrorType = "Path too long";
                    sErrorMsg = ex.Message;
                }
                catch (System.IO.IOException ex)
                {
                    iErrorCode = 410;
                    sErrorType = "IO Error";
                    sErrorMsg = ex.Message;
                }
                catch (System.Security.SecurityException ex)
                {
                    iErrorCode = 403;
                    sErrorType = "Security error";
                    sErrorMsg = ex.Message;
                }
                catch (System.UnauthorizedAccessException ex)
                {
                    iErrorCode = 403;
                    sErrorType = "Unauthorized";
                    sErrorMsg = ex.Message;
                }
                catch (Exception ex)
                {
                    iErrorCode = 500;
                    sErrorType = "Unexpected Exception";
                    sErrorMsg = string.Format("({0}): {1}", ex.GetType().ToString(), ex.Message );
                }
                // If Impersonation active, go back to normal
                if (iImpersonationState > 0)
                {
                    impContext.Undo();
                }
                if (iErrorCode != -1)
                {
                    Response.StatusCode = iErrorCode;
                    Response.Clear();
                    Response.Write(string.Format(
                        Config.RemoteMessages ? "Error {1} - {0}:\n{2}" : "Error: {0}", 
                        sErrorType, iErrorCode, sErrorMsg));
                    Response.End();
                }

				return ;
			}

            // Nothing goes unchecked while Impersonation may be active
            XmlResponseHandler oResponseHandler;
            XmlNode oConnectorNode;
            try
            {
                oResponseHandler = new XmlResponseHandler(this, Response);
                oConnectorNode = oResponseHandler.CreateBaseXml(sCommand, sResourceType, sCurrentFolder);
            }
            catch (Exception ex)
            {
                XmlResponseHandler.SendError(Response, 101, ex.Message);
                return;
            }

            // If impersonation OK or not required, do the work
            if (iImpersonationState != 0)
            {
                try
                {
                    // Execute the required command.
                    switch (sCommand)
                    {
                        case "GetFolders":
                            iErrorCode = this.GetFolders(oConnectorNode, sResourceType, sCurrentFolder);
                            break;
                        case "GetFoldersAndFiles":
                            iErrorCode = this.GetFolders(oConnectorNode, sResourceType, sCurrentFolder);
                            iErrorCode = this.GetFiles(oConnectorNode, sResourceType, sCurrentFolder);
                            break;
                        case "CreateFolder":
                            iErrorCode = this.CreateFolder(oConnectorNode, sResourceType, sCurrentFolder);
                            break;
                    }
                }
                catch (ArgumentException ex)
                {
                    iErrorCode = 102;
                    sErrorMsg = ex.Message;
                }
                catch (System.IO.PathTooLongException ex)
                {
                    iErrorCode = 102;
                    sErrorMsg = ex.Message;
                }
                catch (System.IO.IOException ex)
                {
                    iErrorCode = 101;
                    sErrorMsg = ex.Message;
                }
                catch (System.Security.SecurityException ex)
                {
                    iErrorCode = 103;
                    sErrorMsg = ex.Message;
                }
                catch (System.UnauthorizedAccessException ex)
                {
                    iErrorCode = 103;
                    sErrorMsg = ex.Message;
                }
                catch (Exception ex)
                {
                    iErrorCode = 110;
                    sErrorMsg = string.Format("({0}): {1}", ex.GetType().ToString(), ex.Message);
                }
            }

            // If Impersonation active, go back to normal
            if (iImpersonationState > 0)
            {
                impContext.Undo();
            }

            // Append Error Code to result XML
            if (iErrorCode != -1)
            {
                XmlNode oErrorNode = XmlUtil.AppendElement(oConnectorNode, "Error");
                XmlUtil.SetAttribute(oErrorNode, "number", iErrorCode.ToString());
                if (sErrorMsg.Length > 0 && Config.RemoteMessages)
                {
                    XmlUtil.SetAttribute(oErrorNode, "message", sErrorMsg);
                }
            }

            // Return finished XML
            oResponseHandler.SendResponse();
		}

		#region Command Handlers

		private int GetFolders( XmlNode connectorNode, string resourceType, string currentFolder )
		{
			// Map the virtual path to the local server path.
			string sServerDir = this.ServerMapFolder( resourceType, currentFolder, false ) ;

			// Create the "Folders" node.
			XmlNode oFoldersNode = XmlUtil.AppendElement( connectorNode, "Folders" ) ;

			System.IO.DirectoryInfo oDir = new System.IO.DirectoryInfo( sServerDir ) ;
			System.IO.DirectoryInfo[] aSubDirs = oDir.GetDirectories() ;

			for ( int i = 0 ; i < aSubDirs.Length ; i++ )
			{
				// Create the "Folders" node.
				XmlNode oFolderNode = XmlUtil.AppendElement( oFoldersNode, "Folder" ) ;
				XmlUtil.SetAttribute( oFolderNode, "name", aSubDirs[i].Name ) ;
			}

            return -1;
        }

		private int GetFiles( XmlNode connectorNode, string resourceType, string currentFolder )
		{
			// Map the virtual path to the local server path.
			string sServerDir = this.ServerMapFolder( resourceType, currentFolder, false ) ;

            // Fetch the information
            System.IO.DirectoryInfo oDir = new System.IO.DirectoryInfo(sServerDir);
            System.IO.FileInfo[] aFiles = oDir.GetFiles();

            // Create the "Files" node.
			XmlNode oFilesNode = XmlUtil.AppendElement( connectorNode, "Files" ) ;

            for ( int i = 0 ; i < aFiles.Length ; i++ )
			{
				Decimal iFileSize = Math.Round( (Decimal)aFiles[i].Length / 1024 ) ;
				if ( iFileSize < 1 && aFiles[i].Length != 0 ) iFileSize = 1 ;

				// Create the "File" node.
				XmlNode oFileNode = XmlUtil.AppendElement( oFilesNode, "File" ) ;
				XmlUtil.SetAttribute( oFileNode, "name", aFiles[i].Name ) ;
				XmlUtil.SetAttribute( oFileNode, "size", iFileSize.ToString( CultureInfo.InvariantCulture ) ) ;
			}

            return -1;
		}

		private int CreateFolder( XmlNode connectorNode, string resourceType, string currentFolder )
		{
			string sNewFolderName = Request.QueryString["NewFolderName"] ;
			sNewFolderName = this.SanitizeFolderName( sNewFolderName );

			if ( sNewFolderName == null || sNewFolderName.Length == 0 ) {
				return 102;
            }
		    // Map the virtual path to the local server path of the current folder.
		    string sServerDir = this.ServerMapFolder( resourceType, currentFolder, false ) ;

            Util.CreateDirectory(System.IO.Path.Combine(sServerDir, sNewFolderName), Config.CreateDirectoryWorkaround);

            return 0;
		}


		#endregion

		#region Directory Mapping

		internal string GetUrlFromPath( string resourceType, string folderPath )
		{
			if ( resourceType == null || resourceType.Length == 0 )
				return this.Config.UserFilesPath.TrimEnd( '/' ) + folderPath;
			else
				return this.Config.TypeConfig[ resourceType ].GetFilesPath().TrimEnd( '/' ) + folderPath;
		}

		#endregion
	}
}
