/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2007 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.connector;

import static net.fckeditor.tool.Utils.forceSingleExtension;
import static net.fckeditor.tool.Utils.isEmpty;
import static net.fckeditor.tool.Utils.isValidPath;
import static net.fckeditor.tool.Utils.replaceAll;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.List;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import net.fckeditor.ConfigurationHandler;
import net.fckeditor.ExtensionsHandler;
import net.fckeditor.tool.UploadResponse;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.commons.io.filefilter.FileFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * Servlet to upload and browse files.<br>
 * 
 * This servlet accepts 4 commands used to retrieve and create files and folders
 * from a server directory. The allowed commands are:
 * <ul>
 * <li>GetFolders: Retrive the list of directory under the current folder
 * <li>GetFoldersAndFiles: Retrive the list of files and directory under the
 * current folder
 * <li>CreateFolder: Create a new directory under the current folder
 * <li>FileUpload: Send a new file to the server (must be sent with a POST)
 * </ul>
 * 
 * @version $Id: ConnectorServlet.java 1407 2008-01-24 20:06:06Z mosipov $
 */
public class ConnectorServlet extends HttpServlet {

	private static final long serialVersionUID = -5742008970929377161L;
	private static final Logger logger = LoggerFactory
			.getLogger(ConnectorServlet.class);

	/**
	 * Initialize the servlet.<br>
	 * Retrieve from the servlet configuration the "baseDir" which is the root
	 * of the file repository.
	 */
	public void init() throws ServletException {
		if (getInitParameter("baseDir") != null)
			ConfigurationHandler.setBaseDir(getInitParameter("baseDir"));
		String realBaseDir = getServletContext().getRealPath(
				ConfigurationHandler.getBaseDir());
		File baseFile = new File(realBaseDir);
		if (!baseFile.exists()) {
			baseFile.mkdirs();
		}
		logger.info("*** Connector Servlet initialized successfull!");
	}

	/**
	 * Manage the Get requests (GetFolders, GetFoldersAndFiles, CreateFolder).<br>
	 * 
	 * The servlet accepts commands sent in the following format:<br>
	 * connector?Command=CommandName&Type=ResourceType&CurrentFolder=FolderPath<br>
	 * <br>
	 * It execute the command and then return the results to the client in XML
	 * format.
	 * 
	 */
	public void doGet(HttpServletRequest request, HttpServletResponse response)
			throws ServletException, IOException {
		logger.debug("Entered #doGet.");

		response.setCharacterEncoding("UTF-8");
		response.setContentType("application/xml; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();

		String commandStr = request.getParameter("Command");
		String typeStr = request.getParameter("Type");
		String currentFolderStr = request.getParameter("CurrentFolder");
		// boolean validPath = Utils.isValidPath(currentFolderStr);
		// TODO untersuchen wie es vom Res Browser kommt
		String currentPath = constructTypeBasedFolderString(typeStr,
				currentFolderStr);
		String currentDirPath = getServletContext().getRealPath(currentPath);

		File currentDir = new File(currentDirPath);
		if (!currentDir.exists()) {
			currentDir.mkdirs();
			logger.debug("Dir successfull created: " + currentDirPath);
		}

		Document document = null;
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			document = builder.newDocument();
		} catch (ParserConfigurationException pce) {
			logger.error("Error while parsing DOM: "
					+ pce.getLocalizedMessage(), pce);
		}

		Node root = createCommonXml(document, commandStr, typeStr,
				currentFolderStr, request.getContextPath() + currentPath);
		logger.debug("Command = " + commandStr);

		if (commandStr.equals("GetFolders")) {
			getFolders(currentDir, root, document);
		} else if (commandStr.equals("GetFoldersAndFiles")) {
			getFolders(currentDir, root, document);
			getFiles(currentDir, root, document);
		} else if (commandStr.equals("CreateFolder")) {
			String newFolderStr = request.getParameter("NewFolderName");
			File newFolder = new File(currentDir, newFolderStr);
			String retValue = "110";

			if (newFolder.exists()) {
				retValue = "101";
			} else {
				try {
					boolean dirCreated = newFolder.mkdir();
					if (dirCreated)
						retValue = "0";
					else
						retValue = "102";
				} catch (SecurityException sex) {
					retValue = "103";
				}

			}
			setCreateFolderResponse(retValue, root, document);
		}

		document.getDocumentElement().normalize();
		try {
			TransformerFactory tFactory = TransformerFactory.newInstance();
			Transformer transformer = tFactory.newTransformer();

			DOMSource source = new DOMSource(document);

			StreamResult result = new StreamResult(out);
			transformer.transform(source, result);
			/*
			 * if (debug) { StreamResult dbgResult = new
			 * StreamResult(System.out); transformer.transform(source,
			 * dbgResult); System.out.println(""); System.out.println("--- END
			 * DOGET ---"); }
			 */
		} catch (Exception ex) {
			logger.error(
					"Error while transforming DOM to HttpServletResponse: "
							+ ex.getMessage(), ex);
		}

		out.flush();
		out.close();
		logger.debug("Successfull ended #doGet!");
	}

	/**
	 * Manage the Post requests (FileUpload).<br>
	 * 
	 * The servlet accepts commands sent in the following format:<br>
	 * connector?Command=FileUpload&Type=ResourceType&CurrentFolder=FolderPath<br>
	 * <br>
	 * It store the file (renaming it in case a file with the same name exists)
	 * and then return an HTML file with a javascript command in it.
	 * 
	 */
	public void doPost(HttpServletRequest request, HttpServletResponse response)
			throws ServletException, IOException {
		logger.debug("Entering #doPost");

		response.setCharacterEncoding("UTF-8");
		response.setContentType("text/html; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();

		String commandStr = request.getParameter("Command");
		String typeStr = request.getParameter("Type");
		String currentFolderStr = request.getParameter("CurrentFolder");

		logger.debug("Parameter Command: {}", commandStr);
		logger.debug("Parameter CurrentFolder: {}", currentFolderStr);
		logger.debug("Parameter Type: {}", typeStr);

		if (isEmpty(commandStr) && isEmpty(currentFolderStr)) {
			commandStr = "QuickUpload";
			currentFolderStr = "/";
		}

		UploadResponse ur = null;

		if (isEmpty(commandStr) || isEmpty(currentFolderStr)
				|| isEmpty(typeStr)) {
			ur = new UploadResponse(UploadResponse.EN_ERROR);
			ur.setCustomMessage(UploadResponse.CM_HTTP_400);
		} else if (!commandStr.matches("(File|Quick)Upload"))
			ur = UploadResponse.UR_SECURITY_ERROR;
		else {

			String currentPath = constructTypeBasedFolderString(typeStr,
					currentFolderStr);
			String currentDirPath = getServletContext()
					.getRealPath(currentPath);

			if (!isValidPath(currentFolderStr)
					|| !(new File(currentDirPath).exists())) {
				ur = new UploadResponse(UploadResponse.EN_ERROR);
				ur.setCustomMessage(UploadResponse.CM_HTTP_400);
			} else {

				String newFilename = null;
				FileItemFactory factory = new DiskFileItemFactory();
				ServletFileUpload upload = new ServletFileUpload(factory);
				try {
					List<FileItem> items = (List<FileItem>) upload
							.parseRequest(request);

					// We upload only one file at the same time
					FileItem uplFile = items.get(0);
					String filename = FilenameUtils.getName(uplFile.getName());
					String baseName = FilenameUtils.removeExtension(filename);
					String extension = FilenameUtils.getExtension(filename);

					boolean validExtension = ExtensionsHandler.isAllowed(
							typeStr, extension);

					if (!validExtension)
						ur = UploadResponse.UR_INVALID_EXTENSION;
					else {

						// TODO check if forceSingleExtension is enabled
						if (false) {
							filename = forceSingleExtension(filename);
							baseName = FilenameUtils.removeExtension(filename);
						}

						File pathToSave = new File(currentDirPath, filename);
						int counter = 1;
						while (pathToSave.exists()) {
							newFilename = baseName + "(" + counter + ")" + "."
									+ extension;
							pathToSave = new File(currentDirPath, newFilename);
							counter++;
						}

						uplFile.write(pathToSave);
						if (isEmpty(newFilename)) {
							ur = new UploadResponse(UploadResponse.EN_OK,request.getContextPath()+currentPath+filename);
						} else {
							ur = new UploadResponse(UploadResponse.EN_RENAMED,request.getContextPath()+currentPath+newFilename,newFilename);
						}
					}
				} catch (FileUploadException ex) {
					ur = new UploadResponse(UploadResponse.EN_ERROR);
					ur.setCustomMessage(UploadResponse.CM_HTTP_400);
				} catch (Exception e) {
					ur = UploadResponse.UR_SECURITY_ERROR;
				}
			}
		}
		out.print(ur);
		out.flush();
		out.close();

		logger.debug("Exiting #doPost");
	}

	private void setCreateFolderResponse(String retValue, Node root,
			Document doc) {
		Element myEl = doc.createElement("Error");
		myEl.setAttribute("number", retValue);
		root.appendChild(myEl);
	}

	private void getFolders(File dir, Node root, Document doc) {
		Element folders = doc.createElement("Folders");
		root.appendChild(folders);
		File[] fileList = dir
				.listFiles((FileFilter) DirectoryFileFilter.DIRECTORY);
		for (File file : fileList) {
			Element myEl = doc.createElement("Folder");
			myEl.setAttribute("name", file.getName());
			folders.appendChild(myEl);
		}
	}

	private void getFiles(File dir, Node root, Document doc) {
		Element files = doc.createElement("Files");
		root.appendChild(files);
		File[] fileList = dir.listFiles((FileFilter) FileFileFilter.FILE);

		for (File file : fileList) {
			Element myEl = doc.createElement("File");
			myEl.setAttribute("name", file.getName());
			myEl.setAttribute("size", String.valueOf(file.length() / 1024));
			files.appendChild(myEl);
		}
	}

	private Node createCommonXml(Document doc, String commandStr,
			String typeStr, String currentPath, String currentUrl) {
		Element root = doc.createElement("Connector");
		doc.appendChild(root);
		root.setAttribute("command", commandStr);
		root.setAttribute("resourceType", typeStr);

		Element myEl = doc.createElement("CurrentFolder");
		myEl.setAttribute("path", currentPath);
		myEl.setAttribute("url", currentUrl);
		root.appendChild(myEl);

		return root;
	}

	private String constructTypeBasedFolderString(final String fileType,
			final String currentFolderString) {
		StringWriter retval = new StringWriter();
		retval.append(ConfigurationHandler.getBaseDir());
		retval.append(ConfigurationHandler.getSubDirForType(fileType));
		retval.append(currentFolderString);
		return replaceAll(retval.toString(), "//", "/");
	}

}
