/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */

package net.fckeditor;

import javax.servlet.http.HttpServletRequest;

import net.fckeditor.handlers.ConfigurationHandler;
import net.fckeditor.tool.Compatibility;
import net.fckeditor.tool.Utils;
import net.fckeditor.tool.XHtmlTagTool;

/**
 * FCKeditor control class.<br>
 * It's the container for all properties and the class that generate the output based on browser capabilities and
 * configurations passed by the developer.
 * 
 * @version $Id: FCKeditor.java 1455 2008-02-01 19:59:38Z th-schwarz $
 */
public class FCKeditor {

    private FCKeditorConfigurations oConfig;
    private String instanceName;
    private String value;
    private String basePath;
    private String toolbarSet;
    private String width;
    private String height;

    private HttpServletRequest request;

    /**
     * Initialize the object setting all basic configurations.<br>
     * 
     * The basePath is context root + {@link ConfigurationHandler#getFckEditorDir()}.
     * 
     * @param request
     *                request object
     * @param instanceName
     *                unique name
     * @param width
     *                width
     * @param height
     *                height
     * @param toolbarSet
     *                toolbarSet name
     * @param value
     *                initial value
     */
    public FCKeditor(final HttpServletRequest request, final String instanceName, final String width,
	    final String height, final String toolbarSet, final String value) {
	this.request = request;
	// TODO muss das mit einem Slash abgeschlossen werden?
	this.basePath = request.getContextPath() + ConfigurationHandler.getFckEditorDir() + "/";
	this.instanceName = instanceName;

	// set defaults, if required
	this.value = (value == null) ? "" : value;
	this.toolbarSet = (toolbarSet == null) ? ConfigurationHandler.getFckEditorToolbarSet() : toolbarSet;
	this.width = (width == null) ? ConfigurationHandler.getFckEditorWidth() : width;
	this.height = (height == null) ? ConfigurationHandler.getFckEditorHeight() : height;

	oConfig = new FCKeditorConfigurations();
    }

    /**
     * Just a wrapper to {@link FCKeditor}.
     * 
     * @param req
     *                request object
     * @param parInstanceName
     *                unique name
     */
    public FCKeditor(final HttpServletRequest req, final String parInstanceName) {
	this(req, parInstanceName, null, null, null, null);
    }

    /**
     * Get the unique name of the editor
     * 
     * @return name
     */
    public String getInstanceName() {
	return instanceName;
    }

    /**
     * Set the unique name of the editor
     * 
     * @param value
     *                name
     */
    public void setInstanceName(String value) {
	instanceName = value;
    }

    /**
     * Get the initial value to be edited.<br>
     * In HTML code
     * 
     * @return value
     */
    public String getValue() {
	return value;
    }

    /**
     * Set the initial value to be edited.<br>
     * In HTML code
     * 
     * @param value
     *                value
     */
    public void setValue(String value) {
	this.value = value;
    }

    /**
     * Get the dir where the FCKeditor files reside on the server
     * 
     * @return path
     */
    public String getBasePath() {
	return basePath;
    }

    /**
     * Set the dir where the FCKeditor files reside on the server.<br>
     * <b>Remarks</b>:<br>
     * Avoid using relative paths. It is preferable to set the base path starting from the root (/).<br>
     * Always finish the path with a slash (/).
     * 
     * @param value
     *                path
     */
    public void setBasePath(String value) {
	basePath = value;
    }

    /**
     * Get the name of the toolbar to display
     * 
     * @return toolbar name
     */
    public String getToolbarSet() {
	return toolbarSet;
    }

    /**
     * Set the name of the toolbar to display
     * 
     * @param value
     *                toolbar name
     */
    public void setToolbarSet(String value) {
	toolbarSet = value;
    }

    /**
     * Get the width of the textarea
     * 
     * @return width
     */
    public String getWidth() {
	return width;
    }

    /**
     * Set the width of the textarea
     * 
     * @param value
     *                width
     */
    public void setWidth(String value) {
	width = value;
    }

    /**
     * Get the height of the textarea
     * 
     * @return height
     */
    public String getHeight() {
	return height;
    }

    /**
     * Set the height of the textarea
     * 
     * @param value
     *                height
     */
    public void setHeight(String value) {
	height = value;
    }

    /**
     * Get the advanced configuation set.<br>
     * Adding element to this collection you can override the settings specified in the config.js file.
     * 
     * @return configuration collection
     */
    public FCKeditorConfigurations getConfig() {
	return oConfig;
    }

    /**
     * Set the advanced configuation set.
     * 
     * @param value
     *                configuration collection
     */
    public void setConfig(FCKeditorConfigurations value) {
	oConfig = value;
    }

    private String escapeXml(String txt) {
	txt = txt.replaceAll("&", "&#38;");
	txt = txt.replaceAll("<", "&#60;");
	txt = txt.replaceAll(">", "&#62;");
	txt = txt.replaceAll("\"", "&#34;");
	txt = txt.replaceAll("'", "&#39;");
	return txt;
    }

    /**
     * Minimum implementation, see ticket #27 for detailed information.
     */
    public String create() {
	return createHtml();
    }
    
    /**
     * Generate the HTML Code for the editor. <br>
     * Evalute the browser capabilities and generate the editor if compatible, or a simple textarea otherwise.
     * 
     * @return html code
     */
    public String createHtml() {
	StringBuilder strEditor = new StringBuilder();

	strEditor.append("<div>");
	String encodedValue = escapeXml(value);

	if (Compatibility.check(request.getHeader("user-agent"))) {
	    strEditor.append(createInputForVariable(instanceName, instanceName, encodedValue));
	    
	    // create config html
	    String configStr = oConfig.getUrlParams();
	    if (Utils.isNotEmpty(configStr))
		//configStr = configStr.substring(1);
	    strEditor.append(createInputForVariable(null, instanceName.concat("___Config"), configStr));

	    // create IFrame
	    // TODO eventuell mit neuer Pfadstruktur untersuchen
	    String sLink = basePath + "editor/fckeditor.html?InstanceName=" + instanceName;
	    if (Utils.isNotEmpty(toolbarSet))
		sLink += "&Toolbar=" + toolbarSet;
	    XHtmlTagTool iframeTag = new XHtmlTagTool("iframe");
	    iframeTag.addAttribute("id", instanceName.concat("___Frame"));
	    iframeTag.addAttribute("src", sLink);
	    iframeTag.addAttribute("width", width);
	    iframeTag.addAttribute("height", height);
	    iframeTag.addAttribute("frameborder", "no");
	    iframeTag.addAttribute("scrolling", "no");
	    strEditor.append(iframeTag);

	} else {
	    XHtmlTagTool textareaTag = new XHtmlTagTool("textarea", encodedValue);
	    textareaTag.addAttribute("name", instanceName);
	    textareaTag.addAttribute("rows", "4");
	    textareaTag.addAttribute("cols", "40");
	    textareaTag.addAttribute("wrap", "virtual");
	    textareaTag.addAttribute("style", "width: ".concat(width).concat("; height: ").concat(height));
	}
	strEditor.append("</div>");
	return strEditor.toString();
    }
    
    private String createInputForVariable(final String name, final String id, final String value) {
	XHtmlTagTool tag = new XHtmlTagTool("input");
	if (Utils.isNotEmpty(id))
	    tag.addAttribute("id", id);
	if (Utils.isNotEmpty(name))
	    tag.addAttribute("name", name);
	tag.addAttribute("value", value);
	tag.addAttribute("type", "hidden");
	return tag.toString();
    }

    /**
     * Checks the compatibility of the browser of the current request.
     * 
     * @return True, if compatible or false.
     * @see Compatibility#check(String)
     */
    public boolean isCompatibleBrowser() {
	return (request != null) ? Compatibility.check(request.getHeader("user-agent")) : false;
    }
}
