/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.handlers;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import net.fckeditor.ResourceType;
import net.fckeditor.tool.Utils;

/**
 * Static object which manages the allowed and denied extensions for each file type. The
 * extensions are preset by the defaults defined in default.properties.
 * 
 * Hint: It's recomment use either allowed or denied extensions for one file type.
 * Never use both at the same time! That's why denied extensions of a file type will be 
 * deleted, if you set the allowed one and vice versa.
 * 
 * @version $Id: ExtensionsHandler.java 1432 2008-01-27 20:12:42Z th-schwarz $
 */
public class ExtensionsHandler {

	private static Map<ResourceType, Set<String>> extensionsAllowed = new HashMap<ResourceType, Set<String>>();
	private static Map<ResourceType, Set<String>> extensionsDenied = new HashMap<ResourceType, Set<String>>();

	static {
		// load defaults
		extensionsAllowed.put(ResourceType.File, Utils.getSet(ConfigurationHandler
		    .getDefaultProperty("connector.extensions.file.allowed")));
		extensionsDenied.put(ResourceType.File, Utils.getSet(ConfigurationHandler
		    .getDefaultProperty("connector.extensions.file.denied")));
		extensionsAllowed.put(ResourceType.Media, Utils.getSet(ConfigurationHandler
		    .getDefaultProperty("connector.extensions.media.allowed")));
		extensionsDenied.put(ResourceType.Media, Utils.getSet(ConfigurationHandler
		    .getDefaultProperty("connector.extensions.media.denied")));
		extensionsAllowed.put(ResourceType.Image, Utils.getSet(ConfigurationHandler
		    .getDefaultProperty("connector.extensions.image.allowed")));
		extensionsDenied.put(ResourceType.Image, Utils.getSet(ConfigurationHandler
		    .getDefaultProperty("connector.extensions.image.denied")));
		extensionsAllowed.put(ResourceType.Flash, Utils.getSet(ConfigurationHandler
		    .getDefaultProperty("connector.extensions.flash.allowed")));
		extensionsDenied.put(ResourceType.Flash, Utils.getSet(ConfigurationHandler
		    .getDefaultProperty("connector.extensions.flash.denied")));
	}

	/**
	 * Getter for the allowed extensions of a file type.
	 * 
	 * @param type
	 *          The file type.
	 * @return Set of allowed extensions or an empty set.
	 */
	public static Set<String> getExtensionsAllowed(final ResourceType type) {
		return extensionsAllowed.get(type);
	}

	/**
	 * Setter for the allowed extensions of a file type. The denied extensions will be cleared.<br>
	 * If 'extensionsList' is null, allowed extensions kept untouched.
	 * 
	 * @param type
	 *          The file type.
	 * @param extensionsList
	 *          Required format: <code>ext1&#124;ext2&#124;ext3</code>
	 */
	public static void setExtensionsAllowed(final ResourceType type, final String extensionsList) {
		if (extensionsList != null) {
			extensionsAllowed.put(type, Utils.getSet(extensionsList));
			extensionsDenied.get(type).clear();
		}
	}

	/**
	 * Getter for the denied extensions of a file type.
	 * 
	 * @param type
	 *          The file type.
	 * @return Set of denied extensions or an empty set.
	 */
	public static Set<String> getExtensionsDenied(final ResourceType type) {
		return extensionsDenied.get(type);
	}

	/**
	 * Setter for the denied extensions of a file type. The allowed extensions will be cleared.<br>
	 * If 'extensionsList' is null, denied extensions kept untouched.
	 * 
	 * @param type
	 *          The file type.
	 * @param extensionsList
	 *          Required format: <code>ext1&#124;ext2&#124;ext3</code>
	 */
	public static void setExtensionsDenied(final ResourceType type, final String extensionsList) {
		if (extensionsList != null) {
			extensionsDenied.put(type, Utils.getSet(extensionsList));
			extensionsAllowed.get(type).clear();
		}
	}

	/**
	 * Checks, if an extension is allowed for a file type.
	 * 
	 * @param type
	 * @param extension
	 * @return True, false. False is returned too, if 'type' or 'extensions' is null.
	 */
	public static boolean isAllowed(final ResourceType type, final String extension) {
		if (type == null || extension == null)
			return false;
		String ext = extension.toLowerCase();
		Set<String> allowed = extensionsAllowed.get(type);
		Set<String> denied = extensionsDenied.get(type);
		if (allowed.isEmpty())
			return (denied.contains(ext)) ? false : true;
		if (denied.isEmpty())
			return (allowed.contains(ext)) ? true : false;
		return false;
	}
}
