/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.handlers;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.util.Properties;

import javax.servlet.http.HttpServletRequest;

import net.fckeditor.SessionData;
import net.fckeditor.tool.Utils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This is a static configuration object, which provides access to the following default values:
 * <ul>
 * <li>The directory of the fckeditor.</li>
 * <li>The width and height of the fckeditor.</li>
 * <li>The toolbar of the fckeditor.</li>
 * <li>The base dir (user dir).</li>
 * <li>Some session based method calls. See comment above!</li>
 * </ul>
 * These values are preset by the defaults defined in default.properties and can be changed by
 * calling its corresponding setters. Keep in mind, that these values are applied to ALL instances
 * of the editor, except they are overwritten by the editor's tags of course. <br />
 * <br />
 * Something special is {@link SessionData}. The implemented class provides some session based data.
 * See {@link SessionData} for additional informations. For example, to get a user based base dir,
 * for each request {@link #getUserFilesPath(HttpServletRequest)} is calling. If 'sessionData' is set,
 * {@link SessionData#getBaseDir(HttpServletRequest)} will be returned, otherwise the default base
 * dir.
 * 
 * @version $Id:ConfigurationHandler.java 1099 2008-11-06 15:01:50Z th-schwarz $
 */
public class ConfigurationHandler {
	private static final Logger logger = LoggerFactory.getLogger(ConfigurationHandler.class);

	private static Properties defaultProperties = new Properties();
	private static String userFilesPath;
	private static SessionData sessionData = null;
	private static String basePath;
	private static String fckEditorHeight;
	private static String fckEditorWidth;
	private static String fckEditorToolbarSet;
	private static boolean forceSingleExtension;
	
	public static final String PROPERTY_MESSAGE_COMPATIBLE_BROWSER = "message.enabled_tag.compatible_browser.yes";
	public static final String PROPERTY_MESSAGE_NOT_COMPATIBLE_BROWSER = "message.enabled_tag.compatible_browser.no";
	public static final String PROPERTY_MESSAGE_FILE_UPLOAD_ENABLED = "message.enabled_tag.connector.file_upload.enabled";
	public static final String PROPERTY_MESSAGE_FILE_UPLOAD_DISABLED = "message.enabled_tag.connector.file_upload.disalbed";
	public static final String PROPERTY_MESSAGE_FILE_BROWSING_ENABLED = "message.enabled_tag.connector.file_browsing.enabled";
	public static final String PROPERTY_MESSAGE_FILE_BROWSING_DISABLED = "message.enabled_tag.connector.file_browsing.disabled";

	static {
		// load defaults
		try {
			defaultProperties.load(new BufferedInputStream(ConfigurationHandler.class
			        .getResourceAsStream("default.properties")));
		} catch (IOException e) {
			logger.error("Error while loading the default properties!", e);
			throw new RuntimeException("Can't load default properties!", e);
		}
		userFilesPath = defaultProperties.getProperty("connector.userFilesPath");
		basePath = defaultProperties.getProperty("fckeditor.basePath");
		fckEditorWidth = defaultProperties.getProperty("fckeditor.width");
		fckEditorHeight = defaultProperties.getProperty("fckeditor.height");
		fckEditorToolbarSet = defaultProperties.getProperty("fckeditor.toolbarSet");
		forceSingleExtension = Boolean.valueOf(defaultProperties
		        .getProperty("connector.forceSingleExtension"));

		logger.info("Default properties loaded and initialized successfully.");
	}

	/**
	 * Getter for the base dir (using for user files).
	 * 
	 * @return {@link SessionData#getBaseDir(HttpServletRequest)} or the default base dir, if
	 *         {@link SessionData}} isn't set.
	 */
	public static String getUserFilesPath(final HttpServletRequest servletRequest) {
		if (sessionData == null || sessionData.getBaseDir(servletRequest) == null)
			return getDefaultUserFilesPath();
		return sessionData.getBaseDir(servletRequest);
	}

	/**
	 * Getter for the default userFilesPath.
	 * 
	 * @return
	 */
	public static String getDefaultUserFilesPath() {
		return userFilesPath;
	}

	/**
	 * Setter for the base dir (using for user files). If param 'userFilesPath' is empty, the property
	 * leaves untouched.
	 * 
	 * @param userFilesPath
	 *            relative to the context root (no leading or ending /).
	 */
	public static void setUserFilesPath(final String userFilesPath) {
		if (Utils.isNotEmpty(userFilesPath))
			ConfigurationHandler.userFilesPath = userFilesPath;
	}

	/**
	 * Constructs an object which implemented the interface {@link SessionData}. If the
	 * constuction fails, any user action of the connector is disabled.
	 * 
	 * @param fqcn
	 *            The fully-qualified class name.
	 */
	public static void constructSessionDataObject(final String fqcn) {
		if (Utils.isEmpty(fqcn)) {
			logger
			        .warn("Class name is empty, any user action of the ConnectorServlet is disabled!");
			return;
		}
		try {
			@SuppressWarnings("unchecked")
			Class clazz = Class.forName(fqcn);
			sessionData = (SessionData) clazz.newInstance();
		} catch (Exception e) {
			logger.error("Couldn't instanciate the class [".concat(fqcn).concat(
			        "], any user action of the ConnectorServlet is disabled!"), e);
		}
	}

	/**
	 * Getter for the dir of the fckeditor.
	 * 
	 * @return Dir of the fckeditor relative to the context root.
	 */
	public static String getBasePath() {
		return basePath;
	}

	/**
	 * Setter for the dir of the fckeditor.
	 * 
	 * @param basePath
	 *            relative to the context root (no leading or ending /).
	 */
	public static void setFckEditorDir(final String fckEditorDir) {
		ConfigurationHandler.basePath = fckEditorDir;
	}

	/**
	 * Getter for the default height of the fckeditor.
	 * 
	 * @return Default height of the fckeditor.
	 */
	public static String getFckEditorHeight() {
		return fckEditorHeight;
	}

	/**
	 * Setter for the default height of the fckeditor.
	 * 
	 * @param fckEditorHeight
	 *            the default height of the fckeditor.
	 */
	public static void setFckEditorHeight(final String fckEditorHeight) {
		ConfigurationHandler.fckEditorHeight = fckEditorHeight;
	}

	/**
	 * Getter for the default width of the fckeditor.
	 * 
	 * @return Default width of the fckeditor.
	 */
	public static String getFckEditorWidth() {
		return fckEditorWidth;
	}

	/**
	 * Setter for the default width of the fckeditor.
	 * 
	 * @param fckEditorWidth
	 *            the default width of the fckeditor.
	 */
	public static void setFckEditorWidth(final String fckEditorWidth) {
		ConfigurationHandler.fckEditorWidth = fckEditorWidth;
	}

	/**
	 * Getter of the name of the default toolbarset of the fckeditor.
	 * 
	 * @return Name of the default toolbar set.
	 */
	public static String getFckEditorToolbarSet() {
		return fckEditorToolbarSet;
	}

	/**
	 * Getter of the name of the default toolbarset of the fckeditor.
	 * 
	 * @param fckEditorToolbarSet
	 *            the name of the toolbarset.
	 */
	public static void setFckEditorToolbarSet(final String fckEditorToolbarSet) {
		ConfigurationHandler.fckEditorToolbarSet = fckEditorToolbarSet;
	}

	/**
	 * Getter for the default handling of single extensions.
	 * 
	 * @return the forceSingleExtension
	 */
	public static boolean isForceSingleExtension() {
		return forceSingleExtension;
	}

	/**
	 * Setter for the default handling of single extensions.
	 * 
	 * @param forceSingleExtension
	 *            the forceSingleExtension to set
	 */
	public static void setForceSingleExtension(boolean forceSingleExtension) {
		ConfigurationHandler.forceSingleExtension = forceSingleExtension;
	}

	/**
	 * Just a wrapper to {@link #setForceSingleExtension(boolean)}. If argument
	 * 'forceSingleExtension' is empty, the property leaves untouched.
	 * 
	 * @param forceSingleExtension
	 */
	public static void setForceSingleExtension(final String forceSingleExtension) {
		if (Utils.isNotEmpty(forceSingleExtension))
			setForceSingleExtension(Boolean.parseBoolean(forceSingleExtension));
	}

	/**
	 * Getter for a default property with the name 'key'.
	 * 
	 * @param key
	 * @return The required property or null, if doesn't exists.
	 */
	public static String getDefaultProperty(final String key) {
		return defaultProperties.getProperty(key);
	}

	/**
	 * Getter for the dafault properties.
	 * 
	 * @return Default properties.
	 */
	public static Properties getDefaultProperties() {
		return defaultProperties;
	}

	/**
	 * Just a wrapper to {@link SessionData.isEnabledForFileUpload(HttpServletRequest)}.
	 * 
	 * @param servletRequest
	 * @return {@link SessionData#isEnabledForFileUpload(HttpServletRequest)} or false, if
	 *         sessionData isn't set.
	 */
	public static boolean isEnabledForFileUpload(final HttpServletRequest servletRequest) {
		return (sessionData != null && sessionData.isEnabledForFileUpload(servletRequest));
	}

	/**
	 * Just a wrapper to {@link SessionData.isEnabledForFileBrowsing(HttpServletRequest)}.
	 * 
	 * @param servletRequest
	 * @return {@link SessionData#isEnabledForFileBrowsing(HttpServletRequest)} or false, if
	 *         sessionData isn't set.
	 */
	public static boolean isEnabledForFileBrowsing(final HttpServletRequest servletRequest) {
		return (sessionData != null && sessionData.isEnabledForFileBrowsing(servletRequest));
	}
}
