/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.connector;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.fckeditor.handlers.ExtensionsHandler;
import net.fckeditor.handlers.ConnectorHandler;
import net.fckeditor.handlers.ResourceTypeHandler;
import net.fckeditor.handlers.SessionDataHandler;
import net.fckeditor.response.UploadResponse;
import net.fckeditor.response.XmlResponse;
import net.fckeditor.tool.Utils;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Servlet to upload and browse files.<br>
 * 
 * This servlet accepts 4 commands used to retrieve and create files and folders from a server
 * directory. The allowed commands are:
 * <ul>
 * <li>GetFolders: Retrive the list of directory under the current folder
 * <li>GetFoldersAndFiles: Retrive the list of files and directory under the current folder
 * <li>CreateFolder: Create a new directory under the current folder
 * <li>FileUpload: Send a new file to the server (must be sent with a POST)
 * </ul>
 * 
 * @version $Id: ConnectorServlet.java 1636 2008-02-25 19:08:23Z th-schwarz $
 */
public class ConnectorServlet extends HttpServlet {

	private static final long serialVersionUID = -5742008970929377161L;
	private static final Logger logger = LoggerFactory.getLogger(ConnectorServlet.class);

	/**
	 * Initialize the servlet.<br>
	 * The default directory for user files will be constructed.
	 */
	public void init() throws ServletException, IllegalArgumentException {
		// check, if 'baseDir' exists
		String realDefaultUserFilesPath = getServletContext().getRealPath(
		        ConnectorHandler.getDefaultUserFilesPath());

		File defaultUserFilesDir = new File(realDefaultUserFilesPath);
		if (!defaultUserFilesDir.exists()) {
			defaultUserFilesDir.mkdirs();
		}
		logger.info("ConnectorServlet successfull initialized!");
	}

	/**
	 * Manage the Get requests (GetFolders, GetFoldersAndFiles, CreateFolder).<br>
	 * 
	 * The servlet accepts commands sent in the following format:<br>
	 * connector?Command=CommandName&Type=ResourceType&CurrentFolder=FolderPath<br>
	 * <br>
	 * It executes the command and then return the results to the client in XML format.
	 * 
	 */
	public void doGet(HttpServletRequest request, HttpServletResponse response)
	        throws ServletException, IOException {
		logger.debug("Entering ConnectorServlet#doGet");

		response.setCharacterEncoding("UTF-8");
		response.setContentType("application/xml; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();

		String commandStr = request.getParameter("Command");
		String typeStr = request.getParameter("Type");
		String currentFolderStr = request.getParameter("CurrentFolder");
		ResourceTypeHandler fileType = ResourceTypeHandler.getDefaultResourceType(typeStr);

		logger.debug("Parameter Command: {}", commandStr);
		logger.debug("Parameter CurrentFolder: {}", currentFolderStr);
		logger.debug("Parameter Type: {}", typeStr);

		String currentPath = constructTypeBasedFolderString(fileType, currentFolderStr, request);
		String currentDirPath = getServletContext().getRealPath(currentPath);

		File currentDir = new File(currentDirPath);
		if (!currentDir.exists() && SessionDataHandler.isEnabledForFileBrowsing(request)) {
			currentDir.mkdirs();
			logger.debug("Dir successfully created: {}", currentDirPath);
		}

		XmlResponse xr = new XmlResponse(commandStr, fileType, currentFolderStr, currentPath,
		        request);

		if (!SessionDataHandler.isEnabledForFileBrowsing(request)) {
			xr.setError(1, "The current user isn't authorized for file browsing!");
		} else if (commandStr.equals("GetFolders")) {
			xr.setFolders(currentDir);
		} else if (commandStr.equals("GetFoldersAndFiles")) {
			xr.setFolders(currentDir);
			xr.setFiles(currentDir);
		} else if (commandStr.equals("CreateFolder")) {
			String newFolderStr = request.getParameter("NewFolderName");
			File newFolder = new File(currentDir, newFolderStr);
			int errorNumber = 110;

			if (newFolder.exists()) {
				errorNumber = 101;
			} else {
				try {
					errorNumber = (newFolder.mkdir()) ? 0 : 102;
				} catch (SecurityException sex) {
					errorNumber = 103;
				}
			}
			xr.setError(errorNumber);
		}

		out.print(xr);
		out.flush();
		out.close();
		logger.debug("Exiting ConnectorServlet#doGet");
	}

	/**
	 * Manage the Post requests (FileUpload).<br>
	 * 
	 * The servlet accepts commands sent in the following format:<br>
	 * connector?Command=FileUpload&Type=ResourceType&CurrentFolder=FolderPath<br>
	 * <br>
	 * It store the file (renaming it in case a file with the same name exists) and then return an
	 * HTML file with a javascript command in it.
	 * 
	 */
	public void doPost(HttpServletRequest request, HttpServletResponse response)
	        throws ServletException, IOException {
		logger.debug("Entering Connector#doPost");

		response.setCharacterEncoding("UTF-8");
		response.setContentType("text/html; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();

		String commandStr = request.getParameter("Command");
		String typeStr = request.getParameter("Type");
		String currentFolderStr = request.getParameter("CurrentFolder");

		logger.debug("Parameter Command: {}", commandStr);
		logger.debug("Parameter CurrentFolder: {}", currentFolderStr);
		logger.debug("Parameter Type: {}", typeStr);

		if (!ResourceTypeHandler.isValid(typeStr))
			logger.warn("Unknown Type requested: {}", typeStr);

		ResourceTypeHandler fileType = ResourceTypeHandler.getDefaultResourceType(typeStr);

		// if this is a QuickUpload-Request, 'commandStr' and 'currentFolderStr' are empty and have
		// to preset and construct the full resource type path !!
		if (Utils.isEmpty(commandStr) && Utils.isEmpty(currentFolderStr)) {
			commandStr = "QuickUpload";
			currentFolderStr = "/";
			String quickUploadFolder = constructTypeBasedFolderString(fileType, currentFolderStr,
			        request);
			File file = new File(getServletContext().getRealPath(quickUploadFolder));
			if (!file.exists())
				file.mkdirs();
		}

		UploadResponse ur = null;

		if (!Utils.isValidPath(currentFolderStr)) {
			ur = new UploadResponse(UploadResponse.EN_SECURITY_ERROR, null, null,
			        "'currentFolderStr' isn't valid!");
		} else if (!SessionDataHandler.isEnabledForFileUpload(request)) {
			ur = new UploadResponse(UploadResponse.EN_SECURITY_ERROR, null, null,
			        "The current user isn't authorized for uploading files!");
		} else if (Utils.isEmpty(commandStr) || Utils.isEmpty(currentFolderStr)
		        || Utils.isEmpty(typeStr))
			ur = UploadResponse.UR_BAD_REQUEST;
		else if (!commandStr.matches("(File|Quick)Upload"))
			ur = UploadResponse.UR_SECURITY_ERROR;
		else {

			String currentPath = constructTypeBasedFolderString(fileType, currentFolderStr, request);
			File currentDirPath = new File(getServletContext().getRealPath(currentPath));

			if (!Utils.isValidPath(currentFolderStr) || !currentDirPath.exists())
				ur = UploadResponse.UR_BAD_REQUEST;
			else {

				String newFilename = null;
				FileItemFactory factory = new DiskFileItemFactory();
				ServletFileUpload upload = new ServletFileUpload(factory);
				try {
					List<FileItem> items = upload.parseRequest(request);

					// We upload only one file at the same time
					FileItem uplFile = items.get(0);
					String filename = FilenameUtils.getName(uplFile.getName());
					String baseName = FilenameUtils.removeExtension(filename);
					String extension = FilenameUtils.getExtension(filename);

					boolean validExtension = ExtensionsHandler.isAllowed(fileType, extension);

					if (!validExtension)
						ur = UploadResponse.UR_INVALID_EXTENSION;
					else {

						if (ConnectorHandler.isForceSingleExtension()) {
							filename = Utils.forceSingleExtension(filename);
							baseName = FilenameUtils.removeExtension(filename);
						}

						// construct an unique file
						File pathToSave = new File(currentDirPath, filename);
						int counter = 1;
						while (pathToSave.exists()) {
							newFilename = baseName.concat("(").concat(String.valueOf(counter))
							        .concat(")").concat(".").concat(extension);
							pathToSave = new File(currentDirPath, newFilename);
							counter++;
						}

						if (Utils.isEmpty(newFilename)) {
							ur = new UploadResponse(UploadResponse.EN_OK, Utils
							        .constructServerAddress(request, currentPath).concat(filename));
						} else {
							ur = new UploadResponse(UploadResponse.EN_RENAMED, Utils
							        .constructServerAddress(request, currentPath).concat(
							                newFilename), newFilename);
						}

						// secure image check
						if (fileType.equals(ResourceTypeHandler.IMAGE) && ConnectorHandler.isSecureImageUploads()) {
							boolean check = Utils.isImage(uplFile.getInputStream());
							if (check) {
								uplFile.write(pathToSave);
							} else {
								ur = UploadResponse.UR_INVALID_EXTENSION;
							}
						} else {
							uplFile.write(pathToSave);
						}

					}
				} catch (FileUploadException ex) {
					ur = UploadResponse.UR_BAD_REQUEST;
				} catch (Exception e) {
					ur = UploadResponse.UR_SECURITY_ERROR;
				}
			}
		}
		out.print(ur);
		out.flush();
		out.close();

		logger.debug("Exiting Connector#doPost");
	}

	private String constructTypeBasedFolderString(final ResourceTypeHandler fileType,
	        final String currentFolderString, final HttpServletRequest request) {
		StringBuffer sb = new StringBuffer();
		sb.append(ConnectorHandler.getUserFilesPath(request));
		sb.append(fileType.getPath());
		sb.append(currentFolderString);
		return Utils.replaceAll(sb.toString(), "//", "/");
	}

}
