﻿/*
 * CKEditor - The text editor for Internet - http://ckeditor.com
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 *
 * == BEGIN LICENSE ==
 *
 * Licensed under the terms of any of the following licenses at your
 * choice:
 *
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 *
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 *
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 *
 * == END LICENSE ==
 */

/**
 * This is a base class for classes and objects that require event handling
 * features.
 * @constructor
 * @example
 */
CKEDITOR.event = function()
{
	this._events = {};
};

CKEDITOR.event.implementOn = function( targetObject )
{
	CKEDITOR.event.call( targetObject );
	
	for ( var prop in CKEDITOR.event.prototype )
	{
		if ( targetObject[ prop ] == undefined )
			targetObject[ prop ] = CKEDITOR.event.prototype[ prop ];
	}
};

CKEDITOR.event.prototype = (function()
{
	var eventEntry = function( eventName )
	{
		this.name = eventName;
		this.listeners = [];
	}

	eventEntry.prototype =
	{
		// Get the listener index for a specified function.
		// Returns -1 if not found.
		getListenerIndex : function( listenerFunction )
		{
			for ( var i = 0, listeners = this.listeners ; i < listeners.length ; i++ )
			{
				if ( listeners[i].fn == listenerFunction )
					return i;
			}
			return -1;
		}
	};

	return /** @lends CKEDITOR.event.prototype */ {
		on  : function( eventName, listenerFunction, scopeObj, listenerData, priority )
		{
			// Get the event entry (create it if needed).
			var event = this._events[ eventName ] || ( this._events[ eventName ] = new eventEntry( eventName ) );

			if ( event.getListenerIndex( listenerFunction ) < 0 )
			{
				// Get the listeners.
				var listeners = event.listeners;

				// Fill the scope.
				if ( !scopeObj )
					scopeObj = this;

				// Default the priority, if needed.
				if ( isNaN( priority ) )
					priority = 10;

				// Create the function to be fired for this listener.
				var listenerFirer = function( editor, publisherData, stopFn, cancelFn )
				{
					var ev =
					{
						name : eventName,
						sender : this,
						editor : editor,
						data : publisherData,
						listenerData : listenerData,
						stop : stopFn,
						cancel : cancelFn
					};
					
					listenerFunction.call( scopeObj, ev );
					
					return ev.data;
				};
				listenerFirer.fn = listenerFunction;
				listenerFirer.priority = priority;

				// Search for the right position for this new listener, based on its
				// priority.
				for ( var i = listeners.length - 1 ; i >= 0 ; i-- )
				{
					// Find the item which should be before the new one.
					if ( listeners[ i ].priority <= priority )
					{
						// Insert the listener in the array.
						listeners.splice( i + 1, 0, listenerFirer );
						return;
					}
				}

				// If no position has been found (or zero length), put it in
				// the front of list.
				listeners.unshift( listenerFirer );
			}
		},

		fire : function( eventName, data, editor )
		{
			// Create the function that marks the event as stopped.
			var stopped = false;
			var stopEvent = function()
			{
				stopped = true;
			};

			// Create the function that marks the event as cancelled.
			var cancelled = false;
			var cancelEvent = function()
			{
				cancelled = true;
			};

			// Get the event entry.
			var event = this._events[ eventName ];

			if ( event )
			{
				// Loop through all listeners.
				for ( var i = 0, listeners = event.listeners ; i < listeners.length ; i++ )
				{
					// Call the listener, passing the event data.
					data = listeners[i].call( this, editor, data, stopEvent, cancelEvent );

					// No further calls is stopped or cancelled.
					if ( stopped || cancelled )
						break;
				}
			}

			return cancelled || ( typeof data == 'undefined' ? false : data );
		},
		
		fireOnce : function( eventName, data, editor )
		{
			var ret = this.fire( eventName, editor, data );
			delete this._events[ eventName ];
			return ret;
		},

		removeListener : function( eventName, listenerFunction )
		{
			// Get the event entry.
			var event = this._events[ eventName ];

			if ( event )
			{
				var index = event.getListenerIndex( listenerFunction );
				if ( index >= 0 )
					event.listeners.splice( index, 1 );
			}
		}
	};
})();

// The following is for documentation purposes only. It documents a virtual
// CKEDITOR.eventInfo class that contains the defintions of event object passed
// to event listeners.

/**
 * This class is not really part of the API. It just illustrates the features
 * of the event object passed to event listeners by a {@link CKEDITOR.event}
 * based object.
 * @name CKEDITOR.eventInfo
 * @constructor
 */

/**
 * The event name.
 * @name CKEDITOR.eventInfo.prototype.name
 * @field
 * @type String
 */

/**
 * The object that publishes (sends) the event.
 * @name CKEDITOR.eventInfo.prototype.sender
 * @field
 * @type Object
 */

/**
 * The editor instance that holds the sender. May be the same as sender. May be
 * null if the sender is not part of an editor instance, like a component
 * running in standalone mode.
 * @name CKEDITOR.eventInfo.prototype.editor
 * @field
 * @type CKEDITOR.editor
 */

/**
 * Any kind of additional data. Its format and usage is event dependent.
 * @name CKEDITOR.eventInfo.prototype.data
 * @field
 * @type Object
 */

/**
 * Any extra data appended during the listener registration.
 * @name CKEDITOR.eventInfo.prototype.listenerData
 * @field
 * @type Object
 */

/**
 * Indicates that no further listeners are to be called.
 * @name CKEDITOR.eventInfo.prototype.stop
 * @function
 */

/**
 * Indicates that the event is to be cancelled (if cancelable).
 * @name CKEDITOR.eventInfo.prototype.cancel
 * @function
 */
