﻿/*
 * CKEditor - The text editor for Internet - http://ckeditor.com
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 *
 * == BEGIN LICENSE ==
 *
 * Licensed under the terms of any of the following licenses at your
 * choice:
 *
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 *
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 *
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 *
 * == END LICENSE ==
 */

 /**
 * Represents an editor instance.
 * @constructor
 * @param {CKEDITOR.dom.element} element The original element replaced by this
 *		editor instance.
 */
CKEDITOR.editor = ( function()
{
	// The counter for automatic instance names.
	var nameCounter = 0;

	var getNewName = function()
	{
		var name = 'editor' + nameCounter++;
		return CKEDITOR.instances[ name ] ? getNewName() : name;
	};
	
	// ##### START: Config Privates

	// These function loads custom configuration files and cache the
	// CKEDITOR.editorConfig functions defined on them, so there is no need to
	// download them more than once for several instances.
	var loadConfigLoaded = {};
	var loadConfig = function( editor )
	{
		var customConfig = editor.config.customConfig;

		// Check if there is a custom config to load.
		if ( !customConfig )
			return false;

		var loadedConfig = loadConfigLoaded[ customConfig ] || ( loadConfigLoaded[ customConfig ] = { editors : [] } );

		// If the custom config has already been downloaded, reuse it.
		if ( loadedConfig.fn )
		{
			// Call the cached CKEDITOR.editorConfig defined in the custom
			// config file for the editor instance depending on it.
			loadedConfig.fn.call( editor, editor );

			// If there is no other customConfig in the chain, fire the
			// "configloaded" event.
			if ( editor.config.customConfig == customConfig || !loadConfig( editor ) )
				editor.fireOnce( 'customconfigloaded' );
		}
		else
		{
			// Add the editor to the list of editors waiting for this config.
			loadedConfig.editors.push( editor );

			// Load the custom configuration file.
			CKEDITOR.scriptLoader.load( customConfig, function()
				{
					// If the CKEDITOR.editorConfig function has been properly
					// defined in the custom configuration file, cache it.
					if ( CKEDITOR.editorConfig )
						loadedConfig.fn = CKEDITOR.editorConfig;
					else
						loadedConfig.fn = function(){};

					delete CKEDITOR.editorConfig;

					CKEDITOR.tools.each( loadedConfig.editors, function( editor )
						{
							// Call the load config again. This time the custom
							// config is already cached and so it will get loaded.
							loadConfig( editor );
						});

					delete loadedConfig.editors;
				} );
		}

		return true;
	};

	var initConfig = function( editor, instanceConfig )
	{
		// Setup the lister for the "customconfigloaded" event.
		editor.on( 'customconfigloaded', function()
			{
				// Overwrite the settings from the in-page config.
				if ( instanceConfig )
					CKEDITOR.tools.extend( editor.config, instanceConfig, true );

				// Fire the "configloaded" event.
				editor.fireOnce( 'configloaded' );
				
				// Start loading the plugins.
				loadPlugins( editor );
			});

		// The instance config may override the customConfig setting to avoid
		// loading the default ~/config.js file.
		if ( instanceConfig && instanceConfig.customConfig != undefined )
			editor.config.customConfig = instanceConfig.customConfig;

		// Load configs from the custom configuration files.
		if ( !loadConfig( editor ) )
			editor.fireOnce( 'customconfigloaded' );
	};

	// Basic config class to inherit the default settings from CKEDITOR.config.
	var config = function()
	{}
	config.prototype = CKEDITOR.config;

	// ##### END: Config Privates

	var loadPlugins = function( editor )
	{
		// Load all plugins defined in the "plugins" setting.
		CKEDITOR.plugins.load( editor.config.plugins.split( ',' ), function( plugins )
			{
				// Cache the loaded plugin names.
				editor.plugins = plugins;
				
				// Initialize all plugins that have the "init" method defined.
				CKEDITOR.tools.each( plugins, function( plugin )
					{
						var plugin = CKEDITOR.plugins.get( plugin );
						if ( plugin && plugin.init )
							plugin.init( editor );
					});
			});
	};

	return function( element, instanceConfig )
	{
		// Call the base constructor.
		CKEDITOR.event.call( this );

		this.element = element;
		this.name = element.getId() || element.getNameAtt() || getNewName();

		// Get the default settings.
		this.config = new config();

		// Call initConfig using events, to be sure that instancecreated is
		// fired first.
		this.on( 'instancecreated', function()
			{
				initConfig( this, instanceConfig );
			});
	};
}());

CKEDITOR.editor.prototype =
{
	// The CKEDITOR.editor.fire will always pass itself as the "editor"
	// param in CKEDITOR.event.fire. So, we override it to do that
	// automaticaly.
	fire : function( eventName, data )
	{
		return CKEDITOR.event.prototype.fire.call( this, eventName, data, this );
	},

	fireOnce : function( eventName, data )
	{
		return CKEDITOR.event.prototype.fireOnce.call( this, eventName, data, this );
	}
};

// "Inherit" (copy actually) from CKEDITOR.event.
CKEDITOR.tools.extend( CKEDITOR.editor.prototype, CKEDITOR.event.prototype );
