﻿/*
 * CKEditor - The text editor for Internet - http://ckeditor.com
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 *
 * == BEGIN LICENSE ==
 *
 * Licensed under the terms of any of the following licenses at your
 * choice:
 *
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 *
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 *
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 *
 * == END LICENSE ==
 */

/**
 * @fileOverview Defines the {@link CKEDITOR.event} class, which serves as the
 *		base for classes and objects that require event handling features.
 */

/**
 * This is a base class for classes and objects that require event handling
 * features.
 * @constructor
 * @example
 */
CKEDITOR.event = function()
{
	( this._ || ( this._ = {} ) ).events = {};
};

/**
 * Implements the {@link CKEDITOR.event} features in an object.
 * @param {Object} targetObject The object in which implement the features.
 * @type undefined
 * @example
 * var myObject = { message : 'Example' };
 * <b>CKEDITOR.event.implementOn( myObject }</b>;
 * myObject.on( 'testevent', function()
 *     {
 *         alert( this.message );  // "Example"
 *     });
 * myObject.fire( 'testevent' );
 */
CKEDITOR.event.implementOn = function( targetObject )
{
	CKEDITOR.event.call( targetObject );

	for ( var prop in CKEDITOR.event.prototype )
	{
		if ( targetObject[ prop ] == undefined )
			targetObject[ prop ] = CKEDITOR.event.prototype[ prop ];
	}
};

CKEDITOR.event.prototype = (function()
{
	var eventEntry = function( eventName )
	{
		this.name = eventName;
		this.listeners = [];
	};

	eventEntry.prototype =
	{
		// Get the listener index for a specified function.
		// Returns -1 if not found.
		getListenerIndex : function( listenerFunction )
		{
			for ( var i = 0, listeners = this.listeners ; i < listeners.length ; i++ )
			{
				if ( listeners[i].fn == listenerFunction )
					return i;
			}
			return -1;
		}
	};

	return /** @lends CKEDITOR.event.prototype */ {
		/**
		 * Registers a listener to a specific event in the current object.
		 * @param {String} eventName The event name to which listen.
		 * @param {Function} listenerFunction The function listening to the
		 *		event.
		 * @param {Object} [scopeObj] The object used to scope the listener
		 *		call (the this object. If omitted, the current object is used.
		 * @param {Object} [listenerData] Data to be sent as the
		 *		{@link CKEDITOR.eventInfo#listenerData} when calling the
		 *		listener.
		 * @param {Number} [priority] The listener priority. Lower priority
		 *		listeners are called first. Listeners with the same priority
		 *		value are called in registration order. Defaults to 10.
		 * @type undefined
		 * @example
		 * someObject.on( 'someevent', function()
		 *     {
		 *         alert( this == someObject );  // "true"
		 *     });
		 * @example
		 * someObject.on( 'someevent', function()
		 *     {
		 *         alert( this == anotherObject );  // "true"
		 *     }
		 *     , anotherObject );
		 * @example
		 * someObject.on( 'someevent', function( event )
		 *     {
		 *         alert( event.listenerData );  // "Example"
		 *     }
		 *     , null, 'Example' );
		 * @example
		 * someObject.on( 'someevent', function() { ... } );                   // 2nd called
		 * someObject.on( 'someevent', function() { ... }, null, null, 100 );  // 3rd called
		 * someObject.on( 'someevent', function() { ... }, null, null, 1 );    // 1st called
		 */
		on  : function( eventName, listenerFunction, scopeObj, listenerData, priority )
		{
			// Get the event entry (create it if needed).
			var event = this._.events[ eventName ] || ( this._.events[ eventName ] = new eventEntry( eventName ) );

			if ( event.getListenerIndex( listenerFunction ) < 0 )
			{
				// Get the listeners.
				var listeners = event.listeners;

				// Fill the scope.
				if ( !scopeObj )
					scopeObj = this;

				// Default the priority, if needed.
				if ( isNaN( priority ) )
					priority = 10;

				// Create the function to be fired for this listener.
				var listenerFirer = function( editor, publisherData, stopFn, cancelFn )
				{
					var ev =
					{
						name : eventName,
						sender : this,
						editor : editor,
						data : publisherData,
						listenerData : listenerData,
						stop : stopFn,
						cancel : cancelFn
					};

					listenerFunction.call( scopeObj, ev );

					return ev.data;
				};
				listenerFirer.fn = listenerFunction;
				listenerFirer.priority = priority;

				// Search for the right position for this new listener, based on its
				// priority.
				for ( var i = listeners.length - 1 ; i >= 0 ; i-- )
				{
					// Find the item which should be before the new one.
					if ( listeners[ i ].priority <= priority )
					{
						// Insert the listener in the array.
						listeners.splice( i + 1, 0, listenerFirer );
						return;
					}
				}

				// If no position has been found (or zero length), put it in
				// the front of list.
				listeners.unshift( listenerFirer );
			}
		},

		/**
		 * Fires an specific event in the object. All registered listeners are
		 * called at this point.
		 * @param {String} eventName The event name to fire.
		 * @param {Object} [data] Data to be sent as the
		 *		{@link CKEDITOR.eventInfo#data} when calling the
		 *		listeners.
		 * @param {CKEDITOR.editor} [editor] The editor instance to send as the
		 *		{@link CKEDITOR.eventInfo#editor} when calling the
		 *		listener.
		 * @returns {Boolean|Object} A booloan indicating that the event is to be
		 *		cancelled, or data returned by one of the listeners.
		 * @example
		 * someObject.on( 'someevent', function() { ... } );
		 * someObject.on( 'someevent', function() { ... } );
		 * <b>someObject.fire( 'someevent' )</b>;  // both listeners are called
		 * @example
		 * someObject.on( 'someevent', function( event )
		 *     {
		 *         alert( event.data );  // "Example"
		 *     });
		 * <b>someObject.fire( 'someevent', 'Example' )</b>;
		 */
		fire : function( eventName, data, editor )
		{
			// Create the function that marks the event as stopped.
			var stopped = false;
			var stopEvent = function()
			{
				stopped = true;
			};

			// Create the function that marks the event as cancelled.
			var cancelled = false;
			var cancelEvent = function()
			{
				cancelled = true;
			};

			// Get the event entry.
			var event = this._.events[ eventName ];

			if ( event )
			{
				// Loop through all listeners.
				for ( var i = 0, listeners = event.listeners ; i < listeners.length ; i++ )
				{
					// Call the listener, passing the event data.
					var retData = listeners[i].call( this, editor, data, stopEvent, cancelEvent );

					if ( typeof retData != 'undefined' )
						data = retData;

					// No further calls is stopped or cancelled.
					if ( stopped || cancelled )
						break;
				}
			}

			return cancelled || ( typeof data == 'undefined' ? false : data );
		},

		/**
		 * Fires an specific event in the object, releasing all listeners
		 * registered to that event. The same listeners are not called again on
		 * successive calls of it or of {@link #fire}.
		 * @param {String} eventName The event name to fire.
		 * @param {Object} [data] Data to be sent as the
		 *		{@link CKEDITOR.eventInfo#data} when calling the
		 *		listeners.
		 * @param {CKEDITOR.editor} [editor] The editor instance to send as the
		 *		{@link CKEDITOR.eventInfo#editor} when calling the
		 *		listener.
		 * @returns {Boolean|Object} A booloan indicating that the event is to be
		 *		cancelled, or data returned by one of the listeners.
		 * @example
		 * someObject.on( 'someevent', function() { ... } );
		 * someObject.fire( 'someevent' );  // above listener called
		 * <b>someObject.fireOnce( 'someevent' )</b>;  // above listener called
		 * someObject.fire( 'someevent' );  // no listeners called
		 */
		fireOnce : function( eventName, data, editor )
		{
			var ret = this.fire( eventName, data, editor );
			delete this._.events[ eventName ];
			return ret;
		},

		/**
		 * Unregisters a listener function from being called at the specified
		 *		event. No errors are thrown if the listener has not been
		 *		registered previously.
		 * @param {String} eventName The event name.
		 * @param {Function} listenerFunction The listener function to unregister.
		 * @type undefined
		 * @example
		 * var myListener = function() { ... };
		 * someObject.on( 'someevent', myListener );
		 * someObject.fire( 'someevent' );  // myListener called
		 * <b>someObject.removeListener( 'someevent', myListener )</b>;
		 * someObject.fire( 'someevent' );  // myListener not called
		 */
		removeListener : function( eventName, listenerFunction )
		{
			// Get the event entry.
			var event = this._.events[ eventName ];

			if ( event )
			{
				var index = event.getListenerIndex( listenerFunction );
				if ( index >= 0 )
					event.listeners.splice( index, 1 );
			}
		}
	};
})();
