﻿/*
 * CKEditor - The text editor for Internet - http://ckeditor.com
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 *
 * == BEGIN LICENSE ==
 *
 * Licensed under the terms of any of the following licenses at your
 * choice:
 *
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 *
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 *
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 *
 * == END LICENSE ==
 */

/**
 * @fileOverview Defines the {@link CKEDITOR.editor} class, which represents an
 *		editor instance.
 */

 /**
 * Represents an editor instance. This constructor should be rarely used, being
 * the standard replacement methods preferible.
 * @constructor
 * @param {CKEDITOR.dom.element} element The original element replaced by this
 *		editor instance.
 * @augments CKEDITOR.event
 * @example
 * var myTextarea = CKEDITOR.document.getById( 'myTextarea' );
 * var myEditor = <b>new CKEDITOR.editor( myTextarea )</b>;
 * CKEDITOR.add( myEditor );
 */
CKEDITOR.editor = ( function()
{
	// The counter for automatic instance names.
	var nameCounter = 0;

	var getNewName = function()
	{
		var name = 'editor' + ( ++nameCounter );
		return CKEDITOR.instances[ name ] ? getNewName() : name;
	};

	// ##### START: Config Privates

	// These function loads custom configuration files and cache the
	// CKEDITOR.editorConfig functions defined on them, so there is no need to
	// download them more than once for several instances.
	var loadConfigLoaded = {};
	var loadConfig = function( editor )
	{
		var customConfig = editor.config.customConfig;

		// Check if there is a custom config to load.
		if ( !customConfig )
			return false;

		var loadedConfig = loadConfigLoaded[ customConfig ] || ( loadConfigLoaded[ customConfig ] = { editors : [] } );

		// If the custom config has already been downloaded, reuse it.
		if ( loadedConfig.fn )
		{
			// Call the cached CKEDITOR.editorConfig defined in the custom
			// config file for the editor instance depending on it.
			loadedConfig.fn.call( editor, editor );

			// If there is no other customConfig in the chain, fire the
			// "configLoaded" event.
			if ( editor.config.customConfig == customConfig || !loadConfig( editor ) )
				editor.fireOnce( 'customConfigLoaded' );
		}
		else
		{
			// Add the editor to the list of editors waiting for this config.
			loadedConfig.editors.push( editor );

			// Load the custom configuration file.
			CKEDITOR.scriptLoader.load( customConfig, function()
				{
					// If the CKEDITOR.editorConfig function has been properly
					// defined in the custom configuration file, cache it.
					if ( CKEDITOR.editorConfig )
						loadedConfig.fn = CKEDITOR.editorConfig;
					else
						loadedConfig.fn = function(){};

					delete CKEDITOR.editorConfig;

					for ( var i = 0, length = loadedConfig.editors.length ; i < length ; i++ )
					{
						// Call the load config again. This time the custom
						// config is already cached and so it will get loaded.
						loadConfig( loadedConfig.editors[ i ] );
					}

					delete loadedConfig.editors;
				});
		}

		return true;
	};

	var initConfig = function( editor, instanceConfig )
	{
		// Setup the lister for the "customConfigLoaded" event.
		editor.on( 'customConfigLoaded', function()
			{
				// Overwrite the settings from the in-page config.
				if ( instanceConfig )
					CKEDITOR.tools.extend( editor.config, instanceConfig, true );

				// Fire the "configLoaded" event.
				editor.fireOnce( 'configLoaded' );

				// Start loading the plugins.
				loadPlugins( editor );
			});

		// The instance config may override the customConfig setting to avoid
		// loading the default ~/config.js file.
		if ( instanceConfig && instanceConfig.customConfig != undefined )
			editor.config.customConfig = instanceConfig.customConfig;

		// Load configs from the custom configuration files.
		if ( !loadConfig( editor ) )
			editor.fireOnce( 'customConfigLoaded' );
	};

	// Basic config class to inherit the default settings from CKEDITOR.config.
	var config = function()
	{};
	config.prototype = CKEDITOR.config;

	// ##### END: Config Privates

	var loadPlugins = function( editor )
	{
		// Load all plugins defined in the "plugins" setting.
		CKEDITOR.plugins.load( editor.config.plugins.split( ',' ), function( plugins )
			{
				// Cache the loaded plugin names.
				editor.plugins = plugins;

				// Initialize all plugins that have the "init" method defined.
				for ( var i = 0 ; i < plugins.length ; i++ )
				{
					var pluginName = plugins[ i ];
					var plugin = CKEDITOR.plugins.get( pluginName );
					if ( plugin && plugin.init )
						plugin.init( editor, CKEDITOR.plugins.getPath( pluginName ) );
				}

				// Load the editor skin and theme.
				loadSkinTheme( editor );
			});
	};

	var loadSkinTheme = function( editor )
	{
		// Load the skin.
		CKEDITOR.skins.load( editor.config.skin, 'editor' );

		// Load the theme.
		var theme = editor.config.theme;
		CKEDITOR.themes.load( theme, function()
			{
				CKEDITOR.themes.get( theme ).build( editor, CKEDITOR.themes.getPath( theme ) );
			});
	};

	return function( element, instanceConfig )
	{
		this._ =
		{
			commands : {}
		};

		// Call the base constructor.
		CKEDITOR.event.call( this );

		/**
		 * The DOM element that has been replaced by this editor instance. This
		 * element holds the editor data on load and post.
		 * @name CKEDITOR.editor.prototype.element
		 * @type CKEDITOR.dom.element
		 * @example
		 * var editor = CKEDITOR.instances.editor1;
		 * alert( <b>editor.element</b>.getName() );  "textarea"
		 */
		this.element = element;

		/**
		 * The editor instance name. It hay be the replaced element id, name or
		 * a default name using a progressive counter (editor1, editor2, ...).
		 * @name CKEDITOR.editor.prototype.name
		 * @type String
		 * @example
		 * var editor = CKEDITOR.instances.editor1;
		 * alert( <b>editor.name</b> );  "editor1"
		 */
		this.name = element.getId() || element.getNameAtt() || getNewName();

		/**
		 * The configurations for this editor instance. It inherits all
		 * settings defined in (@link CKEDITOR.config}, combined with settings
		 * loaded from custom configuration files and those defined inline in
		 * the page when creating the editor.
		 * @name CKEDITOR.editor.prototype.config
		 * @type Object
		 * @example
		 * var editor = CKEDITOR.instances.editor1;
		 * alert( <b>editor.config.theme</b> );  "default" e.g.
		 */
		this.config = new config();

		// Call initConfig using events, to be sure that instanceCreated is
		// fired first.
		this.on( 'instanceCreated', function()
			{
				initConfig( this, instanceConfig );
			});
	};
})();

CKEDITOR.editor.prototype =
{
	addCommand : function( commandName, commandDefinition )
	{
		this._.commands[ commandName ] = commandDefinition;
	},

	execCommand : function( commandName, data )
	{
		var command = this.getCommand( commandName );
		if ( command )
			return command.exec( data );

		throw 'Unknown command name "' + commandName + '"';
	},
	
	getCommand : function( commandName )
	{
		return this._.commands[ commandName ] || null;
	},
	
	// Both fire and fireOnce will always pass this editor instance as the
	// "editor" param in CKEDITOR.event.fire. So, we override it to do that
	// automaticaly.

	/** @ignore */
	fire : function( eventName, data )
	{
		return CKEDITOR.event.prototype.fire.call( this, eventName, data, this );
	},

	/** @ignore */
	fireOnce : function( eventName, data )
	{
		return CKEDITOR.event.prototype.fireOnce.call( this, eventName, data, this );
	},

	/**
	 * Gets the editor data. The data will be in raw format. It is the same
	 * data that is posted by the editor.
	 * @type String
	 * @returns (String) The editor data.
	 * @example
	 * if ( CKEDITOR.instances.editor1.<b>getData()</b> == '' )
	 *     alert( 'There is no data available' );
	 */
	getData : function()
	{
		this.fire( 'beforeGetData' );

		// Fire "getData" so data manipulation may happen.
		var eventData = { dataValue : this._.data || this.element.$.value };
		this.fire( 'getData', eventData );

		return eventData.dataValue;
	},

	/**
	 * Sets the editor data. The data must be provided in raw format.
	 * @example
	 * CKEDITOR.instances.editor1.<b>setData( '&lt;p&gt;This is the editor data.&lt;/p&gt;' )</b>;
	 */
	setData : function( data )
	{
		// Fire "setData" so data manipulation may happen.
		var eventData = { dataValue : data };
		this.fire( 'setData', eventData );

		this._.data = eventData.dataValue;

		this.fire( 'afterSetData' );
	},

	/**
	 * Updates the &lt;textarea&gt; element that has been replaced by the editor with
	 * the current data available in the editor.
	 * @example
	 * CKEDITOR.instances.editor1.updateElement();
	 * alert( document.getElementById( 'editor1' ).value );  // The current editor data.
	 */
	updateElement : function()
	{
		this.element.$.value = this.getData();
	}
};

// "Inherit" (copy actually) from CKEDITOR.event.
CKEDITOR.event.implementOn( CKEDITOR.editor.prototype );
