﻿/*
 * CKEditor - The text editor for Internet - http://ckeditor.com
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 *
 * == BEGIN LICENSE ==
 *
 * Licensed under the terms of any of the following licenses at your
 * choice:
 *
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 *
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 *
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 *
 * == END LICENSE ==
 */

(function()
{
	// #### checkSelectionChange : START

	// The selection change check basically saves the element parent tree of
	// the current node and check it on successive requests. If there is any
	// change on the tree, then the selectionChange event gets fired.
	var checkSelectionPreviousPath;
	var checkSelectionChange = function()
	{
		// In IE, the "selectionchange" event may still get thrown when
		// releasing the WYSIWYG mode, so we need to check it first.
		var sel = this.getSelection();
		if ( !sel )
			return;

		// Get the element at the start of the selection.
		var node = this.getSelection().getStartElement(),
			changed,
			currentPath = [],
			counter = 0;

		// Loops through the parent tree of the main node.
		while( node )
		{
			// Look for changes in the parent node tree.
			if ( !changed && ( !checkSelectionPreviousPath || !node.equals( checkSelectionPreviousPath[ counter++ ] ) ) )
				changed = true;

			currentPath.push( node );
			node = node.getParent();
		}

		checkSelectionPreviousPath = currentPath;

		if ( changed )
			this.fire( 'selectionChange' );
	};

	var checkSelectionChangeTimer;
	var checkSelectionChangeTimeoutPending;
	var checkSelectionChangeTimeout = function()
	{
		// Firing the "OnSelectionChange" event on every key press started to
		// be too slow. This function guarantees that there will be at least
		// 200ms delay between selection checks.

		checkSelectionChangeTimeoutPending = true;

		if ( checkSelectionChangeTimer )
			return;

		checkSelectionChangeTimeoutExec.call( this );

		checkSelectionChangeTimer = CKEDITOR.tools.setTimeout( checkSelectionChangeTimeoutExec, 200, this );
	};

	var checkSelectionChangeTimeoutExec = function()
	{
		checkSelectionChangeTimer = null;

		if ( checkSelectionChangeTimeoutPending )
		{
			// Call this with a timeout so the browser properly moves the
			// selection after the mouseup. It happened that the selection was
			// being moved after the mouseup when clicking inside selected text
			// with Firefox.
			CKEDITOR.tools.setTimeout( checkSelectionChange, 0, this );

			checkSelectionChangeTimeoutPending = false;
		}
	};

	// #### checkSelectionChange : END

	CKEDITOR.plugins.add( 'selection',
	{
		init : function( editor, pluginPath )
		{
			editor.on( 'contentDom', function()
				{
					if ( CKEDITOR.env.ie )
					{
						// IE is the only to provide the "selectionchange"
						// event.
						editor.document.on( 'selectionchange', checkSelectionChangeTimeout, editor );
					}
					else
					{
						// In other browsers, we make the selection change
						// check based on other events, like clicks or keys
						// press.

						editor.document.on( 'mouseup', checkSelectionChangeTimeout, editor );
						editor.document.on( 'keyup', checkSelectionChangeTimeout, editor );
					}
				});
		}
	});
})();

/**
 * Gets the current selection from the editing area when in WYSIWYG mode.
 * @returns {CKEDITOR.dom.selection} A selection object or null if not on
 *		WYSIWYG mode or no selection is available.
 * @example
 * var selection = CKEDITOR.instances.editor1.<b>getSelection()</b>;
 * alert( selection.getType() );
 */
CKEDITOR.editor.prototype.getSelection = function()
{
	return this.document ? this.document.getSelection() : null;
};

/**
 * Gets the current selection from the document.
 * @returns {CKEDITOR.dom.selection} A selection object.
 * @example
 * var selection = CKEDITOR.instances.editor1.document.<b>getSelection()</b>;
 * alert( selection.getType() );
 */
CKEDITOR.dom.document.prototype.getSelection = function()
{
	return new CKEDITOR.dom.selection( this );
};

/**
 * No selection.
 * @constant
 * @example
 * if ( editor.getSelection().getType() == CKEDITOR.SELECTION_NONE )
 *     alert( 'Nothing is selected' );
 */
CKEDITOR.SELECTION_NONE		= 1;

/**
 * Text or collapsed selection.
 * @constant
 * @example
 * if ( editor.getSelection().getType() == CKEDITOR.SELECTION_TEXT )
 *     alert( 'Text is selected' );
 */
CKEDITOR.SELECTION_TEXT		= 2;

/**
 * Element selection.
 * @constant
 * @example
 * if ( editor.getSelection().getType() == CKEDITOR.SELECTION_ELEMENT )
 *     alert( 'An element is selected' );
 */
CKEDITOR.SELECTION_ELEMENT	= 3;

/**
 * Manipulates the selection in a DOM document.
 * @constructor
 * @example
 */
CKEDITOR.dom.selection = function( document )
{
	this.document = document;
};

(function()
{
	var styleObjectElements = { img:1,hr:1,li:1,table:1,tr:1,td:1,embed:1,object:1,ol:1,ul:1 };

	CKEDITOR.dom.selection.prototype =
	{
		/**
		 * Gets the native selection object from the browser.
		 * @returns {Object} The native selection object.
		 * @example
		 * var selection = editor.getSelection().<b>getNative()</b>;
		 */
		getNative : (function()
		{
			if ( CKEDITOR.env.ie )
				return function()
					{
						return this.document.$.selection;
					};
			else
				return function()
					{
						return this.document.getWindow().$.getSelection();
					};
		})(),

		/**
		 * Gets the type of the current selection. The following values are
		 * available:
		 * <ul>
		 *		<li>{@link CKEDITOR.SELECTION_NONE} (1): No selection.</li>
		 *		<li>{@link CKEDITOR.SELECTION_TEXT} (2): Text is selected or
		 *			collapsed selection.</li>
		 *		<li>{@link CKEDITOR.SELECTION_ELEMENT} (3): A element
		 *			selection.</li>
		 * </ul>
		 * @returns {Number} One of the following constant values:
		 *		{@link CKEDITOR.SELECTION_NONE}, {@link CKEDITOR.SELECTION_TEXT} or
		 *		{@link CKEDITOR.SELECTION_ELEMENT}.
		 * @example
		 * if ( editor.getSelection().<b>getType()</b> == CKEDITOR.SELECTION_TEXT )
		 *     alert( 'Text is selected' );
		 */
		getType : (function()
		{
			if ( CKEDITOR.env.ie )
				return function()
					{
						try
						{
							var sel = this.getNative(),
								ieType = sel.type;

							if ( ieType == 'Text' )
								return CKEDITOR.SELECTION_TEXT;

							if ( ieType == 'Control' )
								return CKEDITOR.SELECTION_ELEMENT;

							// It is possible that we can still get a text range
							// object even when type == 'None' is returned by IE.
							// So we'd better check the object returned by
							// createRange() rather than by looking at the type.
							if ( sel.createRange().parentElement )
								return CKEDITOR.SELECTION_TEXT;
						}
						catch(e) {}

						return CKEDITOR.SELECTION_NONE;
					};
			else
				return function()
					{
						var sel = this.getNative();
						if ( !sel )
							return CKEDITOR.SELECTION_NONE;

						if ( sel.rangeCount == 1 )
						{
							// Check if the actual selection is a control (IMG,
							// TABLE, HR, etc...).

							var range = sel.getRangeAt(0),
								startContainer = range.startContainer;

							if ( startContainer == range.endContainer
								&& startContainer.nodeType == 1
								&& ( range.endOffset - range.startOffset ) == 1
								&& styleObjectElements[ startContainer.childNodes[ range.startOffset ].nodeName.toLowerCase() ] )
							{
								return CKEDITOR.SELECTION_ELEMENT;
							}
						}

						return CKEDITOR.SELECTION_TEXT;
					};
		})(),

		/**
		 * Gets the DOM element in which the selection starts.
		 * @returns {CKEDITOR.dom.element} The element at the beginning of the
		 *		selection.
		 * @example
		 * var element = editor.getSelection().<b>getStartElement()</b>;
		 * alert( element.getName() );
		 */
		getStartElement : function()
		{
			var node,
				sel = this.getNative();

			switch ( this.getType() )
			{
				case CKEDITOR.SELECTION_ELEMENT :
					return this.getSelectedElement();

				case CKEDITOR.SELECTION_TEXT :

					if ( CKEDITOR.env.ie )
					{
						var range = sel.createRange();
						range.collapse( true );

						node = range.parentElement();
					}
					else
					{
						node = sel.anchorNode;

						if ( node.nodeType != 1 )
							node = node.parentNode;
					}
			}

			return ( node ? new CKEDITOR.dom.element( node ) : null );
		},

		/**
		 * Gets the current selected element.
		 * @returns {CKEDITOR.dom.element} The selected element. Null if no
		 *		selection is available or the selection type is not
		 *		{@link CKEDITOR.SELECTION_ELEMENT}.
		 * @example
		 * var element = editor.getSelection().<b>getSelectedElement()</b>;
		 * alert( element.getName() );
		 */
		getSelectedElement : function()
		{
			var node;

			if ( this.getType() == CKEDITOR.SELECTION_ELEMENT )
			{
				var sel = this.getNative();

				if ( CKEDITOR.env.ie )
				{
					try
					{
						node = sel.createRange().item(0);
					}
					catch(e) {}
				}
				else
				{
					var range = sel.getRangeAt( 0 );
					node = range.startContainer.childNodes[ range.startOffset ];
				}
			}

			return ( node ? new CKEDITOR.dom.element( node ) : null );
		},

		selectElement : (function()
		{
			if ( CKEDITOR.env.ie )
			{
				return function( element )
					{
						this.getNative().empty() ;

						var range ;
						try
						{
							// Try to select the node as a control.
							range = this.document.$.body.createControlRange() ;
							range.addElement( element.$ ) ;
						}
						catch(e)
						{
							// If failed, select it as a text range.
							range = this.document.$.body.createTextRange() ;
							range.moveToElementText( element.$ ) ;
						}

						range.select() ;
					};
			}
			else
			{
				return function( element )
					{
						// Create the range for the element.
						var range = this.document.$.createRange() ;
						range.selectNode( element.$ ) ;

						// Select the range.
						var sel = this.getNative() ;
						sel.removeAllRanges() ;
						sel.addRange( range ) ;
					};
			}
		})()
	};
})();
