﻿/*
 * CKEditor - The text editor for Internet - http://ckeditor.com
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 *
 * == BEGIN LICENSE ==
 *
 * Licensed under the terms of any of the following licenses at your
 * choice:
 *
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 *
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 *
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 *
 * == END LICENSE ==
 */

/**
 * @fileOverview Defines the {@link CKEDITOR.resourceManager} class, which is
 *		the base for resource managers, like plugins and themes.
 */

 /**
 * Base class for resource managers, like plugins and themes. This class is not
 * intended to be used out of the CKEditor core code.
 * @param {String} basePath The path for the resources folder.
 * @param {String} fileName The name used for resource files.
 * @namespace
 * @example
 */
CKEDITOR.resourceManager = function( basePath, fileName )
{
	/**
	 * The base directory containing all resources.
	 * @name CKEDITOR.resourceManager.prototype.basePath
	 * @type String
	 * @example
	 */
	this.basePath = basePath;

	/**
	 * The name used for resource files.
	 * @name CKEDITOR.resourceManager.prototype.fileName
	 * @type String
	 * @example
	 */
	this.fileName = fileName;

	/**
	 * Contains references to all resources that have already been registered
	 * with {@link #add}.
	 * @name CKEDITOR.resourceManager.prototype.registered
	 * @type Object
	 * @example
	 */
	this.registered = {};

	/**
	 * Contains references to all resources that have already been loaded
	 * with {@link #load}.
	 * @name CKEDITOR.resourceManager.prototype.loaded
	 * @type Object
	 * @example
	 */
	this.loaded = {};

	/**
	 * Contains references to all resources that have already been registered
	 * with {@link #addExternal}.
	 * @name CKEDITOR.resourceManager.prototype.externals
	 * @type Object
	 * @example
	 */
	this.externals = {};

	/**
	 * @private
	 */
	this._ =
	{
		// List of callbacks waiting for plugins to be loaded.
		waitingList : {}
	};
};

CKEDITOR.resourceManager.prototype =
{
	/**
	 * Registers a resource.
	 * @param {String} name The resource name.
	 * @param {Object} [definition] The resource definition.
	 * @example
	 * CKEDITOR.plugins.add( 'sample', { ... plugin definition ... } );
	 * @see CKEDITOR.pluginDefinition
	 */
	add : function( name, definition )
	{
		if ( this.registered[ name ] )
			throw '[CKEDITOR.resourceManager.add] The resource name "' + name + '" is already registered.';

		this.registered[ name ] = definition || {};
	},

	/**
	 * Gets the definition of a specific resource.
	 * @param {String} name The resource name.
	 * @type Object
	 * @example
	 * var definition = <b>CKEDITOR.plugins.get( 'sample' )</b>;
	 */
	get : function( name )
	{
		return this.registered[ name ] || null;
	},

	/**
	 * Get the full path for a specific loaded resource.
	 * @param {String} name The resource name.
	 * @type String
	 * @example
	 * alert( <b>CKEDITOR.plugins.getPath( 'sample' )</b> );  // "&lt;editor path&gt;/plugins/sample/"
	 */
	getPath : function( name )
	{
		return this.loaded[ name ] || null;
	},

	/**
	 * Registers a resource to be loaded from an external path instead of the core base path.
	 * @param {String} name The resource name.
	 * @param {String} path The resource external path.
	 * @example
	 * // Loads a plugin from '/myplugin/samples/plugin.js'.
	 * CKEDITOR.plugins.addExternal( 'sample', '/myplugins/sample/' );
	 */
	addExternal : function( name, path )
	{
		if ( this.registered[ name ] || this.externals[ name ] )
			throw '[CKEDITOR.resourceManager.import] The resource name "' + name + '" is already registered or imported.';

		this.externals[ name ] = path;
	},

	/**
	 * Loads one or more resources.
	 * @param {String|Array} name The name of the resource to load. It may be a
	 *		string with a single resource name, or an array with several names.
	 * @param {Function} callback A function to be called when all resources
	 *		are loaded. The callback will receive an array containing all
	 *		loaded names.
	 * @param {Object} [scope] The scope object to be used for the callback
	 *		call.
	 * @example
	 * <b>CKEDITOR.plugins.load</b>( 'myplugin', function( plugins )
	 *     {
	 *         alert( plugins[0] );  // "myplugin"
	 *     });
	 */
	load : function( name, callback, scope )
	{
		// Ensure that we have an Array of names.
		var names = CKEDITOR.tools.isArray( name ) ? name : name ? [ name ] : [],
			total = names.length,
			resources = {};

		// Nothing to load, just call the callback.
		if ( !total )
		{
			callback.call( scope || window, resources );
			return;
		}

		// This function is used to count the loaded plugins and call the
		// callback when finished loading.
		callback._loaded = 0;
		callback._total = total;
		var loadCheck = function( callback )
		{
			if ( ++callback._loaded == callback._total )
				callback.call( scope || window, resources );
		};

		var loaded = this.loaded,
			waitingList = this._.waitingList;

		var loadPlugin = function( name )
		{
			// Calculate the plugin script path.
			var path = this.externals[ name ] || ( this.basePath + name + '/' ),
				filePath = CKEDITOR.getUrl( path + this.fileName + '.js' );

			// Load the plugin script.
			CKEDITOR.scriptLoader.load( filePath, function( success )
				{
					if ( !success )
						throw '[CKEDITOR.resourceManager.load] Resource name "' + name + '" was not found at "' + filePath + '".';

					loaded[ name ] = path;

					resources[ name ] = this.get( name );

					// Check all callbacks that were waiting for this
					// resource.
					for ( var j = 0 ; j < waitingInfo.length ; j++ )
						loadCheck( waitingInfo[ j ] );

					delete waitingList[ name ];
				}, this);
		};

		// Loop through all names.
		for ( var i = 0 ; i < names.length ; i++ )
		{
			name = names[ i ];

			if ( name && typeof resources[ name ] == 'undefined' )
			{
				resources[ name ] = this.get( name );

				// If not loaded already.
				if ( !loaded[ name ] && !this.registered[ name ] )
				{
					var waitingInfo = waitingList[ name ] || ( waitingList[ name ] = [] );
					waitingInfo.push( callback );

					// If this is the first call for it, go ahead loading.
					if ( waitingInfo.length == 1 )
						loadPlugin.call( this, name );

					continue;
				}
			}
			loadCheck( callback );
		}
	}
};
