﻿/*
 * CKEditor - The text editor for Internet - http://ckeditor.com
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 *
 * == BEGIN LICENSE ==
 *
 * Licensed under the terms of any of the following licenses at your
 * choice:
 *
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 *
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 *
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 *
 * == END LICENSE ==
 */

/**
 * @fileOverview Defines the {@link CKEDITOR.resourceManager} class, which is
 *		the base for resource managers, like plugins and themes.
 */

 /**
 * Base class for resource managers, like plugins and themes. This class is not
 * intended to be used out of the CKEditor core code.
 * @param {String} basePath The path for the resources folder.
 * @param {String} fileName The name used for resource files.
 * @namespace
 * @example
 */
CKEDITOR.resourceManager = function( basePath, fileName )
{
	/**
	 * The base directory containing all resources.
	 * @name CKEDITOR.resourceManager.prototype.basePath
	 * @type String
	 * @example
	 */
	this.basePath = basePath;

	/**
	 * The name used for resource files.
	 * @name CKEDITOR.resourceManager.prototype.fileName
	 * @type String
	 * @example
	 */
	this.fileName = fileName;

	/**
	 * Contains references to all resources that have already been registered
	 * with {@link #add}.
	 * @name CKEDITOR.resourceManager.prototype.registered
	 * @type Object
	 * @example
	 */
	this.registered = {};

	/**
	 * Contains references to all resources that have already been loaded
	 * with {@link #load}.
	 * @name CKEDITOR.resourceManager.prototype.loaded
	 * @type Object
	 * @example
	 */
	this.loaded = {};

	/**
	 * Contains references to all resources that have already been registered
	 * with {@link #addExternal}.
	 * @name CKEDITOR.resourceManager.prototype.externals
	 * @type Object
	 * @example
	 */
	this.externals = {};

	/**
	 * @private
	 */
	this._ =
	{
		// List of callbacks waiting for plugins to be loaded.
		waitingList : {}
	};
};

CKEDITOR.resourceManager.prototype =
{
	/**
	 * Registers a resource.
	 * @param {String} name The resource name.
	 * @param {Object} [definition] The resource definition.
	 * @example
	 * CKEDITOR.plugins.add( 'sample', { ... plugin definition ... } );
	 * @see CKEDITOR.pluginDefinition
	 */
	add : function( name, definition )
	{
		if ( this.registered[ name ] )
			throw '[CKEDITOR.resourceManager.add] The resource name "' + name + '" is already registered.';

		this.registered[ name ] = definition || {};
	},

	/**
	 * Gets the definition of a specific resource.
	 * @param {String} name The resource name.
	 * @type Object
	 * @example
	 * var definition = <b>CKEDITOR.plugins.get( 'sample' )</b>;
	 */
	get : function( name )
	{
		return this.registered[ name ] || null;
	},

	/**
	 * Get the full path for a specific loaded resource.
	 * @param {String} name The resource name.
	 * @type String
	 * @example
	 * alert( <b>CKEDITOR.plugins.getPath( 'sample' )</b> );  // "&lt;editor path&gt;/plugins/sample/"
	 */
	getPath : function( name )
	{
		return this.externals[ name ] || this.basePath + name + '/';
	},

	/**
	 * Registers a resource to be loaded from an external path instead of the core base path.
	 * @param {String} name The resource name.
	 * @param {String} path The resource external path.
	 * @example
	 * // Loads a plugin from '/myplugin/samples/plugin.js'.
	 * CKEDITOR.plugins.addExternal( 'sample', '/myplugins/sample/' );
	 */
	addExternal : function( name, path )
	{
		if ( this.registered[ name ] || this.externals[ name ] )
			throw '[CKEDITOR.resourceManager.import] The resource name "' + name + '" is already registered or imported.';

		this.externals[ name ] = path;
	},

	/**
	 * Loads one or more resources.
	 * @param {String|Array} name The name of the resource to load. It may be a
	 *		string with a single resource name, or an array with several names.
	 * @param {Function} callback A function to be called when all resources
	 *		are loaded. The callback will receive an array containing all
	 *		loaded names.
	 * @param {Object} [scope] The scope object to be used for the callback
	 *		call.
	 * @example
	 * <b>CKEDITOR.plugins.load</b>( 'myplugin', function( plugins )
	 *     {
	 *         alert( plugins['myplugin'] );  // "object"
	 *     });
	 */
	load : function( names, callback, scope )
	{
		// Ensure that we have an array of names.
		if ( !CKEDITOR.tools.isArray( names ) )
			names = names ? [ names ] : [];

		var loaded = this.loaded,
			registered = this.registered,
			urls = [],
			urlsNames = {},
			resources = {};

		// Loop through all names.
		for ( var i = 0 ; i < names.length ; i++ )
		{
			var name = names[ i ];

			if ( !name )
				continue;

			// If not available yet.
			if ( !loaded[ name ] && !registered[ name ] )
			{
				var url = CKEDITOR.getUrl( this.getPath( name ) + this.fileName + '.js' );
				urls.push( url );
				urlsNames[ url ] = name;
			}
			else
				resources[ name ] = this.get( name );
		}

		CKEDITOR.scriptLoader.load( urls, function( completed, failed )
			{
				if ( failed.length )
					throw '[CKEDITOR.resourceManager.load] Resource name "' + urlsNames[ failed[ 0 ] ] + '" was not found at "' + failed[ 0 ] + '".';

				for ( var i = 0 ; i < completed.length ; i++ )
				{
					var name = urlsNames[ completed[ i ] ];
					resources[ name ] = this.get( name );

					loaded[ name ] = 1;
				}

				callback.call( scope, resources );
			}
			, this);
	}
};
