/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.response;

import java.io.StringWriter;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import net.fckeditor.handlers.CommandHandler;
import net.fckeditor.handlers.LocalizedPropertiesLoader;
import net.fckeditor.handlers.ResourceType;
import net.fckeditor.requestcycle.ThreadLocalData;
import net.fckeditor.tool.Utils;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Creates an XML response for every <code>GET</code> request of the Connector
 * servlet. This class maps directly to the XML layout described <a
 * href="http://docs.fckeditor.net/FCKeditor_2.x/Developers_Guide/Server_Side_Integration#The_Commands">here</a>.
 *
 * @version $Id: GetResponse.java 2421 2008-09-02 17:09:11Z th-schwarz $
 */
public class GetResponse {

	private Document document;
	private Element errorElement;
	private Element foldersElement;
	private Element filesElement;
	
	/** Error number OK */
	public static final int EN_OK = 0;
	
	/** Error number ERROR */
	public static final int EN_ERROR = 1;

	/** Error number ALREADY EXISTS */
	public static final int EN_ALREADY_EXISTS = 101;

	/** Error number INVALID FOLDER NAME */
	public static final int EN_INVALID_FOLDER_NAME = 102;

	/** Error number SECURITY ERROR */
	public static final int EN_SECURITY_ERROR = 103;

	/** Error number UNKNOWN ERROR */
	public static final int EN_UKNOWN = 110;
	

	/**
	 * 
	 * Use this constructor if want to respond a negative/error message with
	 * custom text.
	 * 
	 * @param number
	 * @param text
	 */
	public GetResponse(int number, String text) {
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			document = builder.newDocument();
		} catch (ParserConfigurationException e) {
			throw new RuntimeException(e);
		}

		Element root = document.createElement("Connector");
		document.appendChild(root);
		setError(number, text);
	}
	
	/**
	 * Use this constructor if want to respond a positive message.
	 * 
	 * @param command
	 * @param resourceType
	 * @param currentFolder
	 * @param constructedUrl
	 */
	public GetResponse(CommandHandler command, ResourceType resourceType, 
			String currentFolder, String constructedUrl) {
	
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			document = builder.newDocument();
		} catch (ParserConfigurationException e) {
			throw new RuntimeException(e);
		}
	
		Element root = document.createElement("Connector");
		document.appendChild(root);
		root.setAttribute("command", command.toString());
		root.setAttribute("resourceType", resourceType.getName());
	
		Element currentFolderElement = document.createElement("CurrentFolder");
		currentFolderElement.setAttribute("path", currentFolder);
	
		currentFolderElement.setAttribute("url", constructedUrl);
		root.appendChild(currentFolderElement);
	
	}

	/**
	 * Use this constructor if want to respond a negative/error message only.
	 * 
	 * @param number
	 */
	public GetResponse(int number) {
		this(number, null);
	}
	
	/**
	 * Sets an error number with a custom message.
	 * 
	 * @param number
	 * @param text
	 */
	public void setError(int number, String text) {

		if (errorElement == null) {
			errorElement = document.createElement("Error");
			document.getDocumentElement().appendChild(errorElement);
		}

		errorElement.setAttribute("number", String.valueOf(number));
		if (Utils.isNotEmpty(text))
			errorElement.setAttribute("text", text);

	}

	/**
	 * Sets an error number.
	 * 
	 * @param number
	 */
	public void setError(int number) {
		setError(number, null);
	}

	/**
	 * Lists all folders as XML tags.
	 * @param dir
	 */
	public void setFolders(final List<String> dirs) {

		if (foldersElement != null) {
			Element parent = (Element) foldersElement.getParentNode();
			parent.removeChild(foldersElement);
		}

		foldersElement = document.createElement("Folders");
		document.getDocumentElement().appendChild(foldersElement);

		for (String file : dirs) {
			Element folderElement = document.createElement("Folder");
			folderElement.setAttribute("name", file);
			foldersElement.appendChild(folderElement);
		}
	}
	
	/**
	 * Lists all files XML tags.
	 * 
	 * @param Map, key is the file name and value is the size of the file in bytes
	 */
	public void setFiles(final Map<String, Long> files) {
		
		if (filesElement != null) {
			Element parent = (Element) filesElement.getParentNode();
			parent.removeChild(filesElement);
		}

		filesElement = document.createElement("Files");
		document.getDocumentElement().appendChild(filesElement);
		
		long length;
		long tempLength;
		for (String fileName : files.keySet()) {
			Element fileElement = document.createElement("File");
			fileElement.setAttribute("name", fileName);
			tempLength = files.get(fileName).longValue();
			if (tempLength < 1024)
				length = 1L;
			else 
				length = tempLength/1024;
			fileElement.setAttribute("size", String.valueOf(length));
			filesElement.appendChild(fileElement);
		}
	}
	
	@Override
	public String toString() {
		document.getDocumentElement().normalize();
		TransformerFactory factory = TransformerFactory.newInstance();

		StringWriter sw = new StringWriter();

		try {
			Transformer transformer = factory.newTransformer();

			DOMSource source = new DOMSource(document);
			StreamResult result = new StreamResult(sw);

			transformer.transform(source, result);
		} catch (TransformerException e) {
			throw new RuntimeException(e);
		}

		return sw.toString();
	}
	
	public static GetResponse getOK() {
		return new GetResponse(EN_OK);
	}
	
	public static GetResponse getErrorInvalidFolderName() {
		return new GetResponse(EN_INVALID_FOLDER_NAME);
	}
	
	public static GetResponse getErrorInvalidCommand() {
		LocalizedPropertiesLoader lpl = LocalizedPropertiesLoader.getInstance(ThreadLocalData.getServletRequest());
		return new GetResponse(EN_ERROR, lpl.getInvalidCommand());
	}
	
	public static GetResponse getErrorInvalidType() {
		LocalizedPropertiesLoader lpl = LocalizedPropertiesLoader.getInstance(ThreadLocalData.getServletRequest());
		return new GetResponse(EN_ERROR, lpl.getInvalidType());
	}
	
	public static GetResponse getErrorInvalidCurrentFolder() {
		LocalizedPropertiesLoader lpl = LocalizedPropertiesLoader.getInstance(ThreadLocalData.getServletRequest());
		return new GetResponse(EN_ERROR, lpl.getInvalidCurrentFolder());
	}
	
	public static GetResponse getErrorFolderAlreadyExists() {
		return new GetResponse(EN_ALREADY_EXISTS);
	}
	
	public static GetResponse getErrorSecurity() {
		return new GetResponse(EN_SECURITY_ERROR);
	}
	
	public static GetResponse getErrorFileBrowsingDisabled() {
		LocalizedPropertiesLoader lpl = LocalizedPropertiesLoader.getInstance(ThreadLocalData.getServletRequest());
		return new GetResponse(EN_ERROR, lpl.getFileBrowsingDisabled());
	}

	public static GetResponse getErrorFolderCreationDisabled() {
		LocalizedPropertiesLoader lpl = LocalizedPropertiesLoader.getInstance(ThreadLocalData.getServletRequest());
		return new GetResponse(EN_ERROR, lpl.getFolderCreationDisabled());
	}
	
	public static GetResponse getErrorUnknown() {
		return new GetResponse(EN_UKNOWN);
	}
}