/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.connector;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.fckeditor.connector.exception.FolderAlreadyExistsException;
import net.fckeditor.connector.exception.InvalidCurrentFolderException;
import net.fckeditor.connector.exception.InvalidFolderNameException;
import net.fckeditor.connector.exception.SecurityIssueException;
import net.fckeditor.connector.exception.UnknownException;
import net.fckeditor.handlers.CommandHandler;
import net.fckeditor.handlers.ConnectorHandler;
import net.fckeditor.handlers.ExtensionsHandler;
import net.fckeditor.handlers.RequestCycleHandler;
import net.fckeditor.handlers.ResourceType;
import net.fckeditor.response.GetResponse;
import net.fckeditor.response.UploadResponse;
import net.fckeditor.tool.Utils;
import net.fckeditor.tool.UtilsFile;
import net.fckeditor.tool.UtilsResponse;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * This class is calling by {@link ConnectorServlet}. It verifies the request parameters 
 * and forward it to the corresponding {@link Connector} methods. If errors are happened inside these methods,
 * well defined exceptions from {@link net.fckeditor.connector.exception} will be thrown. These exceptions or the return
 * values of {@link Connector} methods will be transfered to the HttpServletResponse object.
 * 
 * @version $Id: Dispatcher.java 2462 2008-09-14 18:07:51Z mosipov $
 */
public class Dispatcher {
	private static final Logger logger = LoggerFactory.getLogger(Dispatcher.class);
	private Connector connector = null;
	
	/**
	 * Initializes the {@link Connector}.
	 * 
	 * @param servletContext
	 */
	protected Dispatcher(final ServletContext servletContext) {
		this.connector = RequestCycleHandler.getConnector();
		this.connector.init(servletContext);
	}
	
	
	/**
	 * Manage the <code>GET</code> requests (<code>GetFolders</code>,
	 * <code>GetFoldersAndFiles</code>, <code>CreateFolder</code>).<br/>
	 * 
	 * The method accepts commands sent in the following format:<br/>
	 * <code>connector?Command=&lt;CommandName&gt;&Type=&lt;ResourceType&gt;&CurrentFolder=&lt;FolderPath&gt;</code>
	 * <p>
	 * It executes the relevant commands and then returns the result to the client in XML
	 * format.
	 * </p>
	 */
	public void doGet(final HttpServletRequest request, final HttpServletResponse response) throws IOException {
		logger.debug("Entering Dispatcher#doGet");

		response.setCharacterEncoding("UTF-8");
		response.setContentType("application/xml; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();

		String commandStr = request.getParameter("Command");
		String typeStr = request.getParameter("Type");
		String currentFolderStr = request.getParameter("CurrentFolder");

		logger.debug("Parameter Command: {}", commandStr);
		logger.debug("Parameter Type: {}", typeStr);
		logger.debug("Parameter CurrentFolder: {}", currentFolderStr);
		
		GetResponse getResponse = null;
		// check parameters
		if (!CommandHandler.isValidForGet(commandStr))
			getResponse = GetResponse.getErrorInvalidCommand();
		else if (!ResourceType.isValid(typeStr))
			getResponse = GetResponse.getErrorInvalidType();
		else if (!UtilsFile.isValidPath(currentFolderStr))
			getResponse = GetResponse.getErrorInvalidCurrentFolder();
		else {
			
			ResourceType type = ResourceType.getDefaultResourceType(typeStr);
			CommandHandler command = CommandHandler.getCommand(commandStr);
			
			// check permissions for user action
			if ((command.equals(CommandHandler.GET_FOLDERS) || command.equals(CommandHandler.GET_FOLDERS_AND_FILES))
					&& !RequestCycleHandler.isEnabledForFileBrowsing())
				getResponse = GetResponse.getErrorFileBrowsingDisabled();
			else if (command.equals(CommandHandler.CREATE_FOLDER) && !RequestCycleHandler.isEnabledForFolderCreation())
				getResponse = GetResponse.getErrorFolderCreationDisabled();
			
			else {
				
				// make the connector calls, catch its exceptions and generate the right response object
				try {
					if (command.equals(CommandHandler.CREATE_FOLDER)) {
						String newFolderStr = UtilsFile.sanitizeFolderName(request.getParameter("NewFolderName"));
						if (Utils.isEmpty(newFolderStr))
							getResponse = GetResponse.getErrorInvalidFolderName();
						else {
							logger.debug("Parameter NewFolderName: {}", newFolderStr);
							connector.createFolder(type, currentFolderStr, newFolderStr);
							getResponse = GetResponse.getOK();
						}
					} else if (command.equals(CommandHandler.GET_FOLDERS) || command.equals(CommandHandler.GET_FOLDERS_AND_FILES)) {
						// TODO I don't like this code, it has to be more generic
						String responseUrl = UtilsResponse.constructResponseUrl( 
								ConnectorHandler.getUserFilesPath(),
								"/".concat(typeStr.toLowerCase()), currentFolderStr);
						getResponse = getFileAndFolders(command, type, currentFolderStr, responseUrl);
					} else 
						getResponse = GetResponse.getErrorUnknown();
				} catch (InvalidCurrentFolderException e) {
					getResponse = GetResponse.getErrorInvalidCurrentFolder();
				} catch (SecurityIssueException e) {
					getResponse = GetResponse.getErrorSecurity();
				} catch (InvalidFolderNameException e) {
					getResponse = GetResponse.getErrorInvalidFolderName();
				} catch (FolderAlreadyExistsException e) {
					getResponse = GetResponse.getErrorFolderAlreadyExists();
				} catch (Exception e) {
					getResponse = GetResponse.getErrorUnknown();
				}
			}
		}
		
		out.print(getResponse);
		out.flush();
		out.close();
		logger.debug("Exiting Dispatcher#doGet");
	}
	
	/**
	 * Helper to make the right {@link Connector}-calls for <code>GetFolders</code> and
	 * <code>GetFoldersAndFiles</code>.
	 * 
	 * @param command should be only {@link CommandHandler#GET_FOLDERS} or {@link CommandHandler#GET_FOLDERS_AND_FILES}!!
	 * @param type
	 * @param currentFolderStr
	 * @param responseUrl
	 * @return
	 * @throws InvalidCurrentFolderException
	 * @throws SecurityIssueException
	 * @throws UnknownException
	 */
	private GetResponse getFileAndFolders(final CommandHandler command, final ResourceType type, final String currentFolderStr, final String responseUrl)
			throws InvalidCurrentFolderException, SecurityIssueException, UnknownException {
		GetResponse getResponse = new GetResponse(command, type, currentFolderStr, responseUrl);
		getResponse.setFolders(connector.getFolders(type, currentFolderStr));
		if (command.equals(CommandHandler.GET_FOLDERS_AND_FILES)) 
			getResponse.setFiles(connector.getFiles(type, currentFolderStr));
		return getResponse;
	}
	
	/**
	 * Manage the <code>POST</code> requests (<code>FileUpload</code>).<br />
	 * 
	 * The method accepts commands sent in the following format:<br />
	 * <code>connector?Command=&lt;FileUpload&gt;&Type=&lt;ResourceType&gt;&CurrentFolder=&lt;FolderPath&gt;</code>
	 * with the file in the <code>POST</code> body.<br />
	 * <br>
	 * The Connector stores an uploaded file (renames a file if another exists with the
	 * same name) and then returns the JavaScript callback.
	 * @throws IOException 
	 */
	public void doPost(final HttpServletRequest request, final HttpServletResponse response) throws IOException {
		logger.debug("Entering Dispatcher#doPost");
		response.setCharacterEncoding("UTF-8");
		response.setContentType("text/html; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();

		String commandStr = request.getParameter("Command");
		String typeStr = request.getParameter("Type");
		String currentFolderStr = request.getParameter("CurrentFolder");

		logger.debug("Parameter Command: {}", commandStr);
		logger.debug("Parameter Type: {}", typeStr);
		logger.debug("Parameter CurrentFolder: {}", currentFolderStr);

		// if this is a QuickUpload request, 'commandStr' and 'currentFolderStr'
		// are empty
		if (Utils.isEmpty(commandStr) && Utils.isEmpty(currentFolderStr)) {
			commandStr = "QuickUpload";
			currentFolderStr = "/";
		}
		
		UploadResponse uploadResponse;
		// check permissions for user actions
		if (!RequestCycleHandler.isEnabledForFileUpload())
			uploadResponse = UploadResponse.getErrorFileUploadDisabled();

		// check parameters
		else if (!CommandHandler.isValidForPost(commandStr))
			uploadResponse = UploadResponse.getErrorInvalidCommand();
		else if (!ResourceType.isValid(typeStr))
			uploadResponse = UploadResponse.getErrorInvalidType();
		else if (!UtilsFile.isValidPath(currentFolderStr))
			uploadResponse = UploadResponse.getErrorInvalidCurrentFolder();
		else {

			// call the Connector#fileUpload
			ResourceType type = ResourceType.getDefaultResourceType(typeStr);
			FileItemFactory factory = new DiskFileItemFactory();
			ServletFileUpload upload = new ServletFileUpload(factory);
			try {
				@SuppressWarnings("unchecked") List<FileItem> items = upload.parseRequest(request);
				// We upload just one file at the same time
				FileItem uplFile = items.get(0);
				// check the extension
				if (!ExtensionsHandler.isAllowed(type, FilenameUtils.getExtension(uplFile.getName())))
					uploadResponse = UploadResponse.getErrorInvalidExtension();
				// Secure image check
				else if (type.equals(ResourceType.IMAGE) && ConnectorHandler.isSecureImageUploads() && !UtilsFile.isImage(uplFile.getInputStream())) {
					uploadResponse = UploadResponse.getErrorInvalidExtension();
				} else {
					String fileName = FilenameUtils.getName(UtilsFile.sanitizeFileName(uplFile.getName()));
					String newFileName = connector.fileUpload(type, currentFolderStr, fileName, uplFile.getInputStream());
					// TODO I don't like this either
					String responseUrl = UtilsResponse.constructResponseUrl( 
							ConnectorHandler.getUserFilesPath(),
							"/".concat(typeStr.toLowerCase()), currentFolderStr);
					if (fileName.equals(newFileName))
						uploadResponse = new UploadResponse(UploadResponse.SC_OK, responseUrl);
					else
						uploadResponse =  new UploadResponse(UploadResponse.SC_RENAMED, responseUrl, newFileName); 
				}
				uplFile.delete();
			} catch (InvalidCurrentFolderException e) {
				uploadResponse = UploadResponse.getErrorInvalidCurrentFolder();
			} catch (SecurityIssueException e) {
				uploadResponse = UploadResponse.getErrorSecurity();
			} catch (Exception e) {
				uploadResponse = UploadResponse.getErrorSecurity();
			}
		}
		out.print(uploadResponse);
		out.flush();
		out.close();
		logger.debug("Exiting Dispatcher#doPost");
	}
	
	
}
