/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.handlers;

import java.util.HashMap;
import java.util.Map;

import net.fckeditor.tool.Utils;

/**
 * Common resource type holder without any implementation (connector) specific stuff.
 *
 * @version $Id: ResourceType.java 2463 2008-09-14 20:32:41Z mosipov $
 */
public class ResourceType {
	private String name;
	private String path;
	private static Map<String, ResourceType> types = new HashMap<String, ResourceType>(4);
		
	public static final ResourceType FILE = new ResourceType("file", PropertiesLoader.getProperty("connector.resourceType.file.path"));
	public static final ResourceType FLASH = new ResourceType("flash", PropertiesLoader.getProperty("connector.resourceType.flash.path"));
	public static final ResourceType IMAGE = new ResourceType("image", PropertiesLoader.getProperty("connector.resourceType.image.path"));
	public static final ResourceType MEDIA = new ResourceType("media", PropertiesLoader.getProperty("connector.resourceType.media.path"));
	
	static {
		types.put(FILE.getName(), FILE);
		types.put(FLASH.getName(), FLASH);
		types.put(IMAGE.getName(), IMAGE);
		types.put(MEDIA.getName(), MEDIA);
	}
	
	private ResourceType(final String name, final String path) {
		this.name = name;
		this.path = path;
	}
	
	public String getName() {
		return name;
	}
	
	public String getPath() {
		return path;
	}
	
	/**
	 * Getter for a {@link ResourceType} for a specified string. <br>
	 * The comparison isn't case sensitive!
	 * 
	 * @param name A resource type to retrieve.
	 * @return A {@link ResourceType} object holding the value
	 *         represented by the string argument.
	 * @throws IllegalArgumentException
	 *            If 'name' is <code>null</code>, empty, or does not exist.
	 */
	public static ResourceType valueOf(final String name) {
		if (Utils.isEmpty(name))
			throw new IllegalArgumentException();

		ResourceType rt = types.get(name.toLowerCase());
		if (rt == null)
			throw new IllegalArgumentException();
		return rt;
	}
	
	/**
	 * Checks if a specified string represents a valid resource type.<br>
	 * The comparison isn't case sensitive!
	 * 
	 * @param name
	 *            A resource type string to check.
	 * @return <code>true</code> if the string representation is valid else
	 *         <code>false</code>.
	 */
	public static boolean isValid(final String name) {
		return (Utils.isEmpty(name)) ? false : types.containsKey(name.toLowerCase());
	}

	/**
	 * A wrapper for {@link #valueOf(String)}. It returns <code>null</code>
	 * instead of throwing an exception.<br>
	 * The comparison isn't case sensitive!
	 * 
	 * @param name
	 *            A resource type string to check.
	 * @return A {@link ResourceType} object holding the value
	 *         represented by the string argument, or <code>null</code>.
	 */
	public static ResourceType getResourceType(final String name) {
		try {
			return ResourceType.valueOf(name.toLowerCase());
		} catch (Exception e) {
			return null;
		}
	}
	
	/**
	 * Tries to determine ResourceType from string and return {@link #FILE} if
	 * provided string is invalid.<br><br>
	 * The comparison isn't case sensitive!
	 * 
	 * @param name
	 * @return resource type
	 */
	public static ResourceType getDefaultResourceType(final String name) {
		ResourceType rt = getResourceType(name.toLowerCase());
		return rt == null ? FILE : rt;
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (obj == null || !(obj instanceof ResourceType))
			return false;
		else {
			ResourceType rt = (ResourceType)obj;
			return name.equals(rt.getName());
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		return name.hashCode();
	}
}
