/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.handlers;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.ResourceBundle;

import javax.servlet.http.HttpServletRequest;

import net.fckeditor.localization.LocaleResolver;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class wraps to a {@link LocaleResolver} implementation and retrieves
 * localized strings for a given request. This class safely returns default
 * strings if anything fails. Loaded strings will be cached on load.
 * 
 * @version $Id: LocalizedPropertiesLoader.java 2630 2008-11-05 21:30:52Z
 *          mosipov $
 * 
 */
public class LocalizedPropertiesLoader implements Cloneable {

	private static Map<Locale, LocalizedPropertiesLoader> prototypes = new HashMap<Locale, LocalizedPropertiesLoader>();
	private Properties properties;
	private static LocaleResolver localeResolver;
	private static final Locale NEUTRAL = new Locale("");
	private static final Logger logger = LoggerFactory
			.getLogger(LocalizedPropertiesLoader.class);

	/**
	 * This method return an instance of this class for the given locale.
	 * 
	 * @param request
	 *            The current request instance
	 * @return The LPL instance with localized strings.
	 */
	public static LocalizedPropertiesLoader getInstance(
			HttpServletRequest request) {

		Locale locale;

		if (request == null)
			locale = NEUTRAL;
		else
			locale = getLocaleResolverInstance().resolveLocale(request);

		if (locale == null)
			locale = NEUTRAL;

		LocalizedPropertiesLoader lpl = prototypes.get(locale);
		if (lpl == null) {
			lpl = new LocalizedPropertiesLoader(locale);
			prototypes.put(locale, lpl);
		}

		// for now we don't need any cloning since the values are accessed
		// read-only
		return lpl;

	}

	private static LocaleResolver getLocaleResolverInstance() {

		if (localeResolver == null) {
			String className = PropertiesLoader
					.getProperty("localization.localeResolverImpl");

			try {
				Class<?> clazz = Class.forName(className);
				localeResolver = (LocaleResolver) clazz.newInstance();
				logger.info("LocaleResolver initialized to {}", localeResolver
						.getClass());
			} catch (Exception e) {
				// TODO align message to other warnings
				logger.warn(
						"Error while loading LocaleResolver implementation", e);
			}
		}

		return localeResolver;
	}

	@Override
	protected Object clone() {
		LocalizedPropertiesLoader cloned;
		try {
			cloned = (LocalizedPropertiesLoader) super.clone();
		} catch (CloneNotSupportedException e) {
			throw new RuntimeException(e);
		}
		cloned.properties = (Properties) this.properties.clone();
		return cloned;
	}

	/**
	 * Loads the default strings and the string for the given locale and only
	 * for this locale.
	 * 
	 * @param locale
	 *            Given locale
	 */
	private LocalizedPropertiesLoader(Locale locale) {

		ResourceBundle rb = null;
		try {
			rb = ResourceBundle.getBundle("fckeditor", locale, Thread
					.currentThread().getContextClassLoader());
		} catch (Exception e) {
			logger.debug("No resource bundle found for locale '{}'", locale);
		}

		this.properties = new Properties(PropertiesLoader.getProperties());

		if (rb != null
				&& rb.getLocale().getLanguage().equals(locale.getLanguage())) {
			Enumeration<String> keys = rb.getKeys();

			while (keys.hasMoreElements()) {
				String key = keys.nextElement();
				properties.setProperty(key, rb.getString(key));
			}
		}

	}

	private String getString(String key) {
		return properties.getProperty(key);
	}

	public String getCompatibleBrowser() {
		return getString("message.compatible_browser.yes");
	}

	public String getNotCompatibleBrowser() {
		return getString("message.compatible_browser.no");
	}

	public String getFileUploadEnabled() {
		return getString("message.connector.fileUpload.enabled");
	}

	public String getFileUploadDisabled() {
		return getString("message.connector.fileUpload.disabled");
	}

	public String getFileBrowsingEnabled() {
		return getString("message.connector.fileBrowsing.enabled");
	}

	public String getFileBrowsingDisabled() {
		return getString("message.connector.fileBrowsing.disabled");
	}

	public String getFolderCreationEnabled() {
		return getString("message.connector.folderCreation.enabled");
	}

	public String getFolderCreationDisabled() {
		return getString("message.connector.folderCreation.disabled");
	}

	public String getInvalidCommand() {
		return getString("message.connector.invalid_command_specified");
	}

	public String getInvalidType() {
		return getString("message.connector.invalid_type_specified");
	}

	public String getInvalidCurrentFolder() {
		return getString("message.connector.invalid_current_folder_speficied");
	}

}
