/* 
 * FCKdtd2js - FCKeditor JavaScript DTD map generator - http://www.fckeditor.net 
 * Copyright (C) 2003-2007 Frederico Caldeira Knabben 
 * 
 * == BEGIN LICENSE == 
 * 
 * Licensed under the terms of any of the following licenses at your 
 * choice: 
 * 
 *  - GNU General Public License Version 2 or later (the "GPL") 
 *    http://www.gnu.org/licenses/gpl.html 
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL") 
 *    http://www.gnu.org/licenses/lgpl.html 
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL") 
 *    http://www.mozilla.org/MPL/MPL-1.1.html 
 * 
 * == END LICENSE == 
 */ 

package net.fckeditor.devutil.dtd;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * Represents a group of elements. 
 * 
 * The groups is simply a set, but contains additional methods for finding element
 * intersections, and dividing content into subgroups.
 */
public class ElementGroup<T> extends HashSet<T> {

	/**
	 * Collection containing subgroups.
	 */
	protected Set<ElementGroup<T>> _subGroups;
	
	/**
	 * Creates a new elementgroup instance.
	 */
	public ElementGroup() {
		_subGroups = new HashSet<ElementGroup<T>>();
	}

	/**
	 * Finds the intersection between the elements in the current and
	 * supplied group, combine all common elements in a new group and 
	 * returns is. If there are no common elements in the groups, an
	 * empty group is returned.
	 * 
	 * @param group The group to create intersection with
	 * @return New group that contains the intersecting elements
	 */
	public ElementGroup<T> intersectGroup(ElementGroup<T> group) {
		// Find the smalles and largest of the groups.
		ElementGroup<T> smallGroup = group.size()<this.size()?group:this;
		ElementGroup<T> largeGroup = group.size()>=this.size()?group:this;
		
		ElementGroup<T> intersection = new ElementGroup<T>();
		
		// Iterate all elements in the small group and compare 
		// them with the elements in the large 
		for(Iterator<T> it = smallGroup.iterator(); it.hasNext(); ) {
			T currElement = it.next();
			if(largeGroup.contains(currElement)) {
				intersection.add(currElement);
			}
		}
		
		return intersection;
	}
	
	/**
	 * Adds a given subgroup to the current group. All elements in 
	 * the new subgroup which is also in the current group will be 
	 * removed from the current group.
	 * 
	 * @param group The subgroup to add.
	 */
	public void addSubGroup(ElementGroup<T> group) {
		// Remove all elements already found in subgroup.
		this.removeAll(group);
		// Add the subgroup.
		_subGroups.add(group);
	}

	/**
	 * Returns the groups subgroups as a set.
	 * @return Subgroups as a set.
	 */
	public Set<ElementGroup<T>> getSubGroups() {
		return _subGroups;
	}
	
	/**
	 * Calculates the total number of elements in this and all subgroups.
	 *  
	 * @return Total number of elements in group.
	 */
	public int getTotalSize() {
		int start = this.size();
		if(_subGroups.size()==0) return start;
		for(Iterator<ElementGroup<T>> it = _subGroups.iterator(); it.hasNext(); ) {
			start += it.next().getTotalSize();
		}
		return start;
	}
	
	/**
	 * Returns a string containing all elements in group.
	 */
	public String toString() {
		return recursiveToString("");
	}
	private String recursiveToString(String indent) {
		String result = indent + super.toString() + " ("+System.identityHashCode(this)+")\n";
		for(Iterator<ElementGroup<T>> it = _subGroups.iterator(); it.hasNext(); ) {
			result += indent + it.next().recursiveToString(indent + "  ");
		}
		return result;
	}
	
	/**
	 * Compares the current elementgroup with the specified group
	 * @param cmpGroup The group to compare to
	 * @return True if the groups are equal, and false if not.
	 */
	public boolean equals(ElementGroup cmpGroup) {
		// If the given reference equals this, they must be the same.
		if(this == cmpGroup) return true;
		// If the super method returns false, the groups are not equal.
		if(!super.equals(cmpGroup)) return false;
		// Then we need to check all subgroups.
		return _subGroups.equals(cmpGroup._subGroups);
	}
	
	/**
	 * Returns hashcode for the object.
	 * @return Hashcode for element group.
	 */
	public int hashCode() {
		int hashCode = 0;
		// Create hashcode from group elements.
		for(Iterator it = iterator(); it.hasNext(); ) {
			hashCode ^= it.next().hashCode();
		}
		// Include subgroups as well.
		for(Iterator<ElementGroup<T>> it = _subGroups.iterator(); it.hasNext(); ) {
			hashCode ^= it.next().hashCode();
		}
		return hashCode;
	}
}
