/* 
 * FCKdtd2js - FCKeditor JavaScript DTD map generator - http://www.fckeditor.net 
 * Copyright (C) 2003-2007 Frederico Caldeira Knabben 
 * 
 * == BEGIN LICENSE == 
 * 
 * Licensed under the terms of any of the following licenses at your 
 * choice: 
 * 
 *  - GNU General Public License Version 2 or later (the "GPL") 
 *    http://www.gnu.org/licenses/gpl.html 
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL") 
 *    http://www.gnu.org/licenses/lgpl.html 
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL") 
 *    http://www.mozilla.org/MPL/MPL-1.1.html 
 * 
 * == END LICENSE == 
 */ 

package net.fckeditor.devutil.dtd;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * This is a simple utility class that converts a map of elementgroups
 * into Javascript code.
 */
public class ElementGroupMapJavascriptBuilder {

	private static final String DTDMAP_NAME = "FCK.DTD";
	private static final String MERGE_FUNCTION = "X";
	private static final String MERGE_FUNCTION_DEF = "X = FCKTools.Merge ;";
	private static final String REFERENCE_ALPHABET = "ABCDEFGHIJKLMNOPQRSTUVWYZ";
	
	private int _varCount;
	private Map<ElementGroup, String> _subGroupAliasMap;
	
	/**
	 * Takes a groupmap and generates JavaScript code from it.
	 * @param groupMap
	 * @return
	 */
	public String buildJavaScript(String sourceDesc, Map<String,ElementGroup> groupMap) {
		// Reset the var counter.
		_varCount = 0;
		_subGroupAliasMap = new HashMap<ElementGroup, String>();
		
		// Prepare a stringbuilder.
		StringBuilder str = new StringBuilder();
		
		// Append the header.
		appendHeader(str, sourceDesc);
		
		// Append element map code.
		appendElementMapFunctionCode(str, groupMap);
		
		
		return str.toString();
	}
	
	
	protected void appendHeader(StringBuilder str, String sourceDesc) {
		
		str.append("/*\n");
		str.append(" * FCKeditor - The text editor for Internet - http://www.fckeditor.net\n");
		str.append(" * Copyright (C) 2003-2007 Frederico Caldeira Knabben\n");
		str.append(" *\n");
		str.append(" * == BEGIN LICENSE ==\n");
		str.append(" *\n");
		str.append(" * Licensed under the terms of any of the following licenses at your\n");
		str.append(" * choice:\n");
		str.append(" *\n");
		str.append(" *  - GNU General Public License Version 2 or later (the \"GPL\")\n");
		str.append(" *    http://www.gnu.org/licenses/gpl.html\n");
		str.append(" *\n");
		str.append(" *  - GNU Lesser General Public License Version 2.1 or later (the \"LGPL\")\n");
		str.append(" *    http://www.gnu.org/licenses/lgpl.html\n");
		str.append(" *\n");
		str.append(" *  - Mozilla Public License Version 1.1 or later (the \"MPL\")\n");
		str.append(" *    http://www.mozilla.org/MPL/MPL-1.1.html\n");
		str.append(" *\n");
		str.append(" * == END LICENSE ==\n");
		str.append(" *\n");
		str.append(" * Contains the DTD mapping for <DTD_NAME>.\n");
		str.append(" * ").append(sourceDesc);
		str.append("\n */\n");
	}
	
	
	protected void appendElementMapFunctionCode(StringBuilder outStr, Map<String,ElementGroup> groupMap) {
		// Create an internal stringbuilder for preparing js.
		StringBuilder str = new StringBuilder();
		
		// Create a map containing a count of the base group occurrence.
		Map<ElementGroup, Integer> baseGroupCountMap = new HashMap<ElementGroup, Integer>();	
		
		// Loop through all elements.
		for(Iterator<String> it=groupMap.keySet().iterator(); it.hasNext(); ) {
			// Retrieve the element name and the group.
			String elementName = it.next();
			ElementGroup currGroup = groupMap.get(elementName);
			
			// Start by processing all subgroups of the group.
			appendMissingSubGroupsRecursive(str, currGroup);
			
			// Count the group...
			if(!baseGroupCountMap.containsKey(currGroup)) {
				baseGroupCountMap.put(currGroup, 1);
			}
			else {
				// Increase the counters...
				baseGroupCountMap.put(currGroup, baseGroupCountMap.get(currGroup) + 1);
			}
		}

		str.append("\n");
		
		// Based on the counts, we determine if we need to create references...
		for(Iterator<ElementGroup> it = baseGroupCountMap.keySet().iterator(); it.hasNext(); ) {
			ElementGroup group = it.next();
			
			// Calculate character count of group...
			int contentSize = (group.getSubGroups().size() * 2) + 4;
			if(group.size() > 0) {
				for(Iterator<String> it2 = group.iterator(); it2.hasNext(); ) {
					String elementName = it2.next();
					contentSize += elementName.length() + 3;
				}
				contentSize += 2;
			}
			// Skip group if size = 4, since the group is then empty.
			if(contentSize == 4) continue;
			
			// Calculate sizes for creating a reference, and using the group direcly.
			int occurrenceCount = baseGroupCountMap.get(group);
			int referenceSize = (occurrenceCount * 2) + contentSize + 2;
			int directUseSize = (occurrenceCount * contentSize);
			
			//System.out.println("Occurrences: "+occurrenceCount+", Size: "+contentSize+" ,ReferenceSize: " + referenceSize + ", Direct use size: " + directUseSize);
			//System.out.println(group.toString());
			
			// If the reference size is less than the direct use size, we create a reference for the group.
			if(referenceSize < directUseSize) {
				appendGroupReference(str, group);
			}
		}
		
		// Append a newline...
		str.append("\n");
		str.append("    return {\n");
		
		// Loop through all elements.
		for(Iterator<String> it=groupMap.keySet().iterator(); it.hasNext(); ) {
			// Retrieve the element name and the group.
			String elementName = it.next();
			ElementGroup currGroup = groupMap.get(elementName);
			
			// Then output the root element.
			str.append("        ");
			appendEscapedJsMapKey(str, elementName);
			str.append(": ");
			
			
			// Make sure we append references to all subgroups.
			appendGroupMapContent(str, currGroup);
			if(it.hasNext()) str.append(", ");
			str.append("\n");
		}		
		str.append("    } ;\n");
		
		// Append result to final output stream.
		// Append function definition.
		outStr.append(DTDMAP_NAME).append(" = (function()\n{\n    ");
		outStr.append(MERGE_FUNCTION_DEF).append("\n\n    var ");
		
		for(Iterator<String> it = _subGroupAliasMap.values().iterator(); it.hasNext(); ) {
			String varname = it.next();
			outStr.append(varname);
			if(it.hasNext()) outStr.append(',');
		}
		
		outStr.append("; \n");
		outStr.append(str).append("})() ;");
	}
	
	protected void appendMissingSubGroupsRecursive(StringBuilder str, ElementGroup currGroup) {
		// Get subgroups from element group.
		Set<ElementGroup> subGroups = currGroup.getSubGroups();
		
		// If the group is completely empty, we can ignore it.
		if(subGroups.size()==0 && currGroup.size()==0) return;

		// if there are subgroups, we should process these..
		if(subGroups.size()>0) {
			// Process all subgroups first.
			for(Iterator<ElementGroup> it = subGroups.iterator(); it.hasNext(); ) {
				ElementGroup currSubGroup = it.next();
				
				// Check if there already is an alias for this ElementGroup.
				if(_subGroupAliasMap.containsKey(currSubGroup)) continue;
				
				// Do the tree recursively...
				appendMissingSubGroupsRecursive(str, currSubGroup);
				
				// Then append reference to the group
				appendGroupReference(str, currSubGroup);
			}
		}
	}

	protected void appendGroupReference(StringBuilder str, ElementGroup group) {
		// Create group alias.
		String groupAlias = createNextGroupAlias();
		
		// Then output all elements in group.
		str.append("    ").append(groupAlias).append(" = ");
		appendGroupMapContent(str, group);
		str.append(" ;\n");
		
		// Register the groupalias in alias map.
		_subGroupAliasMap.put(group, groupAlias);
	}
	
	protected String createNextGroupAlias() {
		// Create buffer to hold the reference.
		int baseLen = REFERENCE_ALPHABET.length();
		// Prepare for looping.
		StringBuilder str = new StringBuilder();
		int currNum = _varCount + baseLen;
		do {
			str.append(REFERENCE_ALPHABET.charAt(currNum % baseLen));
			currNum /= baseLen;
		}
		while(currNum > 1);		
		
		_varCount++;
		return str.reverse().toString();
	}
	
	protected void appendGroupMapContent(StringBuilder str, ElementGroup currGroup) {
		
		// If there is a reference for the group, we just add this.
		if(_subGroupAliasMap.containsKey(currGroup)) {
			str.append(_subGroupAliasMap.get(currGroup));
			return;
		}
		
		// Get subgroups.
		Set<ElementGroup> subGroups = currGroup.getSubGroups();
		
		// If there are more than one groups totally, combine them.
		if((currGroup.size()>0?1:0) + subGroups.size() > 1) str.append(MERGE_FUNCTION + "(");
		
		// Add group start.
		if(currGroup.size() > 0 || subGroups.size() == 0) { 
			str.append("{");
			
			// Add all elements in the group.
			for(Iterator<String> it = currGroup.iterator(); it.hasNext(); ) {
				String innerElementName = it.next();
				appendEscapedJsMapKey(str, innerElementName);
				str.append(":1");
				if(it.hasNext()) str.append(", ");
			}
			
			str.append("}");
		}		
		// Check if any subgroups exists.
		if(subGroups.size()==0) return;
		// Append all subgroups
		if(currGroup.size() > 0) str.append(", ");
		//str.append("'#':[");
		for(Iterator<ElementGroup> it = subGroups.iterator(); it.hasNext(); ) {
			ElementGroup group = it.next();
			str.append(_subGroupAliasMap.get(group));
			if(it.hasNext()) str.append(", ");
		}
		//str.append("]");
		
		if(subGroups.size() + currGroup.size() > 1) str.append(")");
	}
	
	protected void appendEscapedJsMapKey(StringBuilder str, String key) {
		if("#".equals(key) || "var".equals(key)) { // || "label".equals(key)
			str.append('\'').append(key).append('\'');
		}
		else {
			str.append(key);
		}
	}
	
}
