/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.handlers;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.ResourceBundle;

import javax.servlet.http.HttpServletRequest;

import net.fckeditor.localization.LocaleResolver;
import net.fckeditor.tool.Utils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class wraps to a {@link LocaleResolver} implementation and retrieves
 * localized strings for a given request. This class safely returns default
 * strings if anything fails. Loaded strings will be cached on load.
 * 
 * @version $Id: LocalizedMessages.java 2704 2008-11-23 21:35:24Z mosipov $
 * 
 */
public class LocalizedMessages {

	private static Map<Locale, LocalizedMessages> prototypes = new HashMap<Locale, LocalizedMessages>();
	private Properties properties;
	private static LocaleResolver localeResolver;
	private static final Locale NEUTRAL = new Locale("");
	private static final Logger logger = LoggerFactory
			.getLogger(LocalizedMessages.class);

	/**
	 * This method return an instance of this class for the given locale.
	 * 
	 * @param request
	 *            The current request instance
	 * @return The LP instance with localized strings.
	 */
	public static LocalizedMessages getMessages(HttpServletRequest request) {

		Locale locale;

		if (request == null)
			locale = NEUTRAL;
		else
			locale = getLocaleResolverInstance().resolveLocale(request);

		if (locale == null)
			locale = NEUTRAL;

		LocalizedMessages lp = prototypes.get(locale);
		if (lp == null) {
			lp = new LocalizedMessages(locale);
			prototypes.put(locale, lp);
		}

		// for now we don't need any cloning since the values are accessed
		// read-only
		return lp;

	}

	private static LocaleResolver getLocaleResolverInstance() {

		if (localeResolver == null) {
			String className = PropertiesLoader
					.getProperty("localization.localeResolverImpl");

			if (Utils.isEmpty(className))
				logger
						.error("Empty LocaleResolver implementation class name provided");
			else
				try {
					Class<?> clazz = Class.forName(className);
					localeResolver = (LocaleResolver) clazz.newInstance();
					logger.info("LocaleResolver initialized to {}", className);
				} catch (Exception e) {
					logger
							.error(
									"LocaleResolver implementation {} could not be instantiated",
									className);
				}
		}

		return localeResolver;
	}

	/**
	 * Loads the default strings and the string for the given locale and only
	 * for this locale.
	 * 
	 * @param locale
	 *            Given locale
	 */
	private LocalizedMessages(Locale locale) {

		ResourceBundle rb = null;
		try {
			rb = ResourceBundle.getBundle("fckeditor", locale, Thread
					.currentThread().getContextClassLoader());
		} catch (Exception e) {
			logger.debug("No resource bundle found for locale {}", locale);
		}

		this.properties = new Properties(PropertiesLoader.getProperties());

		if (rb != null
				&& rb.getLocale().getLanguage().equals(locale.getLanguage())) {
			Enumeration<String> keys = rb.getKeys();

			while (keys.hasMoreElements()) {
				String key = keys.nextElement();
				properties.setProperty(key, rb.getString(key));
			}
		}

	}

	private String getString(String key) {
		return properties.getProperty(key);
	}

	/**
	 * Getter for localized <code>message.compatible_browser.yes</code> property
	 * 
	 * @return CompatibleBrowserYes
	 */
	public String getCompatibleBrowser() {
		return getString("message.compatible_browser.yes");
	}

	/**
	 * Getter for localized <code>message.compatible_browser.no</code> property
	 * 
	 * @return CompatibleBrowserNo
	 */
	public String getNotCompatibleBrowser() {
		return getString("message.compatible_browser.no");
	}

	/**
	 * Getter for localized <code>message.connector.fileUpload.enabled</code>
	 * property
	 * 
	 * @return FileUploadEnabled
	 */
	public String getFileUploadEnabled() {
		return getString("message.connector.fileUpload.enabled");
	}

	/**
	 * Getter for localized <code>message.connector.fileUpload.disabled</code>
	 * property
	 * 
	 * @return FileUploadDisabled
	 */
	public String getFileUploadDisabled() {
		return getString("message.connector.fileUpload.disabled");
	}

	/**
	 * Getter for localized <code>message.connector.fileBrowsing.enabled</code>
	 * property
	 * 
	 * @return FileBrowsingEnabled
	 */
	public String getFileBrowsingEnabled() {
		return getString("message.connector.fileBrowsing.enabled");
	}

	/**
	 * Getter for localized <code>message.connector.fileBrowsing.disabled</code>
	 * property
	 * 
	 * @return FileBrowsingDisabled
	 */
	public String getFileBrowsingDisabled() {
		return getString("message.connector.fileBrowsing.disabled");
	}

	/**
	 * Getter for localized
	 * <code>message.connector.folderCreation.enabled</code> property
	 * 
	 * @return FolderCreationEnabled
	 */
	public String getFolderCreationEnabled() {
		return getString("message.connector.folderCreation.enabled");
	}

	/**
	 * Getter for localized
	 * <code>message.connector.folderCreation.disabled</code> property
	 * 
	 * @return FolderCreationDisabled
	 */
	public String getFolderCreationDisabled() {
		return getString("message.connector.folderCreation.disabled");
	}

	/**
	 * Getter for localized
	 * <code>message.connector.invalid_command_specified</code> property
	 * 
	 * @return InvalidCommandSpecified
	 */
	public String getInvalidCommand() {
		return getString("message.connector.invalid_command_specified");
	}

	/**
	 * Getter for localized
	 * <code>message.connector.invalid_type_specified</code> property
	 * 
	 * @return InvalidTypeSpecified
	 */
	public String getInvalidType() {
		return getString("message.connector.invalid_type_specified");
	}

	/**
	 * Getter for localized
	 * <code>message.connector.invalid_current_folder_speficied</code> property
	 * 
	 * @return InvalidCurrentFolderSpecified
	 */
	public String getInvalidCurrentFolder() {
		return getString("message.connector.invalid_current_folder_speficied");
	}

}
