/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.handlers;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import net.fckeditor.tool.Utils;

/**
 * This class maps to all file browser resource types and their allowed/denied
 * extensions. The design is restricted to the static types only, you cannot
 * create your own types. External access to all fields is read-only.
 * 
 * @version $Id: ResourceType.java 2738 2008-11-28 22:53:16Z mosipov $
 */
public class ResourceType {

	/** The name of the resource type */
	private String name;
	/** The absolute path of the resource type */
	private String path;
	/** The allowed extensions */
	private Set<String> allowedEextensions;
	/** The denied extensions */
	private Set<String> deniedExtensions;

	/** Map holding a String to ResourceType reference */
	private static Map<String, ResourceType> types = new HashMap<String, ResourceType>(
			4);

	/** Resource type <code>File</code> */
	public static final ResourceType FILE = new ResourceType("file",
			PropertiesLoader.getFileResourceTypePath(), Utils
					.getSet(PropertiesLoader
							.getFileResourceTypeAllowedExtensions()), Utils
					.getSet(PropertiesLoader
							.getFileResourceTypeDeniedExtensions()));
	/** Resource type <code>Flash</code> */
	public static final ResourceType FLASH = new ResourceType("flash",
			PropertiesLoader.getFlashResourceTypePath(), Utils
					.getSet(PropertiesLoader
							.getFlashResourceTypeAllowedExtensions()), Utils
					.getSet(PropertiesLoader
							.getFlashResourceTypeDeniedExtensions()));
	/** Resource type <code>Image</code> */
	public static final ResourceType IMAGE = new ResourceType("image",
			PropertiesLoader.getImageResourceTypePath(), Utils
					.getSet(PropertiesLoader
							.getImageResourceTypeAllowedExtensions()), Utils
					.getSet(PropertiesLoader
							.getImageResourceTypeDeniedExtensions()));
	/** Resource type <code>Media</code> */
	public static final ResourceType MEDIA = new ResourceType("media",
			PropertiesLoader.getMediaResourceTypePath(), Utils
					.getSet(PropertiesLoader
							.getMediaResourceTypeAllowedExtensions()), Utils
					.getSet(PropertiesLoader
							.getMediaResourceTypeDeniedExtensions()));

	static {
		types.put(FILE.getName(), FILE);
		types.put(FLASH.getName(), FLASH);
		types.put(IMAGE.getName(), IMAGE);
		types.put(MEDIA.getName(), MEDIA);
	}

	/**
	 * This constructor has been made intentionally made private to provide
	 * pre-defined types only.
	 * 
	 * @param name
	 * @param path
	 * @param allowedEextensions
	 * @param deniedExtensions
	 * @throws IllegalArgumentException
	 *             if both sets are empty
	 * @throws IllegalArgumentException
	 *             if both sets contain extensions
	 */
	private ResourceType(final String name, final String path,
			final Set<String> allowedEextensions,
			final Set<String> deniedExtensions) {
		this.name = name;
		this.path = path;

		if (allowedEextensions.isEmpty() && deniedExtensions.isEmpty())
			throw new IllegalArgumentException(
					"Both sets are empty, one has always to be filled");

		if (!allowedEextensions.isEmpty() && !deniedExtensions.isEmpty())
			throw new IllegalArgumentException(
					"Both sets contain extensions, only one can be filled at the same time");

		this.allowedEextensions = allowedEextensions;
		this.deniedExtensions = deniedExtensions;
	}

	/**
	 * Returns the name of the resource type.
	 * 
	 * @return the name
	 */
	public String getName() {
		return name;
	}

	/**
	 * Returns the absolute path of the resource type. This path will later be
	 * resolved against the
	 * {@link RequestCycleHandler#getUserFilesPath(javax.servlet.http.HttpServletRequest)
	 * UserFilesPath}.
	 * 
	 * @return the path
	 */
	public String getPath() {
		return path;
	}

	/**
	 * Returns a read-only reference to the allowed extensions set.
	 * 
	 * @return the allowedEextensions
	 */
	public Set<String> getAllowedEextensions() {
		return Collections.unmodifiableSet(allowedEextensions);
	}

	/**
	 * Returns a read-only reference to the denied extensions set.
	 * 
	 * @return the deniedExtensions
	 */
	public Set<String> getDeniedExtensions() {
		return Collections.unmodifiableSet(deniedExtensions);
	}

	/**
	 * Returns the {@link ResourceType} for a specified string. <br />
	 * <em>The retrieval is case-insensitive!</em>
	 * 
	 * @param name
	 *            Resource type name.
	 * @return A {@link ResourceType} object holding the value represented by
	 *         the string argument.
	 * @throws IllegalArgumentException
	 *             If 'name' is <code>null</code>, empty, or does not exist.
	 */
	public static ResourceType valueOf(final String name) {
		if (Utils.isEmpty(name))
			throw new IllegalArgumentException("Parameter name is empty");

		ResourceType rt = types.get(name.toLowerCase());
		if (rt == null)
			throw new IllegalArgumentException(
					"No suitable resource type found");
		return rt;
	}

	/**
	 * Determines if a specified string represents a valid resource type.<br />
	 * <em>The retrieval is case-insensitive!</em>
	 * 
	 * @param name
	 *            Resource type name.
	 * @return <code>true</code> if the string represents a valid resource type
	 *         else <code>false</code>.
	 */
	public static boolean isValidType(final String name) {
		return (Utils.isEmpty(name)) ? false : types.containsKey(name
				.toLowerCase());
	}

	/**
	 * This method wraps {@link #valueOf(String)}. It returns <code>null</code>
	 * instead of throwing an IllegalArgumentException.<br />
	 * <em>The retrieval is case-insensitive!</em>
	 * 
	 * @param name
	 *            Resource type string.
	 * @return A {@link ResourceType} object holding the value represented by
	 *         the string argument, or <code>null</code>.
	 */
	public static ResourceType getResourceType(final String name) {
		try {
			return ResourceType.valueOf(name.toLowerCase());
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * This method wraps {@link #getResourceType(String)}. It returns
	 * {@link #FILE} instead of returning <code>null</code>.<br />
	 * <em>The retrieval is case-insensitive!</em>
	 * 
	 * @param name
	 *            Resource type string.
	 * @return A {@link ResourceType} object holding the value represented by
	 *         the string argument.
	 */
	public static ResourceType getDefaultResourceType(final String name) {
		if (Utils.isEmpty(name))
			return null;
		ResourceType rt = getResourceType(name);
		return rt == null ? FILE : rt;
	}

	/**
	 * Determines if an extension passes/suits the allowed or denied set.<br />
	 * <em>Empty extensions will always fail!</em>
	 * 
	 * 
	 * @param extension
	 *            Extension string.
	 * @return <code>true</code> if the extension passes, else
	 *         <code>false</code>.
	 */
	public boolean isAllowedExtension(final String extension) {
		if (Utils.isEmpty(extension))
			return false;
		String ext = extension.toLowerCase();
		if (allowedEextensions.isEmpty())
			return !deniedExtensions.contains(ext);
		if (deniedExtensions.isEmpty())
			return allowedEextensions.contains(ext);
		return false;
	}

	/**
	 * This method wraps {@link #isAllowedExtension(String)}. It simply negates
	 * the return value.
	 * 
	 * @param extension
	 *            Extension string.
	 * @return <code>true</code> if extension is not fails, else
	 *         <code>false</code>.
	 */
	public boolean isNotAllowedExtension(final String extension) {
		return !isAllowedExtension(extension);
	}

	/**
	 * Compares <code>this</code> with the passed object against the name field.
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;

		if (obj == null || this.getClass() != obj.getClass())
			return false;

		final ResourceType rt = (ResourceType) obj;
		return name.equals(rt.getName());
	}

	/**
	 * Computer hash code based on the name field.
	 */
	@Override
	public int hashCode() {
		return name.hashCode();
	}
}
