/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.connector;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.fckeditor.connector.exception.FolderAlreadyExistsException;
import net.fckeditor.connector.exception.InvalidCurrentFolderException;
import net.fckeditor.connector.exception.InvalidNewFolderNameException;
import net.fckeditor.connector.exception.WriteException;
import net.fckeditor.handlers.Command;
import net.fckeditor.handlers.ConnectorHandler;
import net.fckeditor.handlers.PropertiesLoader;
import net.fckeditor.handlers.RequestCycleHandler;
import net.fckeditor.handlers.ResourceType;
import net.fckeditor.requestcycle.Context;
import net.fckeditor.requestcycle.ThreadLocalData;
import net.fckeditor.response.GetResponse;
import net.fckeditor.response.UploadResponse;
import net.fckeditor.tool.Utils;
import net.fckeditor.tool.UtilsFile;
import net.fckeditor.tool.UtilsResponse;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * TODO Review documentation
 * This class is called by {@link ConnectorServlet}. It verifies the request
 * parameters and forwards them to the corresponding {@link Connector} methods.
 * If an error has occurred inside these methods, well-defined exceptions from
 * {@link net.fckeditor.connector.exception} will be thrown. These exceptions or
 * the return values of {@link Connector} methods will be transfered to the
 * HttpServletResponse object.
 * 
 * @version $Id: Dispatcher.java 2765 2008-12-14 19:59:38Z mosipov $
 */
public class Dispatcher {
	private static final Logger logger = LoggerFactory
			.getLogger(Dispatcher.class);
	private Connector connector = null;
	
	/**
	 * Initializes the {@link Connector}.
	 * 
	 * @param servletContext
	 */
	protected Dispatcher(final ServletContext servletContext) throws Exception {
		this.connector = ConnectorHandler.getConnector();
		this.connector.init(servletContext);
	}

	/**
	 * Manages <code>GET</code> requests (<code>GetFolders</code>,
	 * <code>GetFoldersAndFiles</code>, <code>CreateFolder</code>).<br/>
	 * 
	 * The method accepts commands sent in the following format:<br/>
	 * <code>connector?Command=&lt;CommandName&gt;&Type=&lt;ResourceType&gt;&CurrentFolder=&lt;FolderPath&gt;</code>
	 * <p>
	 * It executes the relevant commands and then returns the result to the
	 * client in XML format.
	 * </p>
	 */
	public void doGet(final HttpServletRequest request,
			final HttpServletResponse response) throws IOException {
		logger.debug("Entering Dispatcher#doGet");
		Context context = ThreadLocalData.getContext();

		request.setCharacterEncoding("UTF-8");
		response.setCharacterEncoding("UTF-8");
		response.setContentType("application/xml; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();

		context.logBaseParameters();
		
		GetResponse getResponse = null;
		// check parameters
		if (!Command.isValidForGet(context.getCommandStr()))
			getResponse = GetResponse.getErrorInvalidCommand();
		else if (!ResourceType.isValidType(context.getTypeStr()))
			getResponse = GetResponse.getErrorInvalidType();
		else if (!UtilsFile.isValidPath(context.getCurrentFolderStr()))
			getResponse = GetResponse.getErrorInvalidCurrentFolder();
		else {
			
			// in contrast to doPost the referrer has to send an explicit type
			ResourceType type = context.getResourceType();
			Command command = context.getCommand();
			
			// check permissions for user action
			if ((command.equals(Command.GET_FOLDERS) || command
					.equals(Command.GET_FOLDERS_AND_FILES))
					&& !RequestCycleHandler.isEnabledForFileBrowsing(request))
				getResponse = GetResponse.getErrorFileBrowsingDisabled();
			else if (command.equals(Command.CREATE_FOLDER)
					&& !RequestCycleHandler.isEnabledForFolderCreation(request))
				getResponse = GetResponse.getErrorFolderCreationDisabled();
			else {
				// make the connector calls, catch its exceptions and generate
				// the proper response object
				try {
					if (command.equals(Command.CREATE_FOLDER)) {
						String newFolderNameStr = request
								.getParameter("NewFolderName");
						logger.debug("Parameter NewFolderName: {}",
								newFolderNameStr);				
						String sanitizedNewFolderNameStr = UtilsFile
								.sanitizeFolderName(newFolderNameStr);
						if (Utils.isEmpty(sanitizedNewFolderNameStr))
							getResponse = GetResponse
									.getErrorInvalidFolderName();
						else {
							logger.debug(
									"Parameter NewFolderName (sanitized): {}",
									sanitizedNewFolderNameStr);
							connector.createFolder(type, context
									.getCurrentFolderStr(),
									sanitizedNewFolderNameStr);
							getResponse = GetResponse.getOK();
						}
					} else if (command.equals(Command.GET_FOLDERS)
							|| command
									.equals(Command.GET_FOLDERS_AND_FILES)) {
						String url = UtilsResponse.getUrl(RequestCycleHandler
								.getUserFilesPath(request), type, context
								.getCurrentFolderStr());
						getResponse = getFoldersAndOrFiles(command, type, context
								.getCurrentFolderStr(), url);
					}
				} catch (InvalidCurrentFolderException e) {
					getResponse = GetResponse.getErrorInvalidCurrentFolder();
				} catch (InvalidNewFolderNameException e) {
					getResponse = GetResponse.getErrorInvalidFolderName();
				} catch (FolderAlreadyExistsException e) {
					getResponse = GetResponse.getErrorFolderAlreadyExists();
				}
			}
		}
		
		out.print(getResponse);
		out.flush();
		out.close();
		logger.debug("Exiting Dispatcher#doGet");
	}
	
	/**
	 * Helper to make the right {@link Connector}-calls for
	 * <code>GetFolders</code> and <code>GetFoldersAndFiles</code>.
	 * 
	 * @param command
	 *            should be only {@link Command#GET_FOLDERS} or
	 *            {@link Command#GET_FOLDERS_AND_FILES}!!
	 * @param type
	 * @param currentFolderStr
	 * @param url
	 * @return
	 * @throws InvalidCurrentFolderException
	 */
	private GetResponse getFoldersAndOrFiles(final Command command,
			final ResourceType type, final String currentFolderStr,
			final String url) throws InvalidCurrentFolderException {
		GetResponse getResponse = new GetResponse(command, type,
				currentFolderStr, url);
		getResponse.setFolders(connector.getFolders(type, currentFolderStr));
		if (command.equals(Command.GET_FOLDERS_AND_FILES))
			getResponse.setFiles(connector.getFiles(type, currentFolderStr));
		return getResponse;
	}

	/**
	 * Manage the <code>POST</code> requests (<code>FileUpload</code>).<br />
	 * 
	 * The method accepts commands sent in the following format:<br />
	 * <code>connector?Command=&lt;(File|Quick)Upload&gt;&Type=&lt;ResourceType&gt;&CurrentFolder=&lt;FolderPath&gt;</code>
	 * with the file in the <code>POST</code> body.<br />
	 * <br />
	 * The Connector stores an uploaded file (renames a file if another exists
	 * with the same name) and then returns the JavaScript callback.
	 * 
	 * @throws IOException
	 *             if some unpredictable write error happens
	 */
	public void doPost(final HttpServletRequest request,
			final HttpServletResponse response) throws IOException {
		logger.debug("Entering Dispatcher#doPost");
		Context context = ThreadLocalData.getContext();
		
		request.setCharacterEncoding("UTF-8");
		response.setCharacterEncoding("UTF-8");
		response.setContentType("text/html; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();

		context.logBaseParameters();
		
		UploadResponse uploadResponse;
		// check permissions for user actions
		if (!RequestCycleHandler.isEnabledForFileUpload(request))
			uploadResponse = UploadResponse.getErrorFileUploadDisabled();
		// check parameters  
		else if (!Command.isValidForPost(context.getCommandStr()))
			uploadResponse = UploadResponse.getErrorInvalidCommand();
		else if (!ResourceType.isValidType(context.getTypeStr()))
			uploadResponse = UploadResponse.getErrorInvalidType();
		else if (!UtilsFile.isValidPath(context.getCurrentFolderStr()))
			uploadResponse = UploadResponse.getErrorInvalidCurrentFolder();
		else {

			// call the Connector#fileUpload
			ResourceType type = context.getDefaultResourceType();
			FileItemFactory factory = new DiskFileItemFactory();
			ServletFileUpload upload = new ServletFileUpload(factory);
			try {
				@SuppressWarnings("unchecked")
				List<FileItem> items = upload.parseRequest(request);
				// We upload just one file at the same time
				FileItem uplFile = items.get(0);
				// Some browsers transfer the entire source path not just the
				// filename
				String fileName = FilenameUtils.getName(uplFile.getName());
				logger.debug("Parameter NewFile: {}", fileName);
				// check the extension
				if (type.isNotAllowedExtension(FilenameUtils
						.getExtension(fileName)))
					uploadResponse = UploadResponse.getErrorInvalidExtension();
				// Secure image check (can't be done if QuickUpload)
				else if (type.equals(ResourceType.IMAGE)
						&& PropertiesLoader.isSecureImageUploads()
						&& !UtilsFile.isImage(uplFile.getInputStream())) {
					uploadResponse = UploadResponse.getErrorInvalidExtension();
				} else {
					String sanitizedFileName = UtilsFile
							.sanitizeFileName(fileName);
					logger.debug("Parameter NewFile (sanitized): {}",
							sanitizedFileName);
					String newFileName = connector.fileUpload(type, context
							.getCurrentFolderStr(), sanitizedFileName, uplFile
							.getInputStream());
					String fileUrl = UtilsResponse.fileUrl(RequestCycleHandler
							.getUserFilesPath(request), type, context
							.getCurrentFolderStr(), newFileName);

					if (sanitizedFileName.equals(newFileName))
						uploadResponse = new UploadResponse(
								UploadResponse.SC_OK, fileUrl);
					else {
						uploadResponse = new UploadResponse(
								UploadResponse.SC_RENAMED, fileUrl, newFileName);
						logger.debug("Parameter NewFile (renamed): {}",
								newFileName);
					}
				}
				uplFile.delete();
			} catch (InvalidCurrentFolderException e) {
				uploadResponse = UploadResponse.getErrorInvalidCurrentFolder();
			} catch (WriteException e) {
				uploadResponse = UploadResponse.getErrorUnknown();
				// FIXME what response should be send back?
			} catch (IOException e) {
				// TODO handle it maybe better?!
				uploadResponse = UploadResponse.getErrorUnknown();
			} catch (FileUploadException e) {
				uploadResponse = UploadResponse.getErrorUnknown();
			}
		}
		out.print(uploadResponse);
		out.flush();
		out.close();
		logger.debug("Exiting Dispatcher#doPost");
	}
	
	
}
