/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.handlers;

import java.util.HashMap;
import java.util.Map;

import net.fckeditor.tool.Utils;

/**
 * Handler for <code>GET</code> and <code>POST</code> commands.
 * 
 * @version $Id: Command.java 2865 2009-01-09 20:45:20Z mosipov $
 */
public class Command {

	private String name;
	private static Map<String, Command> getCommands = new HashMap<String, Command>(
			3);
	private static Map<String, Command> postCommands = new HashMap<String, Command>(
			2);
	public static final Command GET_FOLDERS = new Command(
			"GetFolders");
	public static final Command GET_FOLDERS_AND_FILES = new Command(
			"GetFoldersAndFiles");
	public static final Command CREATE_FOLDER = new Command(
			"CreateFolder");
	public static final Command FILE_UPLOAD = new Command(
			"FileUpload");
	public static final Command QUICK_UPLOAD = new Command(
			"QuickUpload");

	static {
		// initialize the get commands
		getCommands.put(GET_FOLDERS.getName(), GET_FOLDERS);
		getCommands.put(GET_FOLDERS_AND_FILES.getName(), GET_FOLDERS_AND_FILES);
		getCommands.put(CREATE_FOLDER.getName(), CREATE_FOLDER);
		
		// initialize the post commands
		postCommands.put(FILE_UPLOAD.getName(), FILE_UPLOAD);
		postCommands.put(QUICK_UPLOAD.getName(), QUICK_UPLOAD);
	}

	private Command(final String name) {
		this.name = name;
	}

	/**
	 * Getter for the name.
	 * 
	 * @return The command name
	 */
	public String getName() {
		return name;
	}

	/**
	 * Getter for an {@link Command} of a specified string.
	 * 
	 * @param name
	 *            A command to retrieve
	 * @return A {@link Command} object holding the value represented by
	 *         the string argument.
	 * @throws IllegalArgumentException
	 *             If 'name' is <code>null</code>, empty, or does not exist.
	 */
	public static Command valueOf(final String name) {
		if (Utils.isEmpty(name))
			throw new NullPointerException("Parameter name is null or empty");

		Command command = getCommands.get(name);
		if (command == null)
			command = postCommands.get(name);
		if (command == null)
			throw new IllegalArgumentException("No suitable command found");
		
		return command;
	}
	

	/**
	 * Checks if a specified string represents a valid <code>GET</code>
	 * command.
	 * 
	 * @param name
	 *            A command string to check
	 * @return <code>true</code> if the string representation is valid else
	 *         <code>false</code>.
	 */
	public static boolean isValidForGet(final String name) {
		return getCommands.containsKey(name);
	}

	/**
	 * Checks if a specified string represents a valid <code>POST</code>
	 * command.
	 * 
	 * @param name
	 *            A command string to check
	 * @return <code>true</code> if the string representation is valid else
	 *         <code>false</code>.
	 */
	public static boolean isValidForPost(final String name) {
		return postCommands.containsKey(name);
	}
	
	
	/**
	 * A wrapper for {@link #valueOf(String)}. It returns null instead of
	 * throwing an exception.
	 * 
	 * @param name A command string to check
	 * @return A {@link Command} object holding the value represented by
	 *         the string argument, or <code>null</code>.
	 */
	public static Command getCommand(final String name) {
		try {
			return Command.valueOf(name);
		} catch (Exception e) {
			return null;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;

		if (obj == null || this.getClass() != obj.getClass())
			return false;

		final Command command = (Command) obj;
		return name.equals(command.getName());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public int hashCode() {
		return name.hashCode();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return name;
	}
}
