/*
** This code has been placed into the Public Domain.
** This code was written by Timothy Gerard Endres in 1999.
** 
*/

package com.ice.tartool;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.zip.*;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;

import javax.swing.*;
import javax.swing.border.*;

import com.ice.tar.TarEntry;
import com.ice.tar.TarInputStream;
import com.ice.util.AWTUtilities;
import com.ice.util.UserProperties;


public
class		ArchivePanel
extends		JPanel
implements	ActionListener
	{
	private static final String		DIV_LOC_PROP = "archivePanel.divider.location";

	protected ArchiveTreePanel		treePan;
	protected JTextField			infoTitle;
	protected JTextArea				infoText;
	protected JToolBar				toolBar;
	protected JSplitPane			split;

	protected DecimalFormat			sizeFormat;
	protected DecimalFormat			sizeKFormat;
	protected SimpleDateFormat		modTimeFormat;
	protected String				fileDlgDir = null;


	public
	ArchivePanel()
		{
		super();

		this.setLayout( new BorderLayout() );

		this.toolBar = new JToolBar();
		this.toolBar.setFloatable( false );
		this.populateToolbar( this.toolBar );

		this.add( "North", toolBar );

		this.treePan = new ArchiveTreePanel();
		this.treePan.addActionListener( this );
		this.treePan.setMinimumSize( new Dimension( 25, 125 ) );
		this.treePan.setPreferredSize( new Dimension( 450, 300 ) );

		JPanel textPan = new JPanel();
		textPan.setLayout( new BorderLayout() );
		textPan.setBorder(
			new TitledBorder(
				new EtchedBorder( EtchedBorder.RAISED ), "Details" ) );

		this.infoTitle = new JTextField();
		this.infoTitle.setOpaque( false );
		this.infoTitle.setBorder( new EmptyBorder( 1, 0, 3, 0 ) );
		this.infoTitle.setFont( new Font( "SansSerif", Font.BOLD, 14 ) );
		textPan.add( "North", this.infoTitle );

		this.infoText = new JTextArea();
		this.infoText.setOpaque( false );
		this.infoText.setMinimumSize( new Dimension( 25, 50 ) );
		this.infoText.setPreferredSize( new Dimension( 450, 100 ) );
		this.infoText.setFont( new Font( "SansSerif", Font.PLAIN, 12 ) );
		textPan.add( "Center", this.infoText );

		this.split =
			new JSplitPane( JSplitPane.VERTICAL_SPLIT,
							true, this.treePan, textPan );

		this.add( "Center", this.split );

		this.modTimeFormat =
			new SimpleDateFormat( "MMM dd, yyyy 'at' hh:mm:ss a z" );

		this.sizeFormat =
			new DecimalFormat( "###,###,###,###" );

		this.sizeKFormat =
			new DecimalFormat( "###,###,###,###.0" );
		}

	protected void
	loadLayoutProps()
		{
		int loc = UserProperties.getProperty( DIV_LOC_PROP, -1 );
		if ( loc == -1 )
			{
			this.split.setDividerLocation( 0.75 );
			}
		else
			{
			this.split.setDividerLocation( loc );
			}
		}

	protected void
	saveLayoutProps()
		{
		int loc = this.split.getDividerLocation();
		UserProperties.setDynamicProperty
			( TarTool.CONFIG_MAIN, DIV_LOC_PROP, "" + loc );
		}

	public void
	adjustDividerLocation()
		{
		this.split.setDividerLocation( 0.8 );
		}

	public void
	openArchive( File archiveFile )
		throws IOException
		{
		this.treePan.openArchive( archiveFile );
		}

	public void
	actionPerformed( ActionEvent event )
		{
		String command = event.getActionCommand();

		if ( command.startsWith( "EXTRACT" ) )
			{
			this.treePan.actionPerformed( event );
			}
		else if ( command.equals( "NODE_SELECTED" ) )
			{
			ArchiveTreeNode ch = (ArchiveTreeNode) event.getSource();
			
			TarEntry entry = ch.getEntry();

			String modTimeStr =
				this.modTimeFormat.format( entry.getModTime() );

			StringBuffer info = new StringBuffer();

			if ( ch == this.treePan.getRootNode() )
				{
				this.infoTitle.setText( /*"Archive: " +*/ ch.getName() );

				String dirStr =
					this.sizeFormat.format( ch.getTotalDirectoryCount() );
				String fileStr =
					this.sizeFormat.format( ch.getTotalFileCount() );

				info.append( "Archive contains " + fileStr + " files.\n" );
				info.append( "Archive contains " + dirStr + " directories.\n" );
				}
			else if ( entry.isDirectory() )
				{
				this.infoTitle.setText( /*"Directory: " +*/ entry.getName() );

				info.append( "User  ID = " + entry.getUserId() + ", " );
				info.append( "Name = '" + entry.getUserName() + "'.\n" );

				info.append( "Group ID = " + entry.getGroupId() + ", " );
				info.append( "Name = '" + entry.getGroupName() + "'.\n" );

				String dirStr =
					this.sizeFormat.format( ch.getDirectoryCount() );
				String fileStr =
					this.sizeFormat.format( ch.getFileCount() );

				info.append
					( "Directory contains " + fileStr
						+ " files and " + dirStr + " directories.\n" );
				}
			else
				{
				this.infoTitle.setText( /*"File: " +*/ entry.getName() );

				info.append( "User  ID = " + entry.getUserId() + ", " );
				info.append( "Name = '" + entry.getUserName() + "'.\n" );

				info.append( "Group ID = " + entry.getGroupId() + ", " );
				info.append( "Name = '" + entry.getGroupName() + "'.\n" );

				String sizeByteStr =
					this.sizeFormat.format( entry.getSize() );

				String sizeKiloStr =
					this.sizeKFormat.format
						( ((double)entry.getSize()) / 1024.0 );

				info.append
					( "Size " + sizeByteStr
						+ " ( " + sizeKiloStr + "K ) bytes.\n" );
				}

			this.infoText.setText( info.toString() );
			}
		else if ( command.equals( "OPEN" ) )
			{
			File archiveFile = null;

			FileDialog dlg =
				new FileDialog(
					(Frame) this.getTopLevelAncestor(),
					"Open Tar Archive", FileDialog.LOAD );

			if ( this.fileDlgDir != null )
				dlg.setDirectory( this.fileDlgDir );

			dlg.show();

			String fileName = dlg.getFile();
			this.fileDlgDir = dlg.getDirectory();

			if ( fileName != null && this.fileDlgDir != null )
				{
				File f = new File( this.fileDlgDir, fileName );
				if ( f.exists() && f.isFile() && f.canRead() )
					{
					try {
						this.treePan.openArchive( f );
						}
					catch ( IOException ex )
						{
						ex.printStackTrace();
						// UNDONE
						}
					}
				}
			}
		else if ( command.equals( "CLOSE" ) )
			{
			try {
				this.infoText.setText( "" );
				this.infoTitle.setText( "" );
				this.treePan.closeArchive();
				}
			catch ( IOException ex )
				{
				ex.printStackTrace();
				}
			}
		else if ( command.equals( "EXTRACT" ) )
			{
			}
		else
			{
			System.err.println
				( "UNKNOWN Command '" + command + "'" );
			}
		}

	private void
	populateToolbar( JToolBar toolBar )
		{
		try {
			Image	iconImg;

			Image iOpen		= AWTUtilities.getImageResource
							( "/com/ice/tartool/images/open.gif" );
			Image iClose	= AWTUtilities.getImageResource
							( "/com/ice/tartool/images/close.gif" );
			Image iExtract	= AWTUtilities.getImageResource
							( "/com/ice/tartool/images/extract.gif" );


			Icon icon;
			JButton btn;

			icon = new ImageIcon( iOpen );
			btn = new JButton( icon );
			btn.addActionListener( this );
			btn.setActionCommand( "OPEN" );
			btn.setToolTipText( "Open Tar Archive" );
		
			toolBar.add( btn );

			toolBar.addSeparator();

			icon = new ImageIcon( iClose );
			btn = new JButton( icon );
			btn.addActionListener( this );
			btn.setActionCommand( "CLOSE" );
			btn.setToolTipText( "Close Archive" );
		
			toolBar.add( btn );

			toolBar.addSeparator();

			icon = new ImageIcon( iExtract );
			btn = new JButton( icon );
			btn.addActionListener( this );
			btn.setActionCommand( "EXTRACT" );
			btn.setToolTipText( "Extract Selected Files" );
		
			toolBar.add( btn );
			}
		catch ( IOException ex )
			{
			ex.printStackTrace();
			}
		}

	}

