/*
 * CKEditor - The text editor for Internet - http://ckeditor.com
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 *
 * == BEGIN LICENSE ==
 *
 * Licensed under the terms of any of the following licenses at your
 * choice:
 *
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 *
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 *
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 *
 * == END LICENSE ==
 */

(function()
{
	var guardDomWalkerNonEmptyTextNode = function( evt )
	{
		if ( evt.to && evt.to.type == CKEDITOR.NODE_TEXT && evt.to.$.length > 0 )
			this.stop();
		CKEDITOR.dom.domWalker.blockBoundary( { br : 1 } ).call( this, evt );
	};

	var domWalkerEventBridge = function()
	{
		var me = this,
			handler = function( evt ){ me.fire( evt.data.type, evt.data ); me._.actionEvents.push( evt.data ); };
		this._.walker.on( 'up', handler );
		this._.walker.on( 'down', handler );
		this._.walker.on( 'sibling', handler );
	};

	var fireCharacterEvent = function()
	{
		var obj = {
			type : 'character',
			character : this.textNode ? this.textNode.getText().charAt( this.offset ) : null,
			hitMatchBoundary : this._.matchBoundary
		};
		this.fire( 'character', obj );
		this._.actionEvents.push( obj );
		return { character : obj.character, events : this._.actionEvents };
	};

	var findDialog = function( editor, startupPage )
	{
		var characterWalker = function( cursorOrTextNode, offset )
		{
			var isCursor = ( cursorOrTextNode instanceof characterWalker );
			this._ = {
				matchBoundary : false,
				actionEvents : null
			};
			this.textNode = isCursor ? cursorOrTextNode.textNode : cursorOrTextNode,
			this.offset = isCursor ? cursorOrTextNode.offset : offset,
			this._.walker = new CKEDITOR.dom.domWalker( this.textNode );
			CKEDITOR.event.implementOn( this );
			domWalkerEventBridge.call( this );
		};

		characterWalker.prototype = {
			next : function()
			{
				// Clear any previous action events and related flags.
				this._.actionEvents = [];
				this._.matchBoundary = false;

				// If there are more characters in the text node, get it and raise an event.
				if ( this.textNode.type == CKEDITOR.NODE_TEXT && this.offset < this.textNode.getLength() - 1 )
				{
					this.offset++;
					return fireCharacterEvent.call( this );
				}

				// If we are at the end of the text node, use dom walker to get the next text node.
				var data = null;
				while ( !data || ( data.node && data.node.type != CKEDITOR.NODE_TEXT ) )
				{
					data = this._.walker.forward( guardDomWalkerNonEmptyTextNode );

					// Block boundary? BR? Document boundary?
					if ( !data.node || data.node.type != CKEDITOR.NODE_TEXT )
						this._.matchBoundary = true;
				}
				this.textNode = data.node;
				this.offset = 0;
				return fireCharacterEvent.call( this );
			},

			back : function()
			{
				// Clear any previous action events and related flags.
				this._.actionEvents = [];
				this._.matchBoundary = false;

				// More characters -> decrement offset and return.
				if ( this.textNode.type == CKEDITOR.NODE_TEXT && this.offset > 0 )
				{
					this.offset--;
					return fireCharacterEvent.call( this );
				}

				// Start of text node -> use dom walker to get the previous text node.
				var data = null;
				while ( !data || ( data.node && data.node.type != CKEDITOR.NODE_TEXT ) )
				{
					data = this._.walker.reverse( guardDomWalkerNonEmptyTextNode );

					// Block boundary? BR? Document boundary?
					if ( !data.node || data.node.type != CKEDITOR.NODE_TEXT )
						this._.matchBoundary = true;
				}
				this.textNode = data.node;
				this.offset = data.node.length - 1;
				return fireCharacterEvent.call( this );
			},

			getChar : function()
			{
				return this.textNode ? this.textNode.getText().charAt( this.offset ) : null;
			}
		};

		var characterRange = function( initCursor, maxLength )
		{
			this._ = {
				cursors : initCursor.push ? initCursor : [ initCursor ],
				maxLength : maxLength,
		   		highlightRange : null
			};
		};

		characterRange.prototype = {
			toDomRange : function()
			{
				var cursors = this._.cursors;
				if ( cursors.length < 1 )
					return null;

				var first = cursors[0],
					last = cursors[ cursors.length - 1 ],
					range = new CKEDITOR.dom.range( editor.document );

				range.setStart( first.textNode, first.offset );
				range.setEnd( last.textNode, last.offset + 1 );
				return range;
			},

			highlight : function()
			{
				// TODO: wait till Fred has implemented style removal for this.
				editor.getSelection().selectRanges( [ this.toDomRange() ] );
			},

			removeHighlight : function()
			{
				// TODO: wait till Fred has implemented style removal for this.
			},

			getHighlightDomRange : function()
			{
				// TODO: wait till Fred has implemented style removal for this.
				return this.toDomRange();
			},

			moveNext : function()
			{
				var next = new characterWalker( this._.cursors[ this._.cursors.length - 1 ] ),
					retval = next.next(),
					cursors = this._.cursors;

				// Clear the cursors queue if we've crossed a match boundary.
				if ( retval.hitMatchBoundary )
					this._.cursors = cursors = [];

				cursors.push( next );
				if ( cursors.length > this._.maxLength )
					cursors.shift();

				return retval;
			},
			
			moveBack : function()
			{
				var prev = new characterWalker( this._.cursors[0] ),
					retval = prev.back(),
					cursors = this._.cursors;

				if ( retval.hitMatchBoundary )
					this._.cursors = cursors = [];

				cursors.unshift( prev );
				if ( cursors.length > this._.maxLength )
					cursors.pop();

				return retval;
			},

			getEndCharacter : function()
			{
				var cursors = this._.cursors;
				if ( cursors.length < 1 )
					return null;

				return cursors[ cursors.length - 1 ].getChar();
			},

			getNextRange : function( maxLength )
			{
				var cursors = this._.cursors;
				if ( cursors.length < 1 )
					return null;

				var next = new characterWalker( cursors[ cursors.length - 1 ] );
				next.next();
				return new characterRange( next, maxLength );
			},

			getCursors : function()
			{
				return this._.cursors;
			}
		};

		var KMP_NOMATCH = 0,
			KMP_ADVANCED = 1,
			KMP_MATCHED = 2;
		var kmpMatcher = function( pattern, ignoreCase )
		{
			var overlap = [ -1 ];
			for ( var i = 0 ; i < pattern.length ; i++ )
			{
				overlap.push( overlap[i] + 1 );
				while ( overlap[ i + 1 ] > 0 && pattern.charAt( i ) != pattern.charAt( overlap[ i + 1 ] - 1 ) )
					overlap[ i + 1 ] = overlap[ overlap[ i + 1 ] - 1 ] + 1;
			}

			this._ = {
				overlap : overlap,
				state : 0,
				ignoreCase : !!ignoreCase,
				pattern : ignoreCase ? pattern.toLowerCase() : pattern
			};
		};
		kmpMatcher.prototype = {
			feedCharacter : function( c )
			{
				if ( this._.ignoreCase )
					c = c.toLowerCase();

				while ( true )
				{
					if ( c == this._.pattern.charAt( this._.state ) )
					{
						this._.state++;
						if ( this._.state == this._.pattern.length )
						{
							this._.state = 0;
							return KMP_MATCHED;
						}
						return KMP_ADVANCED;
					}
					else if ( this._.state == 0 )
						return KMP_NOMATCH;
					else
						this._.state = this._.overlap[ this._.state ];
				}

				return null;
			},

			reset : function()
			{
				this._.state = 0;
			}
		};

		var wordSeparatorRegex = /[.,"'?!;: \u0085\u00a0\u1680\u280e\u2028\u2029\u202f\u205f\u3000]/,
			isWordSeparator = function( c )
			{
				if ( !c )
					return true;
				var code = c.charCodeAt( 0 );
				return ( code >= 9 && code <= 0xd ) || ( code >= 0x2000 && code <= 0x200a ) || wordSeparatorRegex.test( c );
			};

		var finder = {
			range : null,
			find : function( pattern, matchCase, matchWord )
			{
				if ( !this.range )
					this.range = new characterRange( new characterWalker( editor.document.getBody(), 0 ), pattern.length );
				else
				{
					this.range.removeHighlight();
					this.range = this.range.getNextRange( pattern.length );
				}

				var matcher = new kmpMatcher( pattern, !matchCase ),
					matchState = KMP_NOMATCH,
					character = '%';

				while ( character != null )
				{
					while ( ( character = this.range.getEndCharacter() ) )
					{
						matchState = matcher.feedCharacter( character );
						if ( matchState == KMP_MATCHED )
							break;
						if ( this.range.moveNext() )
							matcher.reset();
					}

					if ( matchState == KMP_MATCHED )
					{
						if ( matchWord )
						{
							var cursors = this.range.getCursors(),
								head = new characterCursor( cursors[ cursors.length - 1 ] ),
								tail = new characterCursor( cursors[0] );
							if ( !( head.next().character || isWordSeparator( head.getChar() ) ) )
								continue;
							if ( !( tail.back().character || isWordSeparator( tail.getChar() ) ) )
								continue;
						}

						this.range.highlight();
						return true;
					}
				}

				this.range = null;
				return false;
			}
		};

		return {
			title : editor.lang.findAndReplace.title,
			resizable : CKEDITOR.DIALOG_RESIZE_NONE,
			minWidth : 400,
			minHeight : 245,
			buttons : [ CKEDITOR.dialog.cancelButton ],		//Cancel button only.
			contents : [
				{
					id : 'find',
					label : editor.lang.findAndReplace.find,
					title : editor.lang.findAndReplace.find,
					accessKey : '',
					elements : [
						{
							type : 'hbox',
							widths : [ '230px', '90px' ],
							children :
							[
								{
									type : 'text',
									id : 'txtFindFind',
									label : editor.lang.findAndReplace.findWhat,
									isChanged : false,
									labelLayout : 'horizontal',
									accessKey : 'F',
								},
								{
									type : 'button',
									align : 'left',
									style : 'width:100%',
									label : editor.lang.findAndReplace.find,
									onClick : function()
									{
										var dialog = this.getDialog();
										if ( !finder.find( dialog.getValueOf( 'find', 'txtFindFind' ),
													dialog.getValueOf( 'find', 'txtFindCaseChk' ),
													dialog.getValueOf( 'find', 'txtFindWordChk' ) ) )
											alert( editor.lang.findAndReplace.notFoundMsg );
									}
								},
							]
						},
						{
							type : 'checkbox',
							id : 'txtFindCaseChk',
							isChanged : false,
							style : 'margin-top:28px',
							label : editor.lang.findAndReplace.matchCase,
						},
						{
							type : 'checkbox',
							id : 'txtFindWordChk',
							isChanged : false,
							label : editor.lang.findAndReplace.matchWord,
						},
						{
							type : 'checkbox',
							id : 'txtFindCyclic',
							isChanged : false,
							checked : true,
							label : editor.lang.findAndReplace.matchCyclic,
						},
					]
				},
				{
					id : 'replace',
					label : editor.lang.findAndReplace.replace,
					accessKey : 'M',
					elements : [
						{
							type : 'hbox',
							widths : [ '230px', '90px' ],
							children :
							[
								{
									type : 'text',
									id : 'txtFindReplace',
									label : editor.lang.findAndReplace.findWhat,
									isChanged : false,
									labelLayout : 'horizontal',
									accessKey : 'F',
								},
								{
									type : 'button',
									align : 'left',
									style : 'width:100%',
									label : editor.lang.findAndReplace.replace,
									onClick : function()
									{
										alert( editor.lang.findAndReplace.notFoundMsg );
									}
								},
							]
						},
						{
							type : 'hbox',
							widths : [ '230px', '90px' ],
							children :
							[
								{
									type : 'text',
									id : 'txtReplace',
									label : editor.lang.findAndReplace.replaceWith,
									isChanged : false,
									labelLayout : 'horizontal',
									accessKey : 'R',
								},
								{
									type : 'button',
									align : 'left',
									style : 'width:100%',
									label : editor.lang.findAndReplace.replaceAll,
									isChanged : false,
									onClick : function()
									{
										alert( editor.lang.findAndReplace.notFoundMsg );
									}
								},
							]
						},
						{
							type : 'checkbox',
							id : 'txtReplaceCaseChk',
							isChanged : false,
							label : editor.lang.findAndReplace.matchCase,
						},
						{
							type : 'checkbox',
							id : 'txtReplaceWordChk',
							isChanged : false,
							label : editor.lang.findAndReplace.matchWord,
						},
						{
							type : 'checkbox',
							id : 'txtReplaceCyclic',
							isChanged : false,
							checked : true,
							label : editor.lang.findAndReplace.matchCyclic,
						},
					]
				}
			],
			onShow : function()
			{
				if ( startupPage == 'replace' )
					this.getContentElement( 'replace', 'txtFindReplace' ).focus();
				else
					this.getContentElement( 'find', 'txtFindFind' ).focus();
			},
			onHide : function()
			{
				if ( finder.range )
				{
					finder.range.removeHighlight();
					editor.getSelection().selectRanges( [finder.range.toDomRange()] );
				}
			}
		};
	};

	CKEDITOR.dialog.add( 'find', function( editor ){
			return findDialog( editor, 'find' )
		}
	);

	CKEDITOR.dialog.add( 'replace', function( editor ){
			return findDialog( editor, 'replace' )
		}
	);
})();
