/*
 * CKEditor - The text editor for Internet - http://ckeditor.com
 * Copyright (C) 2003-2008 Frederico Caldeira Knabben
 *
 * == BEGIN LICENSE ==
 *
 * Licensed under the terms of any of the following licenses at your
 * choice:
 *
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 *
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 *
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 *
 * == END LICENSE ==
 */

/**
 * @fileoverview The "fakeobjects" plugin. It handles placeholder DOM nodes
 * 	in WYSIWYG mode.
 */

(function()
{
	var flashExtensionRegex = /\.swf($|#|\?)/i,
		emptyElements = { base:1,col:1,meta:1,link:1,hr:1,br:1,param:1,img:1,area:1,input:1 },
		makeTagOpenerHtml = function( tagName, attributes, selfClosing )
		{
			var attribStr = [], html = [ '<' + tagName ];
			for ( var i in attributes )
				attribStr.push( i + '="' + CKEDITOR.tools.htmlEncode( attributes[i] ) + '"');
			if ( attribStr.length > 0 )
				html.push( ' ', attribStr.join( ' ' ) );
			if ( emptyElements[ tagName ] || selfClosing )
				html.push( ' /' );
			html.push( '>' );
			return html.join( '' );
		},
		cssWidthRegex = /width\s*:\s*([0-9]+)\s*(?:[\w%]+)?\s*;?/i,
		cssHeightRegex = /height\s*:\s*([0-9]+)\s*(?:[\w%]+)?\s*;?/i;

	var copyParser = function()
	{
		this._ = { html : null };
		this.output = [];

		CKEDITOR.htmlParser.call( this );
	};
	copyParser.prototype = {
		onTagOpen : function( tagName, attributes, selfClosing )
		{
			this.output.push( makeTagOpenerHtml( tagName, attributes, selfClosing ) );
		},

		onTagClose : function( tagName )
		{
			if ( !emptyElements[ tagName] )
				this.output.push( '</' + tagName + '>' );
		},

		onText : function( text )
		{
			this.output.push( text );
		},

		onComment : function( comment )
		{
			this.output.push( '<!--' + comment + '-->' );
		},

		parse : function( html )
		{
			this._.html = html;
			return CKEDITOR.htmlParser.prototype.parse.apply( this, arguments );
		}
	};

	/**
	 * Manages element placeholders in WYSIWYG area.
	 * @constructor
	 * @example
	 */
	CKEDITOR.plugins.fakeobjects = function()
	{
		this._ =
		{
			objectTypes : [
				{
					match : function( nodeName, attributes )
					{
						return nodeName == 'embed' &&
							( attributes.type == 'application/x-shockwave-flash' || flashExtensionRegex.test( attributes.src || '' ) );
					},
					cssClass : 'flash',
					priority : 10
				},
				{
					match : function( nodeName, attributes )
					{
						return ( nodeName == 'a' && attributes.name != null && attributes.name != '' );
					},
					cssClass : 'anchor',
					priority : 10
				},
				{
					match : function( nodeName, attributes )
					{
						if ( nodeName == 'div' && attributes.style && attributes.style.pageBreakAfter == 'always' )
							if ( attributes.firstChild )
							{
								var element = new CKEDITOR.dom.element( attributes.firstChild );
								return ( element && element.getName() == 'span');
							}
						return false;
					},
					cssClass : 'pagebreak',
					priority : 10
				},
				{
					match : function( nodeName, attributes )
					{
						return nodeName == 'embed' || nodeName == 'object';
					},
					cssClass : 'object',
					priority : 0x100000000
				}
			]
		};
	};
	CKEDITOR.plugins.fakeobjects.prototype =
	{
		/**
		 * Converts an element into a placeholder &lt;img&gt; element, for adding
		 * into the WYSIWYG area.
		 * @param {CKEDITOR.dom.element} element Input DOM element.
		 * @returns {CKEDITOR.dom.element} The placeholder &lt;img&gt; element.
		 * @example
		 */
		protectElement : function( element )
		{
			var $ = element.$.cloneNode( true ),
				doc = $.ownerDocument,
				temp = doc.createElement( 'div' ),
				html;

			// Get the object's HTML code.
			temp.appendChild( $ );
			html = temp.innerHTML;

			// Get the fake element's CSS class.
			var cssClass = 'unknown';
			var realElementType = element.getName();
			for ( var i = 0 ; i < this._.objectTypes.length ; i++ )
			{
				if ( this._.objectTypes[i].match( element.getName(), element.$ ) )
				{
					cssClass = 'cke_fakeobject' + ' ' + this._.objectTypes[i].cssClass;
					realElementType = this._.objectTypes[i].cssClass;
					break;
				}
			}

			// Make the fake element.
			var fakeRawElement = doc.createElement( 'img' ),
				cssText = $.style.cssText,
				widthMatch = cssText.match( cssWidthRegex ),
				heightMatch = cssText.match( cssHeightRegex );
			fakeRawElement.className = cssClass;
			fakeRawElement.src = CKEDITOR.getUrl( 'images/spacer.gif' ); 
			if ( widthMatch)
				fakeRawElement.style.width = widthMatch[1] + 'px';
			if ( heightMatch )
				fakeRawElement.style.height = heightMatch[1] + 'px';
			fakeRawElement.setAttribute( '_cke_protected_html', encodeURIComponent( html ) );
			fakeRawElement.setAttribute( '_cke_real_element_type', realElementType );
			return new CKEDITOR.dom.element( fakeRawElement );
		},

		/**
		 * Converts a placeholder &lt;img&gt; element back to the real element.
		 * @param {CKEDITOR.dom.element} fakeImgElement The placeholder &lt;img&gt;.
		 * @returns {CKEDITOR.dom.element} The real element.
		 * @example
		 */
		restoreElement : function( fakeImgElement )
		{
			var html = decodeURIComponent( fakeImgElement.getAttribute( '_cke_protected_html' ) ),
				realElement = CKEDITOR.dom.element.createFromHtml( html, editor.document );

			if ( fakeImgElement.$.style.width )
				realElement.setStyle( 'width', fakeImgElement.$.style.width );
			if ( fakeImgElement.$.style.height )
				realElement.setStyle( 'height', fakeImgElement.$.style.height );

			return realElement;
		},

		/**
		 * Converts protectable elements in an HTML string to placeholders.
		 * @param {String} html HTML with protectable elements.
		 * @returns {String} HTML with placeholders.
		 * @example
		 */
		protectHtml : function( html )
		{
			var parser = new CKEDITOR.htmlParser(),
				tagDepth = 0, processedHtml = [],
				protectedHtml = [], inProtection = false,
				objectTypes = this._.objectTypes;

			parser.onTagOpen = function( tagName, attributes, selfClosing )
			{
				if ( inProtection )
				{
					protectedHtml.push( makeTagOpenerHtml( tagName, attributes, selfClosing ) );
					if ( !( emptyElements[ tagName ] || selfClosing ) )
						tagDepth++;
					if ( tagDepth < 1 )
					{
						inProtection = false;
						processedHtml.push( encodeURIComponent( protectedHtml.join( '' ) ), '" />' );
						protectedHtml = [];
					}
					return;
				}

				for ( var i = 0 ; i < objectTypes.length ; i++ )
				{
					if ( objectTypes[i].match( tagName, attributes ) )
					{
						inProtection = true;
						tagDepth = 0;

						// Get the original object's width and height.
						var styles = attributes.style || '',
							widthMatch = styles.match( cssWidthRegex ),
							heightMatch = styles.match( cssHeightRegex );

						// Create the fake <img> tag.
						processedHtml.push( '<img src="',
							CKEDITOR.getUrl( 'images/spacer.gif' ),
							'" ',
							'class="cke_fakeobject ' + objectTypes[i].cssClass + '" ',
							'_cke_real_element_type="' + objectTypes[i].cssClass + '"' );

						if ( widthMatch || heightMatch )
						{
							processedHtml.push( 'style="',
								widthMatch ? 'width:' + widthMatch[1] + 'px;' : '',
								heightMatch ? 'height:' + heightMatch[1] + 'px;' : '',
								'" ' );
						}

						processedHtml.push( '_cke_protected_html="' );
						arguments.callee.call( this, tagName, attributes, selfClosing );
						return;
					}
				}

				processedHtml.push( makeTagOpenerHtml( tagName, attributes ) );
			};

			parser.onText = function( text )
			{
				inProtection ? protectedHtml.push( text ) : processedHtml.push( text );
			};

			parser.onComment = function( comment )
			{
				inProtection ? protectedHtml.push( '<!--' + comment + '-->' ) : processedHtml.push( '<!--' + comment + '-->' );
			};

			parser.onTagClose = function( tagName )
			{
				if ( inProtection )
				{
					if ( !emptyElements[ tagName ] )
						tagDepth--;
					protectedHtml.push( '</' + tagName + '>' );
					if ( tagDepth < 1 )
					{
						inProtection = false;
						processedHtml.push( encodeURIComponent( protectedHtml.join( '' ) ), '" />' );
						protectedHtml = [];
					}
				}
				else
					processedHtml.push( '</' + tagName + '>' );
			};

			parser.parse( html );
			return processedHtml.join( '' );
		},
		/**
		 * Updates HTML into a placeholder
		 * @param {CKEDITOR.dom.element} fakeImgElement The placeholder &lt;img&gt;.
		 * @param {CKEDITOR.dom.element} element Input DOM element.
		 * @returns {String} encoded HTML.
		 * @example
		 */
		updateFakeElement : function( fakeElement, realElement )
		{
			var $ = realElement.$.cloneNode( true ),
				doc = $.ownerDocument,
				temp = doc.createElement( 'div' ),
				html;

			// Get the object's HTML code.
			temp.appendChild( $ );
			html = temp.innerHTML;
			fakeElement.setAttribute( '_cke_protected_html', encodeURIComponent( html ) );
			return html;
		},
		
		/**
		 * Restores placeholders in an HTML string back to their original elements.
		 * @param {String} html HTML with placeholders.
		 * @returns {String} Restored HTML.
		 * @example
		 */
		restoreHtml : function( html )
		{
			var parser = new copyParser(),
				innerParser = new copyParser();
			
			innerParser.onTagOpen = function( tagName, attributes, selfClosing )
			{
				if ( !this.done )
				{
					var styles = attributes.style || '';

					styles = styles.replace( cssWidthRegex, '' ).replace( cssHeightRegex, '' );
					if ( this.width )
						styles += 'width:' + this.width + ';';
					if ( this.height )
						styles += 'height:' + this.height + ';';
					attributes.style = styles;

					this.done = true;
				}

				return copyParser.prototype.onTagOpen.apply( this, arguments );
			};

			parser.onTagOpen = function( tagName, attributes, selfClosing )
			{
				if ( tagName == 'img' && attributes._cke_protected_html !== undefined )
				{
					var styles = attributes.style || '',
						protectedHtml = decodeURIComponent( attributes._cke_protected_html ),
						widthMatch = styles.match( cssWidthRegex ),
						heightMatch = styles.match( cssHeightRegex );

					if ( widthMatch || heightMatch )
					{
						innerParser.width = widthMatch[1] + 'px';
						innerParser.height = heightMatch[1] + 'px';
						innerParser.parse( protectedHtml );
						protectedHtml = innerParser.output.join( '' );
					}

					this.output.push( protectedHtml );
					return;
				}
				return copyParser.prototype.onTagOpen.apply( this, arguments );
			};

			parser.parse( html );
			return parser.output.join( '' );
		},

		/**
		 * Adds an object type to be displayed by placeholders.
		 * @param {Function} matchFunc A predicate function to determine whether
		 * an object needs to be replaced by placeholders or not. The function
		 * should have the following signature:
		 * <blockquote>function( tagName, attributes )</blockquote>
		 * In which tagName is the object's tag name, and attributes is an object
		 * storing all the object's attributes.
		 * @param {String} cssClass The CSS class that should be added to the
		 * placeholder &lt;img&gt; for representing this type of object.
		 * @param {Number} priority (Optional) An integer representing the
		 * priority of this type of object. If an element matches the descriptions
		 * of two object types, the object type of <strong>lower</strong> priority
		 * takes precedance.
		 * @example
		 */
		addObjectType : function( matchFunc, cssClass, priority )
		{
			if ( priority === undefined )
				priority = 10;

			var obj = {
				match : matchFunc,
				cssClass : cssClass,
				priority : priority
			};

			for ( var i = this._.objectTypes.length - 1 ; i >= 0 ; i-- )
			{
				if ( this._.objectTypes[i].priority < priority )
				{
					this._.objectTypes.splice( i + 1, 0, obj );
					return;
				}
			}

			this._.objectTypes.unshift( obj );
		}
	};

	CKEDITOR.plugins.add( 'fakeobjects',
		{
			init : function( editor, pluginPath )
			{
				editor.fakeobjects = new CKEDITOR.plugins.fakeobjects();
			}
		} );
})();
