/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2004-2009 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.handlers;

import java.io.BufferedInputStream;
import java.io.InputStream;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This handler gives you access to properties stored in
 * <code>/net/fckeditor/handlers/default.properties</code> and
 * <code>/fckeditor.properties</code>.<br />
 * This class loads the properties files as follows:
 * <ol>
 * <li>Load <code>default.properties</code></li>
 * <li>Load <code>fckeditor.properties</code> if present.</li>
 * </ol>
 * <em>Attention</em>: Properties specified in <code>fckeditor.properties</code>
 * will override properties loaded from <code>default.properties</code>.
 * (intended behavior)<br />
 * <em>Hint</em>: You may set properties programmatically with
 * {@link #setProperty(String, String)} instead or additionally.
 * 
 * @version $Id: PropertiesLoader.java 2950 2009-01-19 20:40:52Z mosipov $
 */
public class PropertiesLoader {
	private static final Logger logger = LoggerFactory
			.getLogger(PropertiesLoader.class);
	private static final String DEFAULT_FILENAME = "default.properties";
	private static final String LOCAL_PROPERTIES = "/fckeditor.properties";
	private static Properties properties = new Properties();

	static {

		// 1. load library defaults
		InputStream in = PropertiesLoader.class
				.getResourceAsStream(DEFAULT_FILENAME);

		if (in == null) {
			logger.error("{} not found", DEFAULT_FILENAME);
			throw new RuntimeException(DEFAULT_FILENAME + " not found");
		} else {
			if (!(in instanceof BufferedInputStream))
				in = new BufferedInputStream(in);

			try {
				properties.load(in);
				in.close();
				logger.debug("{} loaded", DEFAULT_FILENAME);
			} catch (Exception e) {
				logger.error("Error while processing {}", DEFAULT_FILENAME);
				throw new RuntimeException(
						"Error while processing " + DEFAULT_FILENAME, e);
			}
		}

		// 2. load user defaults if present
		InputStream in2 = PropertiesLoader.class
				.getResourceAsStream(LOCAL_PROPERTIES);

		if (in2 == null) {
			logger.info("{} not found", LOCAL_PROPERTIES);
		} else {

			if (!(in2 instanceof BufferedInputStream))
				in2 = new BufferedInputStream(in2);

			try {
				properties.load(in2);
				in2.close();
				logger.debug("{} loaded", LOCAL_PROPERTIES);
			} catch (Exception e) {
				logger.error("Error while processing {}", LOCAL_PROPERTIES);
				throw new RuntimeException(
						"Error while processing " + LOCAL_PROPERTIES, e);
			}

		}
	}

	/**
	 * Getter for a property.
	 * 
	 * @param key
	 *            The property key.
	 * @return The value for the specified key.
	 * @see Properties#getProperty(String)
	 */
	public static String getProperty(final String key) {
		return properties.getProperty(key);
	}

	/**
	 * Setter for a property. If the property already exists, the value will be
	 * overridden.<br />
	 * <em>Hint</em>: This method is intended as an alternative way to set
	 * properties programmatically instead of using the
	 * <code>fckeditor.properties</code>. It should never used inside
	 * FCKeditor.Java!!!
	 * 
	 * @param key
	 *            The property key.
	 * @param value
	 *            The property value.
	 * @see Properties#setProperty(String, String)
	 */
	public static void setProperty(final String key, final String value) {
		properties.setProperty(key, value);
	}

	public static String getEditorToolbarSet() {
		return properties.getProperty("fckeditor.toolbarSet");
	}

	public static String getEditorWidth() {
		return properties.getProperty("fckeditor.width");
	}

	public static String getEditorHeight() {
		return properties.getProperty("fckeditor.height");
	}

	public static String getEditorBasePath() {
		return properties.getProperty("fckeditor.basePath");
	}

	public static String getFileResourceTypePath() {
		return properties.getProperty("connector.resourceType.file.path");
	}

	public static String getFlashResourceTypePath() {
		return properties.getProperty("connector.resourceType.flash.path");
	}

	public static String getImageResourceTypePath() {
		return properties.getProperty("connector.resourceType.image.path");
	}

	public static String getMediaResourceTypePath() {
		return properties.getProperty("connector.resourceType.media.path");
	}

	public static String getFileResourceTypeAllowedExtensions() {
		return properties
				.getProperty("connector.resourceType.file.extensions.allowed");
	}
	
	public static String getFileResourceTypeDeniedExtensions() {
		return properties
				.getProperty("connector.resourceType.file.extensions.denied");
	}
	
	public static String getFlashResourceTypeAllowedExtensions() {
		return properties
				.getProperty("connector.resourceType.flash.extensions.allowed");
	}
	
	public static String getFlashResourceTypeDeniedExtensions() {
		return properties
				.getProperty("connector.resourceType.flash.extensions.denied");
	}
	
	public static String getImageResourceTypeAllowedExtensions() {
		return properties
				.getProperty("connector.resourceType.image.extensions.allowed");
	}
	
	public static String getImageResourceTypeDeniedExtensions() {
		return properties
				.getProperty("connector.resourceType.image.extensions.denied");
	}
	
	public static String getMediaResourceTypeAllowedExtensions() {
		return properties
				.getProperty("connector.resourceType.media.extensions.allowed");
	}
	
	public static String getMediaResourceTypeDeniedExtensions() {
		return properties
				.getProperty("connector.resourceType.media.extensions.denied");
	}
	
	/**
	 * Getter for <code>connector.userFilesPath</code> property
	 * 
	 * @return UserFilesPath
	 */
	public static String getUserFilesPath() {
		return properties.getProperty("connector.userFilesPath");
	}

	/**
	 * Getter for <code>connector.userFilesAbsolutePath</code> property
	 * 
	 * @return UserFilesAbsolutePath
	 */
	public static String getUserFilesAbsolutePath() {
		return properties.getProperty("connector.userFilesAbsolutePath");
	}

	/**
	 * Getter for <code>connector.forceSingleExtension</code> property
	 * 
	 * @return ForceSingleExtension
	 */
	public static boolean isForceSingleExtension() {
		return Boolean.valueOf(properties
				.getProperty("connector.forceSingleExtension"));
	}

	/**
	 * Getter for <code>connector.secureImageUploads</code> property
	 * 
	 * @return SecureImageUploads
	 */
	public static boolean isSecureImageUploads() {
		return Boolean.valueOf(properties
				.getProperty("connector.secureImageUploads"));
	}
	
	public static String getConnectorImpl() {
		return properties.getProperty("connector.impl");
	}
	
	public static String getUserActionImpl() {
		return properties.getProperty("connector.userActionImpl");
	}
	
	public static String getUserPathBuilderImpl() {
		return properties.getProperty("connector.userPathBuilderImpl");
	}
	
	public static String getLocaleResolverImpl() {
		return properties.getProperty("localization.localeResolverImpl");
	}
}
