/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2004-2009 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.tool;

import java.util.HashMap;
import java.util.Map;

/**
 * Java representation of an XHTML tag.<br />
 * <br>
 * Usage:
 * 
 * <pre>
 * XHtmlTagTool tag = XHtmlTagTool(&quot;a&quot;, &quot;link&quot;);
 * tag.addAttribute(&quot;href&quot;, &quot;http://google.com&quot;);
 * tag.toString();: &lt;a href=&quot;http://google.com&quot;&gt;link&lt;/a&gt;
 * </pre>
 * 
 * <em>Note</em>:
 * <ul>
 * <li>Attributes are not ordered.</li>
 * <li>If you want to avoid self-closing tags without supplying a value, set
 * {@link #SPACE} as the tag's value.</li>
 * </ul>
 * 
 * @version $Id: XHtmlTagTool.java 3358 2009-04-07 09:45:52Z mosipov $
 */
public class XHtmlTagTool {

	/** Name of the tag. */
	private String name;

	/** Container for the attributes. */
	private Map<String, String> attributes = new HashMap<String, String>();

	/** Value of the tag. */
	private String value;

	/** Indicator to uses non self-closing tag. */
	public static final String SPACE = " ";

	/**
	 * Class constructor with tag name.
	 * 
	 * @param name
	 *            tag name
	 * @throws IllegalArgumentException
	 *             if name is empty
	 */
	public XHtmlTagTool(final String name) {
		if (Utils.isEmpty(name))
			throw new IllegalArgumentException(
					"Parameter 'name' shouldn't be empty!");
		this.name = name;
	}

	/**
	 * Class constructor with name and value.
	 * 
	 * @param name
	 *            tag name
	 * @param value
	 *            tag value which is the tag body
	 * @throws IllegalArgumentException
	 *             if name is empty
	 */
	public XHtmlTagTool(final String name, final String value) {
		this(name);
		this.value = value;
	}

	/**
	 * Sets the tag value.
	 * 
	 * @param value
	 *            tag value which is the tag body
	 */
	public void setValue(final String value) {
		this.value = value;
	}

	/**
	 * Adds an attribute to the tag.
	 * 
	 * @param name
	 *            attribute name
	 * @param value
	 *            attribute value
	 * @throws IllegalArgumentException
	 *             if 'name' is empty.
	 */
	public void addAttribute(final String name, final String value) {
		if (Utils.isEmpty(name))
			throw new IllegalArgumentException(
					"Parameter 'name' shouldn't be empty!");
		attributes.put(name, value);
	}

	/**
	 * Converts the tag to HTML.
	 */
	@Override
	public String toString() {
		StringBuffer tag = new StringBuffer();

		// open tag
		tag.append("<").append(name);

		// add attributes
		for (String key : attributes.keySet()) {
			String val = attributes.get(key);
			tag.append(' ').append(key).append('=').append('\"').append(val)
					.append('\"');
		}

		// close the tag
		if (Utils.isNotEmpty(value)) {
			tag.append(">").append(value).append("</").append(name).append('>');
		} else
			tag.append(" />");

		return tag.toString();
	}

	@Override
	public boolean equals(Object obj) {
		if (obj == null)
			return false;
		try {
			XHtmlTagTool tag = (XHtmlTagTool) obj;
			// FIXME potential NPE on value
			return value.equals(tag.value) && name.equals(tag.name)
					&& attributes.equals(tag.attributes);
		} catch (ClassCastException e) {
			return false;
		}
	}

	/**
	 * The hash code is merely a sum of name's, value's, and attributes' hash
	 * code.
	 */
	@Override
	public int hashCode() {

		// FIXME potential NPE on value
		return name.hashCode() + value.hashCode() + attributes.hashCode();
	}
}
