/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2004-2009 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.connector;

import java.io.IOException;
import java.io.PrintWriter;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.fckeditor.requestcycle.Context;
import net.fckeditor.requestcycle.ThreadLocalData;
import net.fckeditor.response.GetResponse;
import net.fckeditor.response.UploadResponse;

/**
 * This is the main Connector servlet of the FCKeditor.<br/>
 * Let's take a close look at the workflow of this servlet and depending classes. <br/>
 * <br/>
 * This is the only servlet, which has to registered in the web.xml. It is not a concrete implementation of a connector which directly
 * interprets the commands of the File Browser (see:
 * http://docs.fckeditor.net/FCKeditor_2.x/Developers_Guide/Server_Side_Integration#The_Commands), but it does a lot of configuration and
 * delegates the requests to the depended classes.<br/>
 * Important depending classes and its functions:
 * <ul>
 * <li>{@link Dispatcher}, it's a kind of 'middleman', which verifies the request parameters and forwards them to the current implementation
 * of {@link Connector}. It directly interprets the commands of the File Browser. Dependent on the verification of the parameters and calls
 * of the methods of the {@link Connector}, {@link GetResponse} or {@link UploadResponse} are initialized, which is used to build the
 * {@link HttpServletResponse}.</li>
 * <li>{@link ThreadLocalData}, it's a container for {@link HttpServletRequest} and {@link Context} and provides static access to then
 * wherever you want.</li>
 * </ul>
 * <br>
 * In detail, for each request the following steps have to be done:
 * <ul>
 * <li>Initialization of the {@link Dispatcher} object.</li>
 * <li>Calling {@link ThreadLocalData#beginRequest(HttpServletRequest)} to initialize the {@link ThreadLocal}s.</li>
 * <li>Forwarding the requests:
 * <ul>
 * <li><code>GET</code> to {@link Dispatcher#doGet(HttpServletRequest, HttpServletResponse)}</li>
 * <li><code>POST</code> to {@link Dispatcher#doPost(HttpServletRequest, HttpServletResponse)}.</li>
 * </ul>
 * </li>
 * <li>Calling {@link ThreadLocalData#endRequest()} to cleanup all {@link ThreadLocal}s independent of if an exception was thrown or not.<br/>
 * <b>Important</b>: This workflow guarantees the correct cleaning up of the {@link ThreadLocalData}, - it's essential because of preventing
 * memory-leaks!</li>
 * </ul>
 * 
 * @see Connector
 * @see Dispatcher
 * 
 * @version $Id: ConnectorServlet.java 3383 2009-04-14 15:01:44Z mosipov $
 */
public class ConnectorServlet extends HttpServlet {
	private static final long serialVersionUID = -5742008970929377161L;
	private final Logger logger = LoggerFactory.getLogger(ConnectorServlet.class);
	private transient Dispatcher dispatcher;

	/**
	 * Initializes the {@link Dispatcher}.
	 */
	@Override
	public void init() throws ServletException {
		try {
			dispatcher = new Dispatcher(getServletContext());
		} catch (Exception e) {
			logger.error("Dispatcher could not be initialized", e);
			throw new ServletException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest , javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected void doGet(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
		request.setCharacterEncoding("UTF-8");
		response.setCharacterEncoding("UTF-8");
		response.setContentType("application/xml; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();
		GetResponse getResponse = null;

		try {
			ThreadLocalData.beginRequest(request);
			getResponse = dispatcher.doGet(request);
		} catch (Exception e) {
			throw new ServletException(e);
		} finally {
			ThreadLocalData.endRequest();
		}

		out.print(getResponse);
		out.flush();
		out.close();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.servlet.http.HttpServlet#doPost(javax.servlet.http.HttpServletRequest , javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected void doPost(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
		request.setCharacterEncoding("UTF-8");
		response.setCharacterEncoding("UTF-8");
		response.setContentType("text/html; charset=UTF-8");
		response.setHeader("Cache-Control", "no-cache");
		PrintWriter out = response.getWriter();
		UploadResponse uploadResponse = null;

		try {
			ThreadLocalData.beginRequest(request);
			uploadResponse = dispatcher.doPost(request);
		} catch (Exception e) {
			throw new ServletException(e);
		} finally {
			ThreadLocalData.endRequest();
		}

		out.print(uploadResponse);
		out.flush();
		out.close();
	}

}
