/*
 * FCKeditor - The text editor for Internet - http://www.fckeditor.net
 * Copyright (C) 2004-2009 Frederico Caldeira Knabben
 * 
 * == BEGIN LICENSE ==
 * 
 * Licensed under the terms of any of the following licenses at your
 * choice:
 * 
 *  - GNU General Public License Version 2 or later (the "GPL")
 *    http://www.gnu.org/licenses/gpl.html
 * 
 *  - GNU Lesser General Public License Version 2.1 or later (the "LGPL")
 *    http://www.gnu.org/licenses/lgpl.html
 * 
 *  - Mozilla Public License Version 1.1 or later (the "MPL")
 *    http://www.mozilla.org/MPL/MPL-1.1.html
 * 
 * == END LICENSE ==
 */
package net.fckeditor.connector.impl;

import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletContext;

import net.fckeditor.connector.Connector;
import net.fckeditor.connector.exception.FolderAlreadyExistsException;
import net.fckeditor.connector.exception.InvalidCurrentFolderException;
import net.fckeditor.connector.exception.InvalidNewFolderNameException;
import net.fckeditor.connector.exception.WriteException;
import net.fckeditor.handlers.RequestCycleHandler;
import net.fckeditor.handlers.ResourceType;
import net.fckeditor.requestcycle.ThreadLocalData;
import net.fckeditor.requestcycle.UserPathBuilder;
import net.fckeditor.tool.UtilsFile;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.commons.io.filefilter.FileFileFilter;

/**
 * This class provides an abstract implementation of the {@link Connector}
 * interface with the default behavior according to the <a href="http://docs.fckeditor.net/FCKeditor_2.x/Developers_Guide/Server_Side_Integration"
 * target="_blank">documentation</a>. It serves files and folders against a
 * specific local directory. You cannot use this class directly, instead you
 * have to subclass it and implement the interface/abstract methods. There are
 * two ready to go subclasses, see above.
 * 
 * @version $Id: AbstractLocalFileSystemConnector.java 3577 2009-05-30 18:29:35Z mosipov $
 */
public abstract class AbstractLocalFileSystemConnector implements Connector {

	protected ServletContext servletContext;

	public String fileUpload(final ResourceType type,
			final String currentFolder, final String fileName,
			final InputStream inputStream)
			throws InvalidCurrentFolderException, WriteException {
		String absolutePath = getRealUserFilesAbsolutePath(RequestCycleHandler
				.getUserFilesAbsolutePath(ThreadLocalData.getRequest()));
		File typeDir = getOrCreateResourceTypeDir(absolutePath, type);
		File currentDir = new File(typeDir, currentFolder);
		if (!currentDir.exists() || !currentDir.isDirectory())
			throw new InvalidCurrentFolderException();

		File newFile = new File(currentDir, fileName);
		File fileToSave = UtilsFile.getUniqueFile(newFile.getAbsoluteFile());

		try {
			IOUtils.copyLarge(inputStream, new FileOutputStream(fileToSave));
		} catch (IOException e) {
			throw new WriteException();
		}
		return fileToSave.getName();
	}

	public void createFolder(final ResourceType type,
			final String currentFolder, final String newFolder)
			throws InvalidCurrentFolderException,
			InvalidNewFolderNameException, FolderAlreadyExistsException {
		String absolutePath = getRealUserFilesAbsolutePath(RequestCycleHandler
				.getUserFilesAbsolutePath(ThreadLocalData.getRequest()));
		File typeDir = getOrCreateResourceTypeDir(absolutePath, type);
		File currentDir = new File(typeDir, currentFolder);
		if (!currentDir.exists() || !currentDir.isDirectory())
			throw new InvalidCurrentFolderException();

		File newDir = new File(currentDir, newFolder);
		if (newDir.exists())
			throw new FolderAlreadyExistsException();
		if (!newDir.mkdir())
			throw new InvalidNewFolderNameException();
	}

	public List<Map<String, Object>> getFiles(ResourceType type,
			String currentFolder) throws InvalidCurrentFolderException {
		String absolutePath = getRealUserFilesAbsolutePath(RequestCycleHandler
				.getUserFilesAbsolutePath(ThreadLocalData.getRequest()));
		File typeDir = getOrCreateResourceTypeDir(absolutePath, type);
		File currentDir = new File(typeDir, currentFolder);
		if (!currentDir.exists() || !currentDir.isDirectory())
			throw new InvalidCurrentFolderException();

		// collect files
		List<Map<String, Object>> files;
		Map<String, Object> fileMap;
		File[] fileList = currentDir
				.listFiles((FileFilter) FileFileFilter.FILE);
		files = new ArrayList<Map<String, Object>>(fileList.length);
		for (File file : fileList) {
			fileMap = new HashMap<String, Object>(2);
			fileMap.put(Connector.KEY_NAME, file.getName());
			fileMap.put(Connector.KEY_SIZE, file.length());
			files.add(fileMap);
		}
		return files;
	}

	public List<String> getFolders(final ResourceType type,
			final String currentFolder) throws InvalidCurrentFolderException {
		String absolutePath = getRealUserFilesAbsolutePath(RequestCycleHandler
				.getUserFilesAbsolutePath(ThreadLocalData.getRequest()));
		File typeDir = getOrCreateResourceTypeDir(absolutePath, type);
		File currentDir = new File(typeDir, currentFolder);
		if (!currentDir.exists() || !currentDir.isDirectory())
			throw new InvalidCurrentFolderException();

		String[] fileList = currentDir.list(DirectoryFileFilter.DIRECTORY);
		return Arrays.asList(fileList);
	}

	/**
	 * This method shall resolve the
	 * {@link UserPathBuilder#getUserFilesAbsolutePath(javax.servlet.http.HttpServletRequest)
	 * UserFilesAbsolutePath} against a specific system. E.g., local filesystem
	 * or context filesystem. It's up to you what it returns but make sure that
	 * it's consistent inside the entire cycle.
	 * 
	 * @param path
	 *            The path to resolve.
	 * @return The resolved path.
	 */
	protected abstract String getRealUserFilesAbsolutePath(String path);

	/**
	 * This method simply checks if the the resource type dir exists and creates
	 * it if necessary.
	 * 
	 * @param baseDir
	 *            BaseDir for the resource type.
	 * @param type
	 *            Resource type to mkdir.
	 * @return The file object of the resource type.
	 */
	private static File getOrCreateResourceTypeDir(final String baseDir,
			final ResourceType type) {
		File dir = new File(baseDir, type.getPath());
		if (!dir.exists())
			dir.mkdirs();
		return dir;
	}
}
