﻿/*
Copyright (c) 2003-2010, CKSource - Frederico Knabben. All rights reserved.
For licensing, see LICENSE.html or http://ckeditor.com/license
*/

/**
 * @fileOverview The "toolbar" plugin. Renders the default toolbar interface in
 * the editor.
 */

(function()
{
	// Trait for toolbar containing unit.
	function container()
	{
		return CKEDITOR.tools.createClass( {
			$ : function()
			{
				this.id = CKEDITOR.tools.getNextId(),
				this.items = [];
			},
			proto :
			{
				/**
				 *  Focus the first/last item of the container.
				 * @param {Boolean} focusEnd
				 */
				focus : function( focusEnd )
				{
					var current = this,
							target;

					while ( 1 )
					{
						target = current.items[ focusEnd? current.items.length -1 : 0 ];
						if ( target && target.focus )
							break;
						// No focusable candidate found, move focus to parent's siblings.
						else if ( current.parent )
							current = current.parent;
						else
						{
							target = current;
							break;
						}
					}

					target.focus( focusEnd );
				},

				add : function( item )
				{
					var index = this.items.push( item ) - 1;

					// Create the next/previous/parent reference.
					if ( index > 0 )
					{
						item.previous = this.items[ index - 1 ];
						item.previous.next = item;
					}
					item.parent = this;
				}
			}
		} );
	}

	var toolbox = container(),
		toolbar = container(),
		group = container();

	var commands =
	{
		toolbarFocus :
		{
			modes : { wysiwyg : 1, source : 1 },

			exec : function( editor )
			{
				if ( editor.toolbox )
				{
					editor.toolbox.focusCommandExecuted = true;

					// Make the first button focus accessible. (#3417)
					if ( CKEDITOR.env.ie )
						setTimeout( function(){ editor.toolbox.focus(); }, 100 );
					else
						editor.toolbox.focus();
				}
			}
		}
	};

	CKEDITOR.plugins.add( 'toolbar',
	{
		init : function( editor )
		{
			var groupCycling = editor.config.toolbarGroupCycling !== false;

			// Cycling the toolbar without considering groups.
			function toolbarNavigate( item, closewise )
			{
				var next;
				while( !next )
				{
					next = item.parent ? item[ closewise ? 'next' : 'previous' ] : item;
					if ( !next )
						item = item.parent;
					else if ( !next.focus )
					{
						item = next;
						next = null;
					}
				}
				next.focus( !closewise );
				return false;
			}

			var buttonKeystroke = function( item, keystroke )
			{
				var next, closewise;
				var rtl = editor.lang.dir == 'rtl';

				switch ( keystroke )
				{
					case 9 :					// TAB
					case CKEDITOR.SHIFT + 9 :	// SHIFT + TAB

						closewise = keystroke == 9;

						// Move to next group if when button group existed.
						if ( groupCycling && item.parent instanceof group )
						{
							item = item.parent;
							while( !next )
							{
								next = item[ closewise ? 'next' : 'previous' ];
								if ( !next )
								{
									if ( item.parent )
										item = item.parent;
									else
										next = item;
								}
								else if ( !next.focus )
								{
									item = next;
									next = null;
								}
							}

							next.focus( !closewise );
							return false;
						}
						else
							return toolbarNavigate( item, closewise )


					case rtl ? 37 : 39 :					// RIGHT-ARROW
					case rtl ? 39 : 37 :					// LEFT-ARROW

						if ( closewise == undefined  )
							closewise = keystroke == ( rtl ? 37 : 39 );

						if ( groupCycling )
						{
							while( !next )
							{
								next = item[ closewise ? 'next' : 'previous' ];
								if ( !next )
								{
									// Cyrcling inside the group when button group existed.
									if ( item.parent instanceof group )
										next = item.parent;
									else if ( item.parent )
										item = item.parent;
									else
										next =  item;
								}
								else if ( !next.focus )
								{
									item = next;
									next = null;
								}
							}

							next.focus( !closewise );
							return false;
						}
						else
							return toolbarNavigate( item, closewise );


					case 27 :					// ESC
						editor.focus();
						return false;

					case 13 :					// ENTER
					case 32 :					// SPACE
						item.execute();
						return false;
				}
				return true;
			};

			editor.on( 'themeSpace', function( event )
				{
					if ( event.data.space == editor.config.toolbarLocation )
					{
						editor.toolbox = new toolbox();

						var labelId = CKEDITOR.tools.getNextId();

						var output = [ '<div class="cke_toolbox" role="group" aria-labelledby="', labelId, '" onmousedown="return false;"' ],
							expanded =  editor.config.toolbarStartupExpanded !== false,
							groupStarted;

						output.push( expanded ? '>' : ' style="display:none">' );

						// Sends the ARIA label.
						output.push( '<span id="', labelId, '" class="cke_voice_label">', editor.lang.toolbar, '</span>' );

						var toolboxObj = editor.toolbox,
							toolbarDef =
									( editor.config.toolbar instanceof Array ) ?
										editor.config.toolbar
									:
										editor.config[ 'toolbar_' + editor.config.toolbar ];

						for ( var r = 0 ; r < toolbarDef.length ; r++ )
						{
							var row = toolbarDef[ r ];

							// It's better to check if the row object is really
							// available because it's a common mistake to leave
							// an extra comma in the toolbar definition
							// settings, which leads on the editor not loading
							// at all in IE. (#3983)
							if ( !row )
								continue;

							var toolbarObj = new toolbar();

							if ( groupStarted )
							{
								output.push( '</div>' );
								groupStarted = 0;
							}

							if ( row === '/' )
							{
								output.push( '<div class="cke_break"></div>' );
								continue;
							}

							output.push( '<span id="', toolbar.id, '" class="cke_toolbar" role="presentation"><span class="cke_toolbar_start"></span>' );

							toolboxObj.add( toolbarObj );

							for ( var i = 0 ; i < row.length ; i++ )
							{
								var item = row[ i ],
									isGroup = ( typeof item == 'object' ),
									buttons = isGroup ? item.buttons : [ item ];

								// Group start.
								if ( isGroup )
								{
									var groupObj = new group();
									output.push( '<span id="' + groupObj.id + '" class="cke_toolgroup"  role="toolbar" aria-labelledby="'+ groupObj.id +  '_label">' +
										'<span id="' + groupObj.id + '_label" class="cke_voice_label">' + ( editor.lang.toolbarGroup[ item.name ] || '' ) + ' </span>' );
									toolbarObj.add( groupObj );
								}

								// Create all items defined for this toolbar.
								for ( var j = 0 ; j < buttons.length ; j++ )
								{
									var button,
										itemName = buttons[ j ];

									// Ignore separater in group mode.
									if ( itemName == '-' && !isGroup )
										button = CKEDITOR.ui.separator;
									else
										button = editor.ui.create( itemName );

									if ( button )
									{
										if ( button.canGroup )
										{
											if ( !( groupStarted || isGroup ) )
											{
												output.push( '<span class="cke_toolgroup" role="presentation">' );
												groupStarted = 1;
											}
										}
										else if ( groupStarted )
										{
											output.push( '</span>' );
											groupStarted = 0;
										}

										var container = isGroup ? groupObj : toolbarObj;
										var buttonObj = button.render( editor, output );
										container.add( buttonObj );

										buttonObj.onkey = buttonKeystroke;
									}
								}

								// Group close.
								if ( isGroup )
								{
									output.push( '</span>' );
									groupStarted = 0;
								}
							}

							// Group must close on toolbar row end.
							if ( groupStarted )
							{
								output.push( '</span>' );
								groupStarted = 0;
							}
							output.push( '<span class="cke_toolbar_end"></span></span>' );
						}

						output.push( '</div>' );

						if ( editor.config.toolbarCanCollapse )
						{
							var collapserFn = CKEDITOR.tools.addFunction(
								function()
								{
									editor.execCommand( 'toolbarCollapse' );
								});

							editor.on( 'destroy', function () {
									CKEDITOR.tools.removeFunction( collapserFn );
								});

							var collapserId = CKEDITOR.tools.getNextId();

							editor.addCommand( 'toolbarCollapse',
								{
									exec : function( editor )
									{
										var collapser = CKEDITOR.document.getById( collapserId ),
											toolbox = collapser.getPrevious(),
											contents = editor.getThemeSpace( 'contents' ),
											toolboxContainer = toolbox.getParent(),
											contentHeight = parseInt( contents.$.style.height, 10 ),
											previousHeight = toolboxContainer.$.offsetHeight,
											collapsed = !toolbox.isVisible();

										if ( !collapsed )
										{
											toolbox.hide();
											collapser.addClass( 'cke_toolbox_collapser_min' );
											collapser.setAttribute( 'title', editor.lang.toolbarExpand );
										}
										else
										{
											toolbox.show();
											collapser.removeClass( 'cke_toolbox_collapser_min' );
											collapser.setAttribute( 'title', editor.lang.toolbarCollapse );
										}

										// Update collapser symbol.
										collapser.getFirst().setText( collapsed ?
											'\u25B2' :		// BLACK UP-POINTING TRIANGLE
											'\u25C0' );		// BLACK LEFT-POINTING TRIANGLE

										var dy = toolboxContainer.$.offsetHeight - previousHeight;
										contents.setStyle( 'height', ( contentHeight - dy ) + 'px' );

										editor.fire( 'resize' );
									},

									modes : { wysiwyg : 1, source : 1 }
								} );

							output.push( '<a title="' + ( expanded ? editor.lang.toolbarCollapse : editor.lang.toolbarExpand )
													  + '" id="' + collapserId + '" tabIndex="-1" class="cke_toolbox_collapser' );

							if ( !expanded )
								output.push( ' cke_toolbox_collapser_min' );

							output.push( '" onclick="CKEDITOR.tools.callFunction(' + collapserFn + ')">',
										'<span>&#9650;</span>',		// BLACK UP-POINTING TRIANGLE
										'</a>' );
						}

						event.data.html += output.join( '' );
					}
				});

			editor.addCommand( 'toolbarFocus', commands.toolbarFocus );
		}
	});
})();

/**
 * The UI element that renders a toolbar separator.
 * @type Object
 * @example
 */
CKEDITOR.ui.separator =
{
	render : function( editor, output )
	{
		output.push( '<span class="cke_separator" role="separator"></span>' );
		return {};
	}
};

/**
 * The "theme space" to which rendering the toolbar. For the default theme,
 * the recommended options are "top" and "bottom".
 * @type String
 * @default 'top'
 * @see CKEDITOR.config.theme
 * @example
 * config.toolbarLocation = 'bottom';
 */
CKEDITOR.config.toolbarLocation = 'top';

/**
 * The toolbar definition. It is an array of toolbars (strips),
 * each one being also an array, containing a list of UI items.
 * Note that this setting is composed by "toolbar_" added by the toolbar name,
 * which in this case is called "Basic". This second part of the setting name
 * can be anything. You must use this name in the
 * {@link CKEDITOR.config.toolbar} setting, so you instruct the editor which
 * toolbar_(name) setting to you.
 * @type Array
 * @example
 * // Defines a toolbar with only one strip containing the "Source" button, a
 * // separator and the "Bold" and "Italic" buttons.
 * <b>config.toolbar_Basic =
 * [
 *     [ 'Source', '-', 'Bold', 'Italic' ]
 * ]</b>;
 * config.toolbar = 'Basic';
 */
CKEDITOR.config.toolbar_Basic =
[
	[
		{
			name: 'basicstyle',
			buttons : [ 'Bold','Italic' ]
		},
		{
			name: 'list',
			buttons : [ 'NumberedList','BulletedList' ]
		},
		{
			name: 'link',
			buttons : [ 'Link','Unlink' ]
		},
		'About'
	]
];

/**
 * A toolbar definition with all default items and groups.
 * that used as the default value of {@link CKEDITOR.config.toolbar}.
 * @type Array
 * @example
 * CKEDITOR.config.toolbar = 'Full';
 */
CKEDITOR.config.toolbar_Full =
[
	[
		{
			name: 'mode',
			buttons : [ 'Source' ]
		},
		{
			name: 'page',
			buttons : [ 'Save','NewPage','Preview','Print' ]
		},
		{
			name: 'template',
			buttons : [ 'Templates' ]
		},
		{
			name: 'clipboard',
			buttons : [ 'Cut','Copy','Paste','PasteText','PasteFromWord' ]
		},
		{
			name: 'spellcheck',
			buttons : [ 'SpellChecker', 'Scayt' ]
		},
		{
			name: 'undo',
			buttons : [ 'Undo','Redo' ]
		},
		{
			name: 'find',
			buttons : [ 'Find','Replace' ]
		},
		{
			name: 'clear',
			buttons : [ 'SelectAll','RemoveFormat' ]
		},
		{
			name: 'form',
			buttons : [ 'Form', 'Checkbox', 'Radio', 'TextField', 'Textarea', 'Select', 'Button', 'ImageButton', 'HiddenField' ]
		}
	],
	'/',
	[
		{
			name: 'basicstyle',
			buttons : [ 'Bold','Italic','Underline','Strike','Subscript','Superscript' ]
		},
		{
			name: 'list',
			buttons : [ 'NumberedList','BulletedList' ]
		},
		{
			name: 'indent',
			buttons : [ 'Outdent','Indent' ]
		},
		{
			name: 'block',
			buttons : [ 'Blockquote','CreateDiv' ]
		},
		{
			name: 'justify',
			buttons : [ 'JustifyLeft','JustifyCenter','JustifyRight','JustifyBlock' ]
		},
		{
			name: 'link',
			buttons : [ 'Link','Unlink','Anchor' ]
		},
		{
			name: 'objects',
			buttons : [ 'Image','Flash','Table','HorizontalRule','Smiley','SpecialChar','PageBreak' ]
		}
	],
	'/',
	[
		'Styles','Format','Font','FontSize',
		{
			name: 'color',
			buttons : [ 'TextColor','BGColor' ]
		},
		{
			name: 'tool',
			buttons : [ 'Maximize', 'ShowBlocks' ]
		},
		'About'
	]
];

/**
 * The toolbox (alias toolbar) definition. It is a toolbar name or an array of
 * toolbars (strips), each one being also an array, containing a list of UI items.
 * @type Array|String
 * @default 'Full'
 * @example
 * // Defines a toolbar with only one strip containing the "Source" button, a
 * // separator and the "Bold" and "Italic" buttons.
 *	 config.toolbar =
 * [
 *			 [
 *					 // Single button as a group, without specifying group name.
 *					 'Source', '-',
 *					 // Two buttons in a group, with "basic style" as group name.
 *					 {
 *						 name : 'basic_style',
 *						 buttons: [ 'Bold', 'Italic' ]
 *					 }
 *			 ]
 *	 ];
 * @example
 * // Load toolbar_Name where Name = Basic.
 * config.toolbar = 'Basic';
 */
CKEDITOR.config.toolbar = 'Full';

/**
 * Whether the toolbar can be collapsed by the user. If disabled, the collapser
 * button will not be displayed.
 * @type Boolean
 * @default true
 * @example
 * config.toolbarCanCollapse = false;
 */
CKEDITOR.config.toolbarCanCollapse = true;

/**
 * Whether the toolbar must start expanded when the editor is loaded.
 * @name CKEDITOR.config.toolbarStartupExpanded
 * @type Boolean
 * @default true
 * @example
 * config.toolbarStartupExpanded = false;
 */

/**
 * Whether enable the following keyboard navigation for toolbar group:
 * <dl>
 * <dt>Arrow Left/Arrow Right:</dt><dd>Cycle inside toolbar group</dd>
 * <dt>Tab/Shift-Tab:</dt><dd>Navigate between toolbar groups, wrap at start or end.</dd>
 * </dl>
 * If set to "false" then both "Arrow" and "Tab" keystrokes cycling the entire toolbar without considering the group.
 * @name CKEDITOR.config.toolbarGroupCycling
 * @type Boolean
 * @default true
 * @example
 * config.toolbarGroupCycling = false;
 */

