/*
Copyright (c) 2003-2011, CKSource - Frederico Knabben. All rights reserved.
For licensing, see http://ckeditor.com/license
*/
package com.ckeditor;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * CKEditor tag for Java configuration class.
 */
public class CKEditorConfig implements Cloneable {

	private Map<String, Object> config;
	
	/**
	 * Default constructor.
	 */
	public CKEditorConfig() {
		config = new HashMap<String, Object>();		
	}
	
	/**
	 * Adds number param to config.
	 * <b>Usage:</b>
	 * <pre>config.addConfigValue("width", 100);</pre>
	 * <pre>config.addConfigValue("dialog_backgroundCoverOpacity", 0.7);</pre>
	 * @param key config param key
	 * @param value config param value.
	 */
	public void addConfigValue(final String key, final Number value) {
		config.put(key, value);
	}
	
	/**
	 * Adds String param to config.
	 * <b>Usage:</b>
	 * <pre>config.addConfigValue("baseHref", "http://www.example.com/path/");</pre>
	 * <pre>config.addConfigValue("toolbar", [[ 'Source', '-', 'Bold', 'Italic' ]]);</pre>
	 * @param key config param key
	 * @param value config param value.
	 */
	public void addConfigValue(final String key, final String value) {
		config.put(key, value);
	}
	
	/**
	 * Adds Map param to config.
	 * <b>Usage:</b>
	 * <pre>Map<String, Object> map = new HashMap<String, Object>();</pre>
	 * <pre>map.put("element", "span");</pre>
	 * <pre>map.put("styles", "{'background-color' : '#(color)'}");</pre>
	 * <pre>config.addConfigValue("colorButton_backStyle", map);</pre>
	 * @param key config param key
	 * @param value config param value.
	 */
	public void addConfigValue(final String key, 
								final Map<String, ? extends Object> value) {
		config.put(key, value);
	}
	
	/**
	 * Adds List param to config.
	 * <b>Usage:</b>
	 * <pre> 
		List<List<String>> list = new ArrayList<List<String>>();
		List<String> subList = new ArrayList<String>();
		subList.add("Source");
		subList.add("-");
		subList.add("Bold");
		subList.add("Italic");
		list.add(subList);
		config.addConfigValue("toolbar", list);
		</pre>
	 * @param key config param key
	 * @param value config param value.
	 */
	public void addConfigValue(final String key, final List<? extends Object> value) {
		config.put(key, value);
	}
	
	/**
	 * Adds boolean param to config.
	 * <b>Usage:</b>
	 * <pre>config.addConfigValue("autoUpdateElement", true);</pre> 
	 * @param key config param key
	 * @param value config param value.
	 */
	public void addConfigValue(final String key, final Boolean value) {
		config.put(key, value);
	}
	
	
	/**
	 * Gets config value by key.
	 * @param key config param key
	 * @return config param value.
	 */
	Object getConfigValue(final String key) {
		return config.get(key);
	}

	/**
	 * @return all config values.
	 */
	Map<String, Object> getConfigValues() {
		return config;
	}
	
	/**
	 * Removes config value by key.
	 * <b>Usage:</b>
	 * <pre>config.removeConfigValue("toolbar");</pre>
	 * @param key config param key.
	 */
	public void removeConfigValue(final String key) {
		config.remove(key);
	}
	
	/**
	 * Configure settings. Merge config and event handlers.
	 * @return setting config.
	 * @param eventHandler events
	 */
	CKEditorConfig configSettings(final EventHandler eventHandler) {
		try {
			CKEditorConfig cfg = (CKEditorConfig) this.clone();
			if (eventHandler != null) {
				for (String eventName : eventHandler.events.keySet()) {
					Set<String> set = eventHandler.events.get(eventName);
					if (set.isEmpty()) {
						continue;
					} else if (set.size() == 1) {
						Map<String, String> hm = new HashMap<String, String>();
						for (String code : set) {
							hm.put(eventName, "@@" + code);	
						}					
						cfg.addConfigValue("on", hm);
					} else {
						Map<String, String> hm = new HashMap<String, String>();
						StringBuilder sb = new StringBuilder("@@function (ev){");
						for (String code : set) {
							sb.append("(");
							sb.append(code);
							sb.append(")(ev);");
						}
						sb.append("}");
						hm.put(eventName, sb.toString());
						cfg.addConfigValue("on", hm);
					}
				}
			}			
			return cfg;
		} catch (CloneNotSupportedException e) {
			return null;
		}
		
		
	}
	
	/**
	 * Checks if config is empty.
	 * @return true if is
	 */
	public boolean isEmpty() {
		return config.isEmpty();
	}
	
	/**
	 * Override.
	 */
	protected Object clone() throws CloneNotSupportedException {
		CKEditorConfig cfg = (CKEditorConfig) super.clone();
		cfg.config = new HashMap<String, Object>(this.config);
		return cfg; 
	}
	
		
}
