/*
Pure Javascript implementation of the GNU Gettext.
Copyright (C) 2008 Joshua I. Miller <unrtst@cpan.org>, all rights reserved

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU Library General Public License as published
by the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.
*/

( function()
{
	function parse_po_dequote( str )
	{
		var match;
		if ( match = str.match( /^"(.*)"/ ) )
		{
			str = match[1];
		}
		// unescale all embedded quotes (fixes bug #17504)
		str = str.replace( /\\"/g, "\"" );
		return str;
	}

	var GLUE = '\004';

	function parse_po(data)
	{
		var rv = {};
		var buffer = {};
		var lastbuffer = "";
		var errors = [];
		var lines = data.split( "\n" );
		for ( var i = 0; i < lines.length; i++ )
		{
			// chomp
			lines[i] = lines[i].replace( /(\n|\r)+$/, '' );

			var match;

			// Empty line / End of an entry.
			if ( /^$/.test( lines[i] ) )
			{
				if ( typeof(buffer['msgid']) != 'undefined' )
				{
					var msg_ctxt_id = (typeof(buffer['msgctxt']) != 'undefined' &&
							buffer['msgctxt'].length) ?
									  buffer['msgctxt'] + GLUE + buffer['msgid'] :
									  buffer['msgid'];
					var msgid_plural = (typeof(buffer['msgid_plural']) != 'undefined' &&
							buffer['msgid_plural'].length) ?
									   buffer['msgid_plural'] :
									   null;

					// find msgstr_* translations and push them on
					var trans = [];
					for ( var str in buffer )
					{
						var match;
						if ( match = str.match( /^msgstr_(\d+)/ ) )
							trans[parseInt( match[1] )] = buffer[str];
					}
					trans.unshift( msgid_plural );

					// only add it if we've got a translation
					// NOTE: this doesn't conform to msgfmt specs
					if ( trans.length > 1 ) rv[msg_ctxt_id] = trans;

					buffer = {};
					lastbuffer = "";
				}

				// comments
			} else if ( /^#/.test( lines[i] ) )
			{
				continue;

				// msgctxt
			} else if ( match = lines[i].match( /^msgctxt\s+(.*)/ ) )
			{
				lastbuffer = 'msgctxt';
				buffer[lastbuffer] = parse_po_dequote( match[1] );

				// msgid
			} else if ( match = lines[i].match( /^msgid\s+(.*)/ ) )
			{
				lastbuffer = 'msgid';
				buffer[lastbuffer] = parse_po_dequote( match[1] );

				// msgid_plural
			} else if ( match = lines[i].match( /^msgid_plural\s+(.*)/ ) )
			{
				lastbuffer = 'msgid_plural';
				buffer[lastbuffer] = parse_po_dequote( match[1] );

				// msgstr
			} else if ( match = lines[i].match( /^msgstr\s+(.*)/ ) )
			{
				lastbuffer = 'msgstr_0';
				buffer[lastbuffer] = parse_po_dequote( match[1] );

				// msgstr[0] (treak like msgstr)
			} else if ( match = lines[i].match( /^msgstr\[0\]\s+(.*)/ ) )
			{
				lastbuffer = 'msgstr_0';
				buffer[lastbuffer] = parse_po_dequote( match[1] );

				// msgstr[n]
			} else if ( match = lines[i].match( /^msgstr\[(\d+)\]\s+(.*)/ ) )
			{
				lastbuffer = 'msgstr_' + match[1];
				buffer[lastbuffer] = parse_po_dequote( match[2] );

				// continued string
			} else if ( /^"/.test( lines[i] ) )
			{
				buffer[lastbuffer] += parse_po_dequote( lines[i] );
				// something strange
			}
			else
			{
				errors.push( "Strange line [" + i + "] : " + lines[i] );
			}
		}


		// handle the final entry
		if ( typeof(buffer['msgid']) != 'undefined' )
		{
			var msg_ctxt_id = (typeof(buffer['msgctxt']) != 'undefined' &&
					buffer['msgctxt'].length) ?
							  buffer['msgctxt'] + GLUE + buffer['msgid'] :
							  buffer['msgid'];
			var msgid_plural = (typeof(buffer['msgid_plural']) != 'undefined' &&
					buffer['msgid_plural'].length) ?
							   buffer['msgid_plural'] :
							   null;

			// find msgstr_* translations and push them on
			var trans = [];
			for ( var str in buffer )
			{
				var match;
				if ( match = str.match( /^msgstr_(\d+)/ ) )
					trans[parseInt( match[1] )] = buffer[str];
			}
			trans.unshift( msgid_plural );

			// only add it if we've got a translation
			// NOTE: this doesn't conform to msgfmt specs
			if ( trans.length > 1 ) rv[msg_ctxt_id] = trans;

			buffer = {};
			lastbuffer = "";
		}


		// parse out the header
		if ( rv[""] && rv[""][1] )
		{
			var cur = {};
			var hlines = rv[""][1].split( /\\n/ );
			for ( var i = 0; i < hlines.length; i++ )
			{
				if ( ! hlines.length ) continue;

				var pos = hlines[i].indexOf( ':', 0 );
				if ( pos != -1 )
				{
					var key = hlines[i].substring( 0, pos );
					var val = hlines[i].substring( pos + 1 );
					var keylow = key.toLowerCase();

					if ( cur[keylow] && cur[keylow].length )
					{
						errors.push( "SKIPPING DUPLICATE HEADER LINE: " + hlines[i] );
					} else if ( /#-#-#-#-#/.test( keylow ) )
					{
						errors.push( "SKIPPING ERROR MARKER IN HEADER: " + hlines[i] );
					}
					else
					{
						// remove begining spaces if any
						val = val.replace( /^\s+/, '' );
						cur[keylow] = val;
					}

				}
				else
				{
					errors.push( "PROBLEM LINE IN HEADER: " + hlines[i] );
					cur[hlines[i]] = '';
				}
			}

			// replace header string with assoc array
			rv[""] = cur;
		}
		else
		{
			rv[""] = {};
		}

		// TODO: XXX: if there are errors parsing, what do we want to do?
		// GNU Gettext silently ignores errors. So will we.
		// alert( "Errors parsing po file:\n" + errors.join("\n") );

		return rv;
	}

	this.parsePO = parse_po;
} )();
