﻿///<reference path="../../ckeditor.js"/>

require(['libs/ckeditor/ckeditor'], function (cke) {

    cke.plugins.add('wsb', {
        requires: ['richcombo'],
        icons: 'addwsblink,removewsblink',
        init: initPlugin
    });

    function initPlugin(editor) {
        editor.addCommand('addWsbLink', { exec: showAddLinkDialog });
        editor.addCommand('removeWsbLink', {
            exec: function (editor) {
                var style = new cke.style({
                    element: 'a',
                    type: cke.STYLE_INLINE,
                    alwaysRemoveElement: true
                });
                editor.removeStyle(style);
            },
            refresh: function (editor, path) {
                var element = path.lastElement && path.lastElement.getAscendant('a', true);
                this.setState(element && element.getName() === 'a' && element.getAttribute('href') && element.getChildCount()
                    ? cke.TRISTATE_OFF
                    : cke.TRISTATE_DISABLED);
            },
            contextSensitive: true,
            startDisabled: true
        });

        editor.setKeystroke(cke.CTRL + 76, 'addWsbLink');

        editor.ui.addButton('AddWsbLink', {
            label: 'Add Link',
            command: 'addWsbLink',
            toolbar: 'links,10'
        });

        editor.ui.addButton('RemoveWsbLink', {
            label: 'Remove Link',
            command: 'removeWsbLink',
            toolbar: 'links,20'
        });

        editor.addMenuItem('link', { label: 'Add Link', command: 'addWsbLink', group: 'link', order: 1 });
        editor.addMenuItem('unlink', { label: 'Remove Link', command: 'removeWsbLink', group: 'link', order: 5 });

        editor.ui.addRichCombo('InsertPlaceholder', {
            label: 'Placeholder',
            title: 'Insert Placeholder',
            voiceLabel: 'Insert a Placeholder',
            toolbar: "placeholder,10",

            panel: {
                css: [ cke.skin.getPath("editor"), editor.config.contentsCss ],
                multiSelect: false,
                attributes: { "aria-label": 'Insert Placeholder' }
            },

            init: function () {
                var combo = this;
                require(['modules/placeholder/placeholder'], function (ph) {
                    for (var i = 0; i < ph.placeholders.length; i++) {
                        var placeholder = ph.placeholders[i];
                        combo.add(placeholder.token, placeholder.name, placeholder.name);
                    }
                });
            },

            onClick: function (value) {
                editor.focus();
                editor.fire('saveSnapshot');
                editor.insertText(value);
                editor.fire('saveSnapshot');
            }
        });
    }

    function showAddLinkDialog(editor) {
        require([
            'jq!starfield/jquery.mod',
            'libs/knockout/knockout',
            'modules/ui/dialog',
            'modules/wsb/wsb'
        ], function ($, ko, dlg, wsb) {

            var selection = editor.getSelection();
            selection.lock();

            var viewModel = {
                href: ko.observable(''),
                pageId: ko.observable(null),
                isNewWindow: ko.observable(false),
                pages: $.map(wsb.actionbar.getPages(), function (page) {
                    return {
                        id: page.Id,
                        name: page.Name,
                        type: 'page'
                    };
                })
            };

            var selectedLink = getSelectedLink(editor);
            if (selectedLink) {
                selection.selectElement(selectedLink);
                viewModel.href(selectedLink.getAttribute('href'));
                viewModel.isNewWindow(selectedLink.getAttribute('target') === "_blank");
                viewModel.pageId(selectedLink.data('wsb-page-id'));
            }

            var dialog = $('<div data-bind="template: \'insert-link-dialog\'"/>').wsbDialog({
                title: 'Insert Link',
                
                buttons: [
                    {
                        id: 'ok',
                        text: selectedLink ? 'Update' : 'Insert',
                        onClick: function () {
                            var pageId = viewModel.pageId();
                            if (selectedLink) {
                                updateHyperlink(selection, selectedLink, viewModel);
                            } else {
                                insertHyperlink(selection, viewModel);
                            }
                            dialog.wsbDialog('close');
                            return false;
                        }
                    },
                    { id: 'cancel', text: 'Cancel', cancel: true }
                ],
                onOpen: function (dialog) {
                    // Manipulate the dialog to allow the URL picker to extend beyond the dimensions of the dialog
                    dialog
                        .add(dialog.find('.sf-dialog-inner'))
                        .css({ overflow: 'visible', height: 120, width: 400 });
                },
                onClose: function () {
                    selection.unlock();
                    editor.element.focus();
                }
            });

            ko.applyBindings(viewModel, dialog[0]);

            function getSelectedLink(editor) {
                var selection = editor.getSelection();
                var selectedElement = selection.getSelectedElement();
                if (selectedElement && selectedElement.is('a')) {
                    return selectedElement;
                }

                var range = selection.getRanges(true)[0];
                if (range) {
                    range.shrink(cke.SHRINK_TEXT);
                    return editor.elementPath(range.getCommonAncestor()).contains('a', true);
                }

                return null;
            }

            function insertHyperlink(selection, viewModel) {
                var url = viewModel.href();
                var openInNewWindow = viewModel.isNewWindow();

                var range = selection.getRanges()[0];
                if (range.collapsed) {

                    var text;
                    var pageUrlMatch = /page:(\d+)/.exec(url);
                    if (pageUrlMatch) {
                        var pageId = parseInt(pageUrlMatch[1], 10);
                        for (var i = 0; i < viewModel.pages.length; i++) {
                            if (viewModel.pages[i].id === pageId) {
                                text = viewModel.pages[i].name;
                                break;
                            }
                        }
                    } else {
                        text = url;
                    }

                    var textNode = new cke.dom.text(text, selection.document);
                    range.insertNode(textNode);
                    range.selectNodeContents(textNode);
                }

                var attributes = { href: url };
                if (openInNewWindow)
                    attributes.target = '_blank';

                var ckStyle = new cke.style({
                    element: 'a',
                    attributes: attributes,
                    type: cke.STYLE_INLINE
                });

                ckStyle.applyToRange(range);
                range.select();
            }

            function updateHyperlink(selection, hyperlink, viewModel) {
                hyperlink.setAttribute('href', viewModel.href());
                if (viewModel.isNewWindow()) {
                    hyperlink.setAttribute('target', '_blank');
                } else {
                    hyperlink.removeAttribute('target');
                }
                hyperlink.removeAttribute('data-cke-saved-href');
                selection.selectElement(hyperlink);
            }
        });
    }
});
